<?php

namespace Drupal\pdf_services\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Pager\PagerManagerInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Url;
use Drupal\Core\Database\Query\TableSortExtender;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Controller for the queue status page.
 */
class QueueStatusController extends ControllerBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The pager manager service.
   *
   * @var \Drupal\Core\Pager\PagerManagerInterface
   */
  protected $pagerManager;

  /**
   * The queue factory service.
   *
   * @var \Drupal\Core\Queue\QueueFactory
   */
  protected $queueFactory;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Constructs a QueueStatusController object.
   */
  public function __construct(
    Connection $database,
    DateFormatterInterface $date_formatter,
    PagerManagerInterface $pager_manager,
    QueueFactory $queue_factory,
    RequestStack $request_stack,
    EntityTypeManagerInterface $entity_type_manager,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->database = $database;
    $this->dateFormatter = $date_formatter;
    $this->pagerManager = $pager_manager;
    $this->queueFactory = $queue_factory;
    $this->requestStack = $request_stack;
    $this->entityTypeManager = $entity_type_manager;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('date.formatter'),
      $container->get('pager.manager'),
      $container->get('queue'),
      $container->get('request_stack'),
      $container->get('entity_type.manager'),
      $container->get('logger.factory')
    );
  }

  /**
   * Displays the queue status.
   *
   * @return array
   *   Render array for the queue status page.
   */
  public function status() {
    $build = [];

    // Get the status counts for the summary.
    $summary = $this->getStatusSummary();

    // Add summary section.
    $build['summary'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['pdf-services-status-summary']],
      'content' => [
        '#theme' => 'item_list',
        '#items' => [
          $this->t('Pending: @count', ['@count' => $summary['pending']]),
          $this->t('Processing: @count', ['@count' => $summary['processing']]),
          $this->t('Completed: @count', ['@count' => $summary['completed']]),
          $this->t('Failed: @count', ['@count' => $summary['failed']]),
          $this->t('Total: @count', ['@count' => $summary['total']]),
        ],
        '#wrapper_attributes' => ['class' => ['pdf-services-status-counts']],
      ],
    ];

    // Status filters - use our dedicated filter form
    $build['filters'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['container-inline', 'pdf-services-filters']],
    ];

    // Get current filter value from URL
    $current_request = $this->requestStack->getCurrentRequest();
    $status_filter = $current_request->query->get('status', 'all');

    // Add our filter form
    $build['filters']['form'] = $this->formBuilder()->getForm('Drupal\pdf_services\Form\QueueFilterForm');

    // TableSort extender setup.
    $header = [
      'filename' => [
        'data' => $this->t('Filename'),
        'field' => 'filename',
      ],
      'created' => [
        'data' => $this->t('Created'),
        'field' => 'created',
      ],
      'state' => [
        'data' => $this->t('Status'),
        'field' => 'state',
      ],
      'operation' => [
        'data' => $this->t('Operation'),
        'field' => 'operation',
      ],
      'retries' => [
        'data' => $this->t('Retries'),
        'field' => 'retries',
      ],
      'settings' => [
        'data' => $this->t('Settings'),
      ],
      'changed' => [
        'data' => $this->t('Last Updated'),
        'field' => 'changed',
        'sort' => 'desc',
        'default_sort' => TRUE,
      ],
      'entity' => ['data' => $this->t('Used In')],
    ];

    // Get processing statuses with paging.
      $query = $this->database->select('pdf_processing_status', 'p')
      ->fields('p');

      // Add field_settings explicitly if the field exists
      $schema = $this->database->schema();
      if ($schema->fieldExists('pdf_processing_status', 'field_settings')) {
      $query->addField('p', 'field_settings');
      }

      // Extend with table sorting
      $query = $query->extend(TableSortExtender::class);

      if ($status_filter !== 'all') {
      $query->condition('state', $status_filter);
      }

      // Apply sorting
      $query->orderByHeader($header);

    $total = $query->countQuery()->execute()->fetchField();
    $page = $this->pagerManager->createPager($total, 20);
    $offset = $page->getCurrentPage() * 20;

    $result = $query
      ->range($offset, 20)
      ->execute();

    $rows = [];
    foreach ($result as $record) {
      // Find the entity that references this file.
      $file = $this->entityTypeManager->getStorage('file')->load($record->fid);
      $referencing_entity = NULL;

      if ($file) {
        $usage = $this->findEntityReferencing($file->id());
        if ($usage) {
          try {
            $entity = $this->entityTypeManager
              ->getStorage($usage['type'])
              ->load($usage['id']);

            if ($entity && $entity->access('view')) {
              $referencing_entity = Link::createFromRoute(
                $entity->label(),
                'entity.' . $usage['type'] . '.canonical',
                [$usage['type'] => $usage['id']]
              );
            }
          }
          catch (\Exception $e) {
            $this->loggerFactory->get('pdf_services')->error('Error loading referencing entity: @error', [
              '@error' => $e->getMessage(),
            ]);
          }
        }
      }

      // Format the field settings for display
      $settings_display = $this->formatFieldSettings($record->field_settings);

      $rows[] = [
        'data' => [
          'filename' => $record->filename,
          'created' => $this->dateFormatter->format($record->created, 'short'),
          'state' => $record->state,
          'operation' => $record->operation,
          'retries' => $record->retries,
          'settings' => [
            'data' => $settings_display,
          ],
          'changed' => $this->t('@time ago', [
            '@time' => $this->dateFormatter->formatTimeDiffSince($record->changed),
          ]),
          'entity' => $referencing_entity ? $referencing_entity : $this->t('N/A'),
        ],
        'class' => ['status-' . $record->state],
      ];
    }

    $build['table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No processing records found.'),
      '#attributes' => ['class' => ['pdf-processing-status-table']],
    ];

    $build['pager'] = [
      '#type' => 'pager',
    ];

    // Add some CSS to style the summary.
    $build['#attached']['library'][] = 'pdf_services/admin';

    $build['summary'] = [
      '#theme' => 'pdf_services_status_summary',
      '#counts' => $summary,
    ];

    return $build;
  }

  /**
   * Formats field settings for display in the report.
   *
   * @param mixed $field_settings
   *   The field_settings value from the database.
   *
   * @return string
   *   Formatted string representation of settings.
   */
  protected function formatFieldSettings($field_settings) {
    // Handle case when field_settings is empty
    if (empty($field_settings)) {
      return $this->t('Default settings');
    }

    // Try to unserialize if it's a string
    if (is_string($field_settings)) {
      $settings = @unserialize($field_settings);
      if ($settings === FALSE) {
        // If unserialize fails, try decoding as JSON
        $settings = json_decode($field_settings, TRUE);
        if (json_last_error() !== JSON_ERROR_NONE) {
          return $this->t('Invalid settings format');
        }
      }
    } else {
      $settings = $field_settings;
    }

    // Build a list of settings
    $items = [];

    // Map compression level to readable text
    $compression_levels = [
      'NONE' => $this->t('None'),
      'LOW' => $this->t('Low'),
      'MEDIUM' => $this->t('Medium'),
      'HIGH' => $this->t('High'),
    ];

    // Format check_properties
    if (isset($settings['check_properties'])) {
      $items[] = $this->t('Properties: @value', [
        '@value' => !empty($settings['check_properties']) ? $this->t('Yes') : $this->t('No'),
      ]);
    }

    // Format check_accessibility
    if (isset($settings['check_accessibility'])) {
      $items[] = $this->t('Accessibility: @value', [
        '@value' => !empty($settings['check_accessibility']) ? $this->t('Yes') : $this->t('No'),
      ]);
    }

    // Format page_size_threshold
    if (isset($settings['page_size_threshold'])) {
      $threshold_kb = number_format($settings['page_size_threshold'] / 1000, 0);
      $items[] = $this->t('Threshold: @value KB', ['@value' => $threshold_kb]);
    }

    // Format compression_level
    if (isset($settings['compression_level'])) {
      $level = $settings['compression_level'];
      $items[] = $this->t('Compression: @value', [
        '@value' => isset($compression_levels[$level]) ? $compression_levels[$level] : $level,
      ]);
    }

    // Return a formatted list or a message if no recognized settings are found
    if (!empty($items)) {
      return implode(' | ', $items);
    }
    else {
      return $this->t('Custom settings');
    }
  }

  /**
   * Finds the first entity that references a file.
   *
   * @param int $fid
   *   The file ID.
   *
   * @return array
   *   Array containing entity type and ID, or NULL if not found.
   */
  protected function findEntityReferencing($fid) {
    $query = $this->database->select('file_usage', 'fu');
    $query->fields('fu', ['type', 'id'])
      ->condition('fid', $fid)
      ->range(0, 1);

    return $query->execute()->fetchAssoc();
  }

/**
 * Gets counts of processing status by state.
 *
 * @return array
 *   Array with counts for each status.
 */
protected function getStatusSummary() {
  $summary = [
    'pending' => 0,
    'processing' => 0,
    'completed' => 0,
    'failed' => 0,
    'total' => 0,
    'monthly_count' => 0,
    'monthly_limit' => 500,
  ];

  try {
    // Build a query to count records grouped by state
    $query = $this->database->select('pdf_processing_status', 'p');
    $query->addExpression('COUNT(p.id)', 'count');
    $query->addField('p', 'state');
    $query->groupBy('p.state');

    $result = $query->execute();

    // Process each record and map to our summary array
    foreach ($result as $record) {
      $state = $record->state;
      $count = (int) $record->count;

      // Add to the appropriate summary category
      if (isset($summary[$state])) {
        $summary[$state] = $count;
      }

      // Add to total regardless of state
      $summary['total'] += $count;
    }

    // Get the current month's request count using changed column and excluding pending status
    // This ensures all completed, processing and failed items are counted accurately
    $first_day_of_month = mktime(0, 0, 0, date('n'), 1, date('Y'));

    $monthly_query = $this->database->select('pdf_processing_status', 'p')
      ->condition('changed', $first_day_of_month, '>=')
      ->condition('state', 'pending', '!=');
    $monthly_query->addExpression('COUNT(id)', 'count');

    $summary['monthly_count'] = (int) $monthly_query->execute()->fetchField();

  }
  catch (\Exception $e) {
    $this->getLogger('pdf_services')->error('Error generating status summary: @error', [
      '@error' => $e->getMessage(),
    ]);
    // If there's an error, we'll return the default summary with zeros
  }

  return $summary;
}
}
