<?php

namespace Drupal\pdf_services\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityChangedTrait;

/**
 * Defines the PDF processing status entity.
 *
 * @ContentEntityType(
 *   id = "pdf_processing_status",
 *   label = @Translation("PDF Processing Status"),
 *   base_table = "pdf_processing_status",
 *   entity_keys = {
 *     "id" = "id",
 *     "uuid" = "uuid",
 *     "fid" = "fid"
 *   },
 *   handlers = {
 *     "views_data" = "Drupal\views\EntityViewsData",
 *     "access" = "Drupal\Core\Entity\EntityAccessControlHandler",
 *   }
 * )
 */
class PdfProcessingStatus extends ContentEntityBase {

  use EntityChangedTrait;

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['fid'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('File ID'))
      ->setDescription(t('The ID of the PDF file being processed.'))
      ->setRequired(TRUE)
      ->setSetting('unsigned', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'number_integer',
        'weight' => -5,
      ]);

    $fields['filename'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Filename'))
      ->setDescription(t('The name of the PDF file.'))
      ->setRequired(TRUE)
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ]);

    $fields['checksum'] = BaseFieldDefinition::create('string')
      ->setLabel(t('File Checksum'))
      ->setDescription(t('MD5 checksum of the file to detect changes.'))
      ->setRequired(TRUE)
      ->setSettings([
        'max_length' => 32,
        'text_processing' => 0,
      ]);

    $fields['state'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Processing State'))
      ->setDescription(t('Current processing state (pending, processing, completed, failed).'))
      ->setRequired(TRUE)
      ->setDefaultValue('pending')
      ->setSettings([
        'max_length' => 32,
        'text_processing' => 0,
      ]);

    $fields['callback_url'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Callback URL'))
      ->setDescription(t('Adobe API callback URL for checking operation status.'))
      ->setSettings([
        'max_length' => 2048,
        'text_processing' => 0,
      ]);

    $fields['job_id'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Job ID'))
      ->setDescription(t('Adobe API job identifier.'))
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ]);

    $fields['asset_id'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Asset ID'))
      ->setDescription(t('Adobe API asset identifier.'))
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ]);

    $fields['retries'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Retry Count'))
      ->setDescription(t('Number of times this job has been retried.'))
      ->setDefaultValue(0);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time the processing status was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time the processing status was last updated.'));

    $fields['skip_processing'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Skip Processing'))
      ->setDescription(t('Flag to prevent processing loops.'))
      ->setDefaultValue(FALSE);

    $fields['retry_after'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(t('Retry After'))
      ->setDescription(t('Timestamp after which to retry the operation.'))
      ->setDefaultValue(0);

    $fields['operation'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Operation'))
      ->setDescription(t('The name of the PDF processing operation.'))
      ->setRequired(TRUE)
      ->setDefaultValue('')
      ->setSettings([
        'max_length' => 64,
        'text_processing' => 0,
      ]);

    $fields['field_settings'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Field Settings'))
      ->setDescription(t('Field-specific PDF processing settings.'))
      ->setRevisionable(FALSE);

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage) {
    parent::preSave($storage);
  }

}
