<?php

namespace Drupal\pdf_services\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure PDF Services settings.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['pdf_services.settings', 'pdf_services.adobe_embed_settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'pdf_services_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('pdf_services.settings');
    $adobe_config = $this->config('pdf_services.adobe_embed_settings');

    // Master toggle for PDF processing
    $form['processing_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable PDF processing'),
      '#default_value' => $config->get('processing_enabled') ?? TRUE,
      '#description' => $this->t('When disabled, PDFs will still be queued but will not be processed until this setting is enabled again.'),
      '#weight' => -100,
    ];

    // API Credentials Section
    $form['api'] = [
      '#type' => 'details',
      '#title' => $this->t('Adobe PDF Services API Credentials'),
      '#open' => TRUE,
    ];

    $form['api']['client_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Client ID'),
      '#default_value' => $config->get('client_id'),
      '#description' => $this->t('Enter your Adobe PDF Services API Client ID. Get one at <a href="@url" target="_blank">Adobe Developer Console</a>.', [
        '@url' => 'https://acrobatservices.adobe.com/dc-integration-creation-app-cdn/main.html?api=pdf-services-api',
      ]),
    ];

    $form['api']['client_secret'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Client Secret'),
      '#default_value' => $config->get('client_secret'),
      '#description' => $this->t('Enter your Adobe PDF Services API Client Secret. Get one at <a href="@url" target="_blank">Adobe Developer Console</a>.', [
        '@url' => 'https://acrobatservices.adobe.com/dc-integration-creation-app-cdn/main.html?api=pdf-services-api',
      ]),
    ];

    $form['api']['help_text'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('These credenticals are for server to server PDF operations accessibility checking, compression and analysis.') . '</p>',
    ];

    // Adobe PDF Embed API section
    $form['embed_api'] = [
      '#type' => 'details',
      '#title' => $this->t('Adobe PDF Embed API'),
      '#open' => TRUE,
    ];

    $form['embed_api']['adobe_embed_client_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Adobe PDF Embed API Client ID'),
      '#default_value' => $adobe_config->get('client_id'),
      '#description' => $this->t('Enter your Adobe PDF Embed API client ID. Get one at <a href="https://acrobatservices.adobe.com/dc-integration-creation-app-cdn/main.html?api=pdf-embed-api" target="_blank">Adobe Developer Console</a>.'),
      '#required' => FALSE,
    ];

    $form['embed_api']['help_text'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('This client ID for the Adobe PDF Embed field formatter to view pdfs.') . '</p>',
    ];

    // Queue Settings Section
    $form['queue'] = [
      '#type' => 'details',
      '#title' => $this->t('Queue Settings'),
      '#open' => TRUE,
    ];

    $form['queue']['batch_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Batch Size'),
      '#default_value' => $config->get('batch_size') ?? 5,
      '#min' => 1,
      '#max' => 25,
      '#description' => $this->t('Number of PDFs to process in each cron run (max 25 due to API rate limits).'),
    ];

    $form['queue']['retry_limit'] = [
      '#type' => 'number',
      '#title' => $this->t('Retry Limit'),
      '#default_value' => $config->get('retry_limit') ?? 3,
      '#min' => 0,
      '#max' => 10,
      '#description' => $this->t('Number of times to retry failed operations.'),
    ];

    $form['notifications'] = [
      '#type' => 'details',
      '#title' => $this->t('Email Notifications'),
      '#open' => TRUE,
    ];

    $form['notifications']['email_notifications_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable email notifications for accessibility issues'),
      '#default_value' => $config->get('email_notifications.enabled') ?? FALSE,
      '#description' => $this->t('Send email notifications when accessibility issues are detected.'),
    ];

    $form['notifications']['email_notifications_include_report'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include full accessibility report in email'),
      '#default_value' => $config->get('email_notifications.include_report') ?? TRUE,
      '#description' => $this->t('Include the complete accessibility report in the notification email.'),
      '#states' => [
        'visible' => [
          ':input[name="email_notifications_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['notifications']['notification_fallback_uid'] = [
      '#type' => 'entity_autocomplete',
      '#title' => $this->t('Fallback notification recipient'),
      '#description' => $this->t('User to notify if no active editors are found. Leave empty for no fallback.'),
      '#target_type' => 'user',
      '#default_value' => $config->get('notification_fallback_uid') ?
        $this->entityTypeManager->getStorage('user')->load($config->get('notification_fallback_uid')) :
        NULL,
      '#selection_settings' => [
        'include_anonymous' => FALSE,
      ],
      '#states' => [
        'visible' => [
          ':input[name="email_notifications_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // Validate batch size against API rate limits
    $batch_size = $form_state->getValue('batch_size');
    if ($batch_size > 25) {
      $form_state->setError($form['queue']['batch_size'], $this->t('Batch size cannot exceed 25 due to API rate limits.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('pdf_services.settings');
    $adobe_config = $this->config('pdf_services.adobe_embed_settings');

    // Save master toggle setting
    $config->set('processing_enabled', $form_state->getValue('processing_enabled'));

    // Save API settings
    $config->set('client_id', $form_state->getValue('client_id'))
           ->set('client_secret', $form_state->getValue('client_secret'));

    // Save Adobe PDF Embed API settings
    $adobe_config->set('client_id', $form_state->getValue('adobe_embed_client_id'))
                 ->save();

    // Save queue settings
    $config->set('batch_size', $form_state->getValue('batch_size'))
           ->set('retry_limit', $form_state->getValue('retry_limit'));

    // Save email notification settings
    $config->set('email_notifications.enabled', $form_state->getValue('email_notifications_enabled'))
           ->set('email_notifications.include_report', $form_state->getValue('email_notifications_include_report'))
            ->set('notification_fallback_uid', $form_state->getValue('notification_fallback_uid') ? $form_state->getValue('notification_fallback_uid')->id() : NULL);

    $config->save();

    parent::submitForm($form, $form_state);
  }
}
