<?php

namespace Drupal\pdf_services\Plugin\Action;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Action\ConfigurableActionBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\file\FileInterface;
use Drupal\pdf_services\Service\PdfServicesManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Queue PDF files for optimization.
 *
 * @Action(
 *   id = "pdf_services_queue_optimization",
 *   label = @Translation("Queue PDF Optimization"),
 *   type = "file",
 *   confirm = TRUE,
 * )
 */
class QueuePdfOptimization extends ConfigurableActionBase implements ContainerFactoryPluginInterface {

  /**
   * The PDF services manager.
   *
   * @var \Drupal\pdf_services\Service\PdfServicesManager
   */
  protected $pdfServicesManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    PdfServicesManager $pdf_services_manager,
    EntityTypeManagerInterface $entity_type_manager
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->pdfServicesManager = $pdf_services_manager;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('pdf_services.manager'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'check_properties' => TRUE,
      'compression_level' => 'MEDIUM',
      'page_size_threshold' => 0,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['check_properties'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Analyze PDF Properties'),
      '#description' => $this->t('Required for optimization to determine file characteristics.'),
      '#default_value' => $this->configuration['check_properties'],
      '#disabled' => TRUE,
      '#value' => TRUE,
    ];

    $form['compression_level'] = [
      '#type' => 'select',
      '#title' => $this->t('PDF Compression Level'),
      '#description' => $this->t('Select the compression level for PDF optimization.'),
      '#options' => [
        'LOW' => $this->t('Low - Faster processing, larger file size'),
        'MEDIUM' => $this->t('Medium - Balanced compression (recommended)'),
        'HIGH' => $this->t('High - Maximum compression, slower processing'),
      ],
      '#default_value' => $this->configuration['compression_level'],
      '#required' => TRUE,
    ];

    $form['page_size_threshold'] = [
      '#type' => 'number',
      '#title' => $this->t('Average Page Size Threshold'),
      '#description' => $this->t('Optimize PDFs where the average page size exceeds this threshold. Set to 0 to optimize all files regardless of size.'),
      '#default_value' => $this->configuration['page_size_threshold'] / 1000,
      '#min' => 0,
      '#step' => 1,
      '#required' => TRUE,
      '#field_suffix' => $this->t('KB'),
    ];

    $form['info'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('This action will queue the selected PDF files for optimization using the specified compression level. PDF properties will be checked first to determine file characteristics.') . '</p>',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->configuration['check_properties'] = TRUE;
    $this->configuration['compression_level'] = $form_state->getValue('compression_level');
    // Convert KB to bytes for storage.
    $this->configuration['page_size_threshold'] = $form_state->getValue('page_size_threshold') * 1000;
  }

  /**
   * {@inheritdoc}
   */
  public function execute($entity = NULL) {
    // Only process file entities.
    if (!$entity instanceof FileInterface) {
      return;
    }

    // Only process PDF files.
    $mime_type = $entity->getMimeType();
    if ($mime_type !== 'application/pdf') {
      \Drupal::messenger()->addWarning($this->t('Skipped non-PDF file: @filename', [
        '@filename' => $entity->getFilename(),
      ]));
      return;
    }

    // Check if processing is enabled.
    $config = \Drupal::config('pdf_services.settings');
    $processing_enabled = $config->get('processing_enabled') ?? TRUE;

    if (!$processing_enabled) {
      \Drupal::messenger()->addError($this->t('PDF processing is currently disabled.'));
      return;
    }

    // Prepare field settings.
    $field_settings = [
      'check_properties' => TRUE,
      'check_accessibility' => FALSE,
      'compression_level' => $this->configuration['compression_level'],
      'page_size_threshold' => $this->configuration['page_size_threshold'],
    ];

    // Create processing status.
    $status_id = $this->pdfServicesManager->createProcessingStatus($entity, $field_settings);

    if ($status_id) {
      \Drupal::messenger()->addStatus($this->t('Queued @filename for PDF optimization with @level compression.', [
        '@filename' => $entity->getFilename(),
        '@level' => strtolower($this->configuration['compression_level']),
      ]));
    }
    else {
      \Drupal::messenger()->addWarning($this->t('File @filename could not be queued for processing. It may already be queued, or the file may not be accessible.', [
        '@filename' => $entity->getFilename(),
      ]));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, AccountInterface $account = NULL, $return_as_object = FALSE) {
    /** @var \Drupal\file\FileInterface $object */
    // Check if user has permission for bulk operations.
    $access = $account->hasPermission('administer pdf services')
      || $account->hasPermission('bulk process pdf files');

    return $return_as_object ? AccessResult::allowedIf($access) : $access;
  }

}
