<?php

namespace Drupal\pdf_services\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Url;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'adobe_pdf_embed' formatter.
 *
 * @FieldFormatter(
 *   id = "adobe_pdf_embed",
 *   label = @Translation("Adobe PDF Embed"),
 *   field_types = {
 *     "file",
 *     "string",
 *     "uri"
 *   }
 * )
 */
class AdobePdfEmbedFormatter extends FormatterBase implements ContainerFactoryPluginInterface {

  /**
   * The file URL generator service.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected $fileUrlGenerator;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs an AdobePdfEmbedFormatter object.
   *
   * @param string $plugin_id
   *   The plugin_id for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The formatter label display setting.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\Core\File\FileUrlGeneratorInterface $file_url_generator
   *   The file URL generator service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, $label, $view_mode, array $third_party_settings, FileUrlGeneratorInterface $file_url_generator, ConfigFactoryInterface $config_factory) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->fileUrlGenerator = $file_url_generator;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('file_url_generator'),
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'embed_mode' => 'FULL_WINDOW',
      'width' => '100%',
      'height' => '600px',
      'show_download' => TRUE,
      'show_print' => TRUE,
      'show_annotations' => TRUE,
      'show_bookmark' => TRUE,
      'show_full_screen' => TRUE,
      'default_view' => 'FIT_WIDTH',
      'show_direct_link' => TRUE,
      'allow_override' => FALSE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form['embed_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Embed Mode'),
      '#options' => [
        'FULL_WINDOW' => $this->t('Full Window'),
        'SIZED_CONTAINER' => $this->t('Sized Container'),
        'IN_LINE' => $this->t('In-Line'),
        'LIGHT_BOX' => $this->t('Lightbox'),
      ],
      '#default_value' => $this->getSetting('embed_mode'),
      '#description' => $this->t('Select how the PDF should be embedded in the page.For detailed information about embed modes, refer to the <a href="@url" target="_blank">Adobe PDF Embed API documentation</a>.',
      ['@url' => 'https://developer.adobe.com/document-services/docs/overview/pdf-embed-api/howtos/']),
    ];

    // Add option for direct link to PDF for SEO
    $form['show_direct_link'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show direct link to PDF'),
      '#description' => $this->t('Displays a direct link to the PDF file above the viewer. This helps search engines index your PDF content.'),
      '#default_value' => $this->getSetting('show_direct_link'),
      '#weight' => 2,
    ];

    $form['width'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Width'),
      '#default_value' => $this->getSetting('width'),
      '#description' => $this->t('Width of the PDF viewer (e.g., 100%, 600px).'),
      '#states' => [
        'visible' => [
          ':input[name$="[settings][embed_mode]"]' => [
            ['value' => 'SIZED_CONTAINER'],
            ['value' => 'IN_LINE'],
          ],
        ],
      ],
    ];

    $form['height'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Height'),
      '#default_value' => $this->getSetting('height'),
      '#description' => $this->t('Height of the PDF viewer (e.g., 600px).'),
      '#states' => [
        'visible' => [
          ':input[name$="[settings][embed_mode]"]' => [
            ['value' => 'SIZED_CONTAINER'],
          ],
        ],
      ],
    ];

    $form['default_view'] = [
      '#type' => 'select',
      '#title' => $this->t('Default View'),
      '#options' => [
        'FIT_WIDTH' => $this->t('Fit Width'),
        'FIT_PAGE' => $this->t('Fit Page'),
        'TWO_COLUMN' => $this->t('Two Column'),
        'TWO_COLUMN_FIT_PAGE' => $this->t('Two Column Fit Page'),
      ],
      '#default_value' => $this->getSetting('default_view'),
      '#description' => $this->t('Default view mode for the PDF.'),
      '#states' => [
        'visible' => [
          ':input[name$="[settings][embed_mode]"]' => [
            ['value' => 'FULL_WINDOW'],
            ['value' => 'LIGHT_BOX'],
          ],
        ],
      ],
    ];

    $form['show_download'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show download button'),
      '#default_value' => $this->getSetting('show_download'),
    ];

    $form['show_print'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show print button'),
      '#default_value' => $this->getSetting('show_print'),
    ];

    $form['show_annotations'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show annotation tools'),
      '#default_value' => $this->getSetting('show_annotations'),
      '#states' => [
        'visible' => [
          ':input[name$="[settings][embed_mode]"]' => ['value' => 'FULL_WINDOW'],
        ],
      ],
    ];

    $form['show_bookmark'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show bookmarks'),
      '#default_value' => $this->getSetting('show_bookmark'),
      '#states' => [
        'visible' => [
          ':input[name$="[settings][embed_mode]"]' => [
            ['value' => 'FULL_WINDOW'],
            ['value' => 'LIGHT_BOX'],
          ],
        ],
      ],
    ];

    $form['show_full_screen'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show full screen button'),
      '#default_value' => $this->getSetting('show_full_screen'),
      '#description' => $this->t('When enabled, displays a full screen button in the viewer toolbar.'),
      '#states' => [
        'visible' => [
          ':input[name$="[settings][embed_mode]"]' => [
            ['value' => 'FULL_WINDOW'],
            ['value' => 'SIZED_CONTAINER'],
          ],
        ],
      ],
    ];

    // Add allow override option
    $form['allow_override'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow per-entity overrides'),
      '#default_value' => $this->getSetting('allow_override'),
      '#description' => $this->t('When enabled, content editors will be able to override these PDF viewer settings on a per-entity basis.'),
      '#weight' => 100,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    $summary[] = $this->t('Mode: @mode', ['@mode' => $this->getSetting('embed_mode')]);

    if (in_array($this->getSetting('embed_mode'), ['SIZED_CONTAINER', 'IN_LINE'])) {
      $summary[] = $this->t('Size: @width x @height', [
        '@width' => $this->getSetting('width'),
        '@height' => $this->getSetting('height'),
      ]);
    }

    $summary[] = $this->t('Download button: @state', ['@state' => $this->getSetting('show_download') ? $this->t('Yes') : $this->t('No')]);
    $summary[] = $this->t('Print button: @state', ['@state' => $this->getSetting('show_print') ? $this->t('Yes') : $this->t('No')]);

    if ($this->getSetting('embed_mode') === 'FULL_WINDOW') {
      $summary[] = $this->t('Annotation tools: @state', ['@state' => $this->getSetting('show_annotations') ? $this->t('Yes') : $this->t('No')]);
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];
    $entity = $items->getEntity();
    $field_name = $items->getFieldDefinition()->getName();
    $embed_mode = $this->getSetting('embed_mode');

    // Only use the global client ID from settings
    $client_id = $this->configFactory->get('pdf_services.adobe_embed_settings')->get('client_id');

    // If no client ID is configured, log a warning and show a message to users with proper permissions
    if (empty($client_id)) {
      \Drupal::logger('pdf_services')->warning('Adobe PDF Embed API client ID is not configured. Configure it in the PDF Services settings.');

      // Only show the warning with link to users with permission to administer the module
      if (\Drupal::currentUser()->hasPermission('administer pdf services')) {
        $url = Url::fromRoute('pdf_services.settings');
        $link = \Drupal\Core\Link::fromTextAndUrl(t('PDF Services settings page'), $url)->toString();

        $elements[0] = [
          '#type' => 'markup',
          '#markup' => '<div class="pdf-viewer-placeholder admin-message">' .
                         t('Adobe PDF Embed API client ID is not configured. Please visit the @link to configure it.',
                           ['@link' => $link]) .
                       '</div>',
          '#attached' => [
            'library' => ['pdf_services/pdf_placeholder_styles'],
          ],
        ];
      } else {
        // For non-admin users, show a simple generic placeholder
        $elements[0] = [
          '#type' => 'markup',
          '#markup' => '<div class="pdf-viewer-placeholder">' . t('PDF viewer is currently unavailable.') . '</div>',
          '#attached' => [
            'library' => ['pdf_services/pdf_placeholder_styles'],
          ],
        ];
      }

      return $elements;
    }

    // Get the page parameter from the URL query, if available
    $page = \Drupal::request()->query->get('page');
    // Ensure the page parameter is a positive integer
    $page = (is_numeric($page) && $page > 0) ? (int) $page : NULL;

    // Load entity storage for files.
    $entity_type_manager = \Drupal::entityTypeManager();
    $file_storage = $entity_type_manager->getStorage('file');

    foreach ($items as $delta => $item) {
      $file_url = '';
      $file_name = '';
      $file = NULL;

      // Handle different field types.
      if ($item->getFieldDefinition()->getType() === 'file' && !empty($item->target_id)) {
        $file = $file_storage->load($item->target_id);
        if ($file) {
          $file_url = $this->fileUrlGenerator->generateAbsoluteString($file->getFileUri());
          $file_name = $file->getFilename();
        }
      }
      elseif ($item->getFieldDefinition()->getType() === 'string' || $item->getFieldDefinition()->getType() === 'uri') {
        if (!empty($item->value) && preg_match('/\.pdf$/i', $item->value)) {
          $file_url = $item->value;
          $file_name = basename($item->value);
        }
      }

      if (!empty($file_url)) {
        // Check for overrides if they're allowed for this field
        $settings = $this->getSettings();

        if ($settings['allow_override'] && $entity->id()) {
          $override = $this->loadOverride($entity->getEntityTypeId(), $entity->id(), $field_name, $delta);

          if ($override) {
            // Apply overrides to settings that are set
            foreach ($override as $key => $value) {
              if ($value !== NULL && isset($settings[$key])) {
                $settings[$key] = $value;
              }
            }
          }
        }

        // Check if file is a PDF for file fields
        $is_pdf = FALSE;
        if (isset($file) && $file) {
          $is_pdf = $file->getMimeType() === 'application/pdf';
        }
        else {
          // For string/uri fields, we already filtered for PDFs in the above check
          $is_pdf = TRUE;
        }

        if ($is_pdf) {
          // Prepare the render array for PDF using Adobe PDF Embed API
          $element = [
            '#theme' => 'pdf_services_adobe_embed',
            '#file_url' => $file_url,
            '#file_name' => $file_name,
            '#embed_mode' => $settings['embed_mode'],
            '#client_id' => $client_id,
            '#width' => $settings['width'],
            '#height' => $settings['height'],
            '#show_download' => $settings['show_download'],
            '#show_print' => $settings['show_print'],
            '#show_annotations' => $settings['show_annotations'],
            '#show_bookmark' => $settings['show_bookmark'],
            '#show_full_screen' => $settings['show_full_screen'],
            '#show_direct_link' => $settings['show_direct_link'],
            '#default_view' => $settings['default_view'],
            '#page' => $page,
            '#attached' => [
              'library' => [
                'pdf_services/adobe_pdf_embed',
              ],
            ],
          ];

          // Add div ID for container.
          $element['#div_id'] = 'adobe-dc-view-' . $delta;

          // Add a class for overridden settings
          if (!empty($override)) {
            $element['#attributes']['class'][] = 'pdf-embed-overridden';
          }
        }
        else {
          // For non-PDF files, create a file link using a template
          $element = [
            '#theme' => 'pdf_services_file_link',
            '#file_url' => $file_url,
            '#file_name' => $file_name,
          ];
        }

        $elements[$delta] = $element;
      }
    }

    return $elements;
  }

  /**
   * Loads PDF embed override settings for a specific entity field item.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param string $entity_id
   *   The entity ID.
   * @param string $field_name
   *   The field name.
   * @param int $delta
   *   The field delta.
   *
   * @return array|null
   *   An array of override settings, or null if no override exists.
   */
  protected function loadOverride($entity_type, $entity_id, $field_name, $delta) {
    $result = \Drupal::database()->select('pdf_embed_override', 'p')
      ->fields('p', [
        'embed_mode',
        'width',
        'height',
        'show_download',
        'show_print',
        'show_annotations',
        'show_bookmark',
        'show_full_screen',
        'default_view',
        'show_direct_link',
      ])
      ->condition('entity_type', $entity_type)
      ->condition('entity_id', $entity_id)
      ->condition('field_name', $field_name)
      ->condition('delta', $delta)
      ->execute()
      ->fetchAssoc();

    return $result;
  }

  /**
   * Saves PDF embed override settings for a specific entity field item.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param string $entity_id
   *   The entity ID.
   * @param string $field_name
   *   The field name.
   * @param int $delta
   *   The field delta.
   * @param array $values
   *   The values to save.
   */
  protected function saveOverride($entity_type, $entity_id, $field_name, $delta, array $values) {
    $database = \Drupal::database();

    // Check if override already exists
    $exists = $database->select('pdf_embed_override', 'p')
      ->fields('p', ['id'])
      ->condition('entity_type', $entity_type)
      ->condition('entity_id', $entity_id)
      ->condition('field_name', $field_name)
      ->condition('delta', $delta)
      ->execute()
      ->fetchField();

    $values['entity_type'] = $entity_type;
    $values['entity_id'] = $entity_id;
    $values['field_name'] = $field_name;
    $values['delta'] = $delta;

    if ($exists) {
      $database->update('pdf_embed_override')
        ->fields($values)
        ->condition('id', $exists)
        ->execute();
    }
    else {
      $database->insert('pdf_embed_override')
        ->fields($values)
        ->execute();
    }
  }

  /**
   * Deletes PDF embed override settings for a specific entity field item.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param string $entity_id
   *   The entity ID.
   * @param string $field_name
   *   The field name.
   * @param int $delta
   *   The field delta.
   */
  protected function deleteOverride($entity_type, $entity_id, $field_name, $delta) {
    \Drupal::database()->delete('pdf_embed_override')
      ->condition('entity_type', $entity_type)
      ->condition('entity_id', $entity_id)
      ->condition('field_name', $field_name)
      ->condition('delta', $delta)
      ->execute();
  }
}
