<?php

namespace Drupal\pdf_services\Plugin\QueueWorker;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\file\FileInterface;
use Drupal\pdf_services\Service\PdfServicesClient;
use Drupal\pdf_services\Service\PdfServicesManager;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Processes PDF service operations in a chain.
 *
 * @QueueWorker(
 *   id = "pdf_services_processing",
 *   title = @Translation("PDF Services Processing"),
 *   cron = {
 *     "time" = 60
 *   }
 * )
 */
class PdfProcessingQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * The PDF services manager.
   *
   * @var \Drupal\pdf_services\Service\PdfServicesManager
   */
  protected $servicesManager;

  /**
   * The PDF services client.
   *
   * @var \Drupal\pdf_services\Service\PdfServicesClient
   */
  protected $pdfClient;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The queue factory.
   *
   * @var \Drupal\Core\Queue\QueueFactory
   */
  protected $queueFactory;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * Constructs a new PdfProcessingQueueWorker.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    array $plugin_definition,
    PdfServicesManager $services_manager,
    PdfServicesClient $pdf_client,
    EntityTypeManagerInterface $entity_type_manager,
    LoggerChannelFactoryInterface $logger_factory,
    FileSystemInterface $file_system,
    ClientInterface $http_client,
    QueueFactory $queue_factory,
    ConfigFactoryInterface $config_factory,
    TimeInterface $time
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->servicesManager = $services_manager;
    $this->pdfClient = $pdf_client;
    $this->entityTypeManager = $entity_type_manager;
    $this->logger = $logger_factory->get('pdf_services');
    $this->fileSystem = $file_system;
    $this->httpClient = $http_client;
    $this->queueFactory = $queue_factory;
    $this->configFactory = $config_factory;
    $this->time = $time;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('pdf_services.manager'),
      $container->get('pdf_services.client'),
      $container->get('entity_type.manager'),
      $container->get('logger.factory'),
      $container->get('file_system'),
      $container->get('http_client'),
      $container->get('queue'),
      $container->get('config.factory'),
      $container->get('datetime.time')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data) {
    try {
      // Check if processing is enabled in settings
      $processing_enabled = $this->configFactory->get('pdf_services.settings')->get('processing_enabled') ?? TRUE;

      // Check if valid API credentials are configured
      $has_valid_credentials = $this->pdfClient->hasValidCredentials();

      // Load the status entity
      $status = $this->entityTypeManager
        ->getStorage('pdf_processing_status')
        ->load($data['status_id']);

      if (!$status) {
        $this->logger->error('Invalid processing status ID: @id', [
          '@id' => $data['status_id'],
        ]);
        return;
      }

      // If processing is disabled or credentials are missing, leave the status in pending state
      if (!$processing_enabled || !$has_valid_credentials) {
        // If status is already in processing state, put it back to pending
        if ($status->get('state')->value === 'processing') {
          $status->set('state', 'pending');
          $status->save();

          if (!$processing_enabled) {
            $this->logger->notice('PDF processing is disabled. Setting @filename back to pending state.', [
              '@filename' => $status->get('filename')->value,
            ]);
          }
          else {
            $this->logger->notice('Adobe API credentials are missing. Setting @filename back to pending state.', [
              '@filename' => $status->get('filename')->value,
            ]);
          }
        } else {
          if (!$processing_enabled) {
            $this->logger->info('PDF processing is disabled. Skipping @filename.', [
              '@filename' => $status->get('filename')->value,
            ]);
          }
          else {
            $this->logger->info('Adobe API credentials are missing. Skipping @filename.', [
              '@filename' => $status->get('filename')->value,
            ]);
          }
        }
        return;
      }

      // Extract field settings from the status entity
      $field_settings = $status->get('field_settings')->getValue();

      // If field_settings is empty or has default serialization, handle that case
      if (empty($field_settings) || is_string($field_settings)) {
        if (is_string($field_settings)) {
          // Unserialize if it's a string
          $field_settings = unserialize($field_settings);
        } else {
          // Use default settings
          $field_settings = [
            'check_properties' => TRUE,
            'check_accessibility' => TRUE,
            'page_size_threshold' => 500000,
            'compression_level' => 'MEDIUM',
          ];
        }
      }

      // Set current service being processed.
      $status->state = 'processing';
      $status->operation = $data['service'];
      $status->save();

      // Load the file entity
      $file = $this->entityTypeManager
        ->getStorage('file')
        ->load($status->fid->value);

      if (!$file) {
        $this->logger->error('Missing file for processing status: @id', [
          '@id' => $status->id(),
        ]);
        $this->servicesManager->recordServiceFailure(
          $status,
          $data['service'],
          'File not found'
        );
        return;
      }

      // Phase 1: Submit file and get callback URL.
      if (empty($status->callback_url->value)) {
        $this->initiateProcessing($status, $file, $data['service']);
        return;
      }

      // Phase 2: Check callback URL for completion.
      $this->checkProcessingStatus($status, $file, $data['service']);
    }
    catch (\Exception $e) {
      $this->logger->error('Error processing service @service: @error', [
        '@service' => $data['service'],
        '@error' => $e->getMessage(),
      ]);
      if (isset($status)) {
        $this->servicesManager->recordServiceFailure(
          $status,
          $data['service'],
          $e->getMessage()
        );
        // Check if we're within retry limits
        $retries = $status->get('retries')->value ?: 0;
        $retry_limit = $this->configFactory->get('pdf_services.settings')->get('retry_limit') ?: 3;

        if ($retries < $retry_limit) {
          // Increment retry count
          $status->set('retries', $retries + 1);
          // Back to pending for next cron run
          $status->set('state', 'pending');
          // Add exponential backoff delay
          $status->set('retry_after', time() + pow(2, $retries) * 60);
          $status->save();

          $this->logger->notice('Queued retry @current of @max for @file (@service)', [
            '@current' => $retries + 1,
            '@max' => $retry_limit,
            '@file' => $file->getFilename(),
            '@service' => $data['service'],
          ]);
        } else {
          // Mark as failed - reached retry limit
          $status->set('state', 'failed');
          $status->save();

          $this->logger->error('Maximum retry attempts reached for @file', [
            '@file' => $file->getFilename(),
          ]);
        }
      }
    }
  }

  /**
   * Creates a new processing status entity.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file entity.
   * @param string $checksum
   *   The file checksum.
   * @param string $operation
   *   The operation to perform.
   *
   * @return \Drupal\Core\Entity\EntityInterface
   *   The created status entity.
   */
  protected function createStatusEntity(FileInterface $file, $checksum, $operation, array $field_settings = []) {
    $status = $this->entityTypeManager->getStorage('pdf_processing_status')
      ->create([
        'fid' => $file->id(),
        'filename' => $file->getFilename(),
        'state' => 'pending',
        'checksum' => $checksum,
        'operation' => $operation,
        'created' => $this->time->getRequestTime(),
        'changed' => $this->time->getRequestTime(),
        'field_settings' => $field_settings,
      ]);
    $status->save();
    return $status;
  }

  /**
   * Queues a new processing operation.
   *
   * @param int $status_id
   *   The status entity ID.
   * @param string $service
   *   The service ID to queue.
   */
  protected function queueOperation($status_id, $service) {
    $queue = $this->queueFactory->get('pdf_services_processing');
    $queue->createItem([
      'status_id' => $status_id,
      'service' => $service,
    ]);
  }

  /**
   * Gets the number of retry attempts allowed.
   *
   * @return int
   *   The maximum number of retries allowed.
   */
  protected function getMaxRetries() {
    return $this->configFactory
      ->get('pdf_services.settings')
      ->get('retry_limit') ?? 3;
  }

  /**
   * Initiates processing with Adobe API.
   */
  protected function initiateProcessing($status, $file, $service_id) {
    try {
      // Calculate checksum early.
      $realpath = $this->fileSystem->realpath($file->getFileUri());
      if (!$realpath) {
        throw new \Exception('Failed to get real path for file: ' . $file->getFilename());
      }

      $checksum = md5_file($realpath);
      if (!$checksum) {
        throw new \Exception('Failed to calculate checksum for file: ' . $file->getFilename());
      }

      // Ensure checksum is set on status entity.
      if (!$status->checksum->value) {
        $status->checksum = $checksum;
        $status->save();
      }

      // If we already have an asset ID, skip upload.
      if (!$status->asset_id->value) {
        $uploadResult = $this->pdfClient->uploadFile($file->getFileUri());
        if (!$uploadResult) {
          throw new \Exception('File upload failed');
        }
        $status->asset_id = $uploadResult['assetId'];
      }

      // Start the operation.
      $operation = $this->pdfClient->startOperation(
        $service_id,
        $status->asset_id->value
      );

      if (!$operation || empty($operation['statusUrl'])) {
        throw new \Exception('Failed to start operation');
      }

      // Save callback URL and job ID.
      $status->callback_url = $operation['statusUrl'];
      $status->job_id = $operation['jobId'];
      $status->save();

      // Requeue for status check on next cron run.
      $this->queueOperation($status->id(), $service_id);

      $this->logger->info('Initiated @service processing for @filename', [
        '@service' => $service_id,
        '@filename' => $file->getFilename(),
      ]);
    }
    catch (\Exception $e) {
      throw new \Exception('Failed to initiate processing: ' . $e->getMessage());
    }
  }

  /**
   * Checks processing status via callback URL.
   */
  protected function checkProcessingStatus($status, $file, $service_id) {
    try {
      $result = $this->pdfClient->checkOperationStatus(
        $status->callback_url->value
      );

      if ($result === NULL) {
        // Still processing, requeue for next cron run.
        $this->queueOperation($status->id(), $service_id);
        return;
      }

      if ($result === FALSE) {
        // Check if this was due to INVALID_JOB_ID (expired after 24h)
        if (!empty($status->job_id->value)) {
          // If we have a job ID but the request failed, check how old the status is
          $status_age = $this->time->getRequestTime() - $status->changed->value;

          // If the status is older than 20 hours, assume it's an expired job ID
          // (Adobe removes jobs after 24 hours, but we're using 20 hours to be safe)
          if ($status_age > (20 * 3600)) {
            $this->logger->warning('Job ID has likely expired (jobs are removed after 24 hours): @job_id', [
              '@job_id' => $status->job_id->value,
            ]);
          }
        }

        // Check status failed due to an Error response - mark as failed
        $this->servicesManager->recordServiceFailure(
          $status,
          $service_id,
          'Failed to check processing status'
        );
        return; // Add this return to prevent executing the code below with FALSE
      }

      // At this point, $result must be an array, so we can safely access array elements
      // Add status ID to result data for processServiceResult.
      $result['status_id'] = $status->id();

      // Process completed results based on service type.
      $processedResult = $this->processServiceResult($service_id, $result, $file);

      // Processing completed successfully.
      $this->servicesManager->recordServiceCompletion(
        $status,
        $service_id,
        $processedResult
      );
    }
    catch (\Exception $e) {
      throw new \Exception('Failed to check processing status: ' . $e->getMessage());
    }
  }

  /**
 * Gets field settings from a status entity.
 *
 * @param int $status_id
 *   The status entity ID.
 *
 * @return array
 *   The field settings array.
 */
protected function getFieldSettings($status_id) {
  try {
    $status = $this->entityTypeManager
      ->getStorage('pdf_processing_status')
      ->load($status_id);

    if (!$status) {
      return [];
    }

    // Get the value from the string_long field
    $settings_json = $status->get('field_settings')->value;

    // Return empty array if the field is empty
    if (empty($settings_json)) {
      return [];
    }

    // Decode the JSON settings
    $settings = json_decode($settings_json, TRUE);

    // Return the decoded settings or empty array if decoding failed
    return (json_last_error() === JSON_ERROR_NONE) ? $settings : [];
  }
  catch (\Exception $e) {
    $this->logger->error('Error getting field settings: @error', [
      '@error' => $e->getMessage(),
    ]);
    return [];
  }
}

  /**
   * Process service-specific results.
   */
  protected function processServiceResult($service_id, $result, $file) {
    // First, ensure we can get a valid checksum.
    $checksum = $this->getFileChecksum($file);
    if (!$checksum) {
      throw new \Exception('Failed to calculate checksum for file: ' . $file->getFilename());
    }

    // Extract field settings from the status entity
    $field_settings = !empty($result['status_id']) ?
      $this->getFieldSettings($result['status_id']) : [];

    switch ($service_id) {
      case 'pdfproperties':
        if (empty($result['metadata'])) {
          throw new \Exception('No metadata in properties result');
        }

        // Create analysis result entity
        $storage = $this->entityTypeManager->getStorage('pdf_analysis_result');
        $analysis = $storage->create([
          'fid' => $file->id(),
          'timestamp' => $this->time->getRequestTime(),
          'is_linearized' => !empty($result['metadata']['document']['is_linearized']),
          'is_tagged' => !empty($result['metadata']['document']['is_tagged']),
          'is_encrypted' => !empty($result['metadata']['document']['is_encrypted']),
          'file_size' => $file->getSize(),
          'page_count' => $result['metadata']['document']['page_count'] ?? 0,
          'avg_page_bytes' => $file->getSize() / ($result['metadata']['document']['page_count'] ?? 1),
          'results' => json_encode($result['metadata']),
          'checksum' => $checksum,
        ]);
        $analysis->save();

        // Skip additional services if the file is encrypted
        if (!empty($result['metadata']['document']['is_encrypted'])) {
          $this->logger->warning('Skipping additional processing for encrypted PDF: @filename', [
            '@filename' => $file->getFilename(),
          ]);
          return $result['metadata'];
        }

        // Check if compression is enabled and not set to NONE
        $compression_level = $field_settings['compression_level'] ?? 'MEDIUM';
        $threshold = $field_settings['page_size_threshold'] ?? 500000;
        $avg_page_bytes = $file->getSize() / ($result['metadata']['document']['page_count'] ?? 1);

        // Queue compression ONLY if:
        // 1. The compression level is not 'NONE' AND
        // 2. The file is not already linearized AND
        // 3. Either the file exceeds our size threshold OR compression is explicitly enabled
        $needs_compression = empty($result['metadata']['document']['is_linearized']);
        $exceeds_threshold = $avg_page_bytes > $threshold;
        $compression_enabled = $compression_level !== 'NONE';

        if ($compression_enabled && $needs_compression && $exceeds_threshold) {
          $status = $this->createStatusEntity($file, $checksum, 'compresspdf');
          $this->queueOperation($status->id(), 'compresspdf');

          $this->logger->info('Queued linearization for @filename (reason: @reason)', [
            '@filename' => $file->getFilename(),
            '@reason' => $exceeds_threshold ? 'exceeds size threshold' : 'not linearized'
          ]);
        } else {
          $this->logger->info('Skipping compression for @filename (already linearized: @lin, size below threshold: @size)', [
            '@filename' => $file->getFilename(),
            '@lin' => !empty($result['metadata']['document']['is_linearized']) ? 'yes' : 'no',
            '@size' => $avg_page_bytes <= $threshold ? 'yes' : 'no'
          ]);
        }

        // Queue accessibility check if the PDF is tagged AND we don't already have a recent check
        $check_accessibility = $field_settings['check_accessibility'] ?? TRUE;
        if ($check_accessibility && !empty($result['metadata']['document']['is_tagged'])) {
          // Check if we already have a recent accessibility check
          $accessibility_storage = $this->entityTypeManager->getStorage('pdf_accessibility_result');
          $existing_checks = $accessibility_storage->getQuery()
            ->condition('fid', $file->id())
            ->condition('created', $this->time->getRequestTime() - 86400, '>')  // Within last 24h
            ->count()
            ->accessCheck(FALSE)
            ->execute();

          if ($existing_checks == 0) {
            $status = $this->createStatusEntity($file, $checksum, 'accessibilitychecker');
            $this->queueOperation($status->id(), 'accessibilitychecker');

            $this->logger->info('Queued accessibility check for tagged PDF: @filename', [
              '@filename' => $file->getFilename(),
            ]);
          } else {
            $this->logger->info('Skipping accessibility check for @filename (recent check exists)', [
              '@filename' => $file->getFilename(),
            ]);
          }
        }

        return $result['metadata'];

      case 'accessibilitychecker':
        if (empty($result['report']['downloadUri'])) {
          throw new \Exception('No download URI in accessibility result');
        }

        try {
          // Process and save accessibility check.
          $processedResult = $this->processAccessibilityCheck($result, $file);

          // Email editor if accessibility check fails.
          if (!$processedResult['passes']) {
            /** @var \Drupal\pdf_services\Service\PdfAccessibilityNotificationService $notification_service */
            $notification_service = \Drupal::service('pdf_services.notification');
            $notification_service->sendNotification($file, $processedResult);
          }

          // Update the processing status entity with the checksum.
          if ($status = $this->entityTypeManager->getStorage('pdf_processing_status')->load($result['status_id'])) {
            $status->checksum = $this->getFileChecksum($file);
            $status->save();
          }

          $this->logger->notice('Saved accessibility check results for @filename. Status: @status', [
            '@filename' => $file->getFilename(),
            '@status' => $processedResult['passes'] ? 'PASSED' : 'FAILED',
          ]);

          return $processedResult;
        }
        catch (\Exception $e) {
          $this->logger->error('Failed to process accessibility report: @error', [
            '@error' => $e->getMessage(),
          ]);
          throw $e;
        }

        case 'compresspdf':
          if (empty($result['asset']['downloadUri'])) {
            throw new \Exception('No download URI in compression result');
          }

          // Download optimized file
          $temp_destination = 'temporary://pdf_services/optimized_' . $file->getFilename();
          if (!$this->pdfClient->downloadResult($result['asset']['downloadUri'], $temp_destination)) {
            throw new \Exception('Failed to download optimized file');
          }

          // Get original and new file sizes for comparison
          $original_size = $file->getSize();
          $optimized_size = filesize($temp_destination);

          // Only replace the file if we actually reduced the size by at least 5%
          $size_reduction_percentage = 100 - (($optimized_size / $original_size) * 100);
          $size_reduction_significant = $size_reduction_percentage >= 5;

          $this->logger->notice('Compression results for @filename: original @orig, optimized @opt (@pct% reduction)', [
            '@filename' => $file->getFilename(),
            '@orig' => $this->formatBytes($original_size),
            '@opt' => $this->formatBytes($optimized_size),
            '@pct' => number_format($size_reduction_percentage, 1)
          ]);

          // Update the existing analysis with compression data
          $analysis_storage = $this->entityTypeManager->getStorage('pdf_analysis_result');
          $previous_analyses = $analysis_storage->loadByProperties(['fid' => $file->id()]);

          if (empty($previous_analyses)) {
            throw new \Exception('No existing analysis found for file before compression');
          }

          // Get the most recent analysis
          usort($previous_analyses, function ($a, $b) {
            return $b->get('timestamp')->value - $a->get('timestamp')->value;
          });
          $analysis = reset($previous_analyses);

          // Update the existing analysis with compression data
          $results_data = json_decode($analysis->get('results')->value, TRUE);
          $results_data['compression'] = [
            'original_size' => $original_size,
            'compressed_size' => $optimized_size,
            'compression_ratio' => ($optimized_size / $original_size) * 100,
            'compression_level' => $this->configFactory->get('pdf_services.settings')->get('compression_level'),
            'timestamp' => $this->time->getRequestTime(),
          ];

          $analysis->set('is_linearized', TRUE)
            ->set('file_size', $original_size) // Keep the original size in analysis
            ->set('original_size', $original_size)
            ->set('compressed_size', $optimized_size)
            ->set('compression_ratio', ($optimized_size / $original_size) * 100)
            ->set('results', json_encode($results_data));

          $analysis->save();

          $this->logger->notice('Updated analysis record with compression data for @filename', [
            '@filename' => $file->getFilename(),
            'compression_ratio' => number_format(($optimized_size / $original_size) * 100, 1) . '%',
            'size_reduction' => $this->formatBytes($original_size - $optimized_size),
          ]);

          // Check if auto-replace is enabled AND we had significant size reduction
          $config = $this->configFactory->get('pdf_services.settings');
          if ($size_reduction_significant) {
            // Set a flag to prevent circular processing when we update the file
            $file->pdf_services_skip_processing = TRUE;

            // Replace the file
            $this->logger->debug('Attempting to replace original file:', [
              'source' => $temp_destination,
              'destination' => $file->getFileUri()
            ]);

            if (!$this->fileSystem->copy($temp_destination, $file->getFileUri(), FileSystemInterface::EXISTS_REPLACE)) {
              throw new \Exception('Failed to replace original file with optimized version');
            }

            // Clear file status cache and verify new size
            clearstatcache(TRUE, $file->getFileUri());
            $new_size = filesize($file->getFileUri());

            $this->logger->debug('File replacement complete:', [
              'new_size' => $new_size,
              'uri' => $file->getFileUri()
            ]);

            // Update file entity
            $file->setSize($new_size);
            $file->save();

            $this->logger->notice('Replaced original file with optimized version: @filename', [
              '@filename' => $file->getFilename(),
            ]);

            $destination = $file->getFileUri();
          } else {
            $reason = 'insufficient size reduction';
            $this->logger->notice('Skipping file replacement for @filename (@reason)', [
              '@filename' => $file->getFilename(),
              '@reason' => $reason
            ]);
            $destination = $temp_destination;
          }

          return [
            'original_size' => $original_size,
            'optimized_size' => $optimized_size,
            'reduction_percent' => $size_reduction_percentage,
            'file_path' => $destination,
          ];

      default:
        throw new \Exception("Unknown service: $service_id");
    }
  }

  /**
   * Safely gets a checksum for any type of Drupal file stream.
   */
  protected function getFileChecksum(FileInterface $file) {
    $uri = $file->getFileUri();
    $realpath = $this->fileSystem->realpath($uri);

    if (!$realpath) {
      // For streams that don't support realpath, try to read the file directly.
      if ($handle = fopen($uri, 'rb')) {
        $ctx = hash_init('md5');
        while (!feof($handle)) {
          hash_update($ctx, fread($handle, 8192));
        }
        fclose($handle);
        return hash_final($ctx);
      }
      return FALSE;
    }

    return md5_file($realpath);
  }

  /**
   * Downloads a file from a URI.
   *
   * @param string $uri
   *   The URI to download from.
   *
   * @return string
   *   The response body content.
   */
  protected function downloadFile($uri) {
    $response = $this->httpClient->request('GET', $uri);
    return (string) $response->getBody();
  }

  /**
   * Process accessibility checks.
   *
   * @param array $result
   *   The result array from the API.
   * @param \Drupal\file\FileInterface $file
   *   The file being processed.
   *
   * @return array
   *   The processed results.
   */
  protected function processAccessibilityCheck(array $result, FileInterface $file) {
    // Download report.
    $report_content = $this->downloadFile($result['report']['downloadUri']);
    $accessibility_report = json_decode($report_content, TRUE);

    if (json_last_error() !== JSON_ERROR_NONE) {
      throw new \Exception('Failed to parse accessibility report JSON');
    }

    $summary = $accessibility_report['Summary'] ?? [];
    $passes = FALSE;

    if (!empty($summary)) {
      $total_failed = ($summary['Failed'] ?? 0)
          + ($summary['Failed manually'] ?? 0);
      $needs_manual = $summary['Needs manual check'] ?? 0;
      $passes = ($total_failed === 0 && $needs_manual <= 2);
    }

    $storage = $this->entityTypeManager->getStorage('pdf_accessibility_result');
    $current_time = $this->time->getRequestTime();

    $accessibility = $storage->create([
      'fid' => $file->id(),
      'passes' => $passes,
      'report' => json_encode([
        'summary' => $summary,
        'detailed_report' => $accessibility_report['Detailed Report'] ?? [],
        'timestamp' => $current_time,
      ]),
      'created' => $current_time,
    ]);
    $accessibility->save();

    return [
      'passes' => $passes,
      'summary' => $summary,
      'detailed_report' => $accessibility_report['Detailed Report'] ?? [],
    ];
  }

  /**
 * Formats bytes into human readable sizes.
 *
 * @param int $bytes
 *   The size in bytes.
 * @param int $precision
 *   The number of decimal places to show.
 *
 * @return string
 *   Human readable size.
 */
protected function formatBytes($bytes, $precision = 2) {
  $units = ['B', 'KB', 'MB', 'GB', 'TB'];

  $bytes = max($bytes, 0);
  $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
  $pow = min($pow, count($units) - 1);

  $bytes /= pow(1024, $pow);

  return round($bytes, $precision) . ' ' . $units[$pow];
}

}
