<?php

namespace Drupal\pdf_services\Service;

use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Service for enhancing accessibility descriptions.
 */
class AccessibilityEnhancementService {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * Enhances a report with custom descriptions.
   *
   * @param array $report
   *   The original report.
   *
   * @return array
   *   The enhanced report.
   */
  public function enhanceReport(array $report) {
    // Get the enhancements from configuration.
    $config = $this->configFactory->get('pdf_services.enhanced_descriptions');
    $enhancements = $config->get('enhancements') ?: [];

    if (empty($enhancements) || empty($report['detailed_report'])) {
      return $report;
    }

    // Create a lookup map for faster processing.
    $enhancementMap = [];
    foreach ($enhancements as $enhancement) {
      if (!empty($enhancement['rule_title']) && !empty($enhancement['enhanced_description'])) {
        $enhancementMap[$enhancement['rule_title']] = $enhancement['enhanced_description'];
      }
    }

    // Process each section and rule in the report.
    foreach ($report['detailed_report'] as $section => $rules) {
      foreach ($rules as $key => $rule) {
        if (isset($rule['Rule']) && isset($enhancementMap[$rule['Rule']])) {
          // Store original description
          $report['detailed_report'][$section][$key]['OriginalDescription'] = $rule['Description'];

          // Add enhancement as a separate property
          $report['detailed_report'][$section][$key]['EnhancedDescription'] = $enhancementMap[$rule['Rule']];
        }
      }
    }

    return $report;
  }
}
