<?php

namespace Drupal\pdf_services\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;
use Drupal\file\FileInterface;
use Drupal\Core\File\FileSystemInterface;

/**
 * Service for interacting with Adobe PDF Services API.
 */
class PdfServicesClient {

  /**
   * The base URL for Adobe PDF Services API.
   */
  const API_BASE_URL = 'https://pdf-services.adobe.io';

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The logger.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * Constructs a new PdfServicesClient.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    ClientInterface $http_client,
    LoggerChannelFactoryInterface $logger_factory,
    FileSystemInterface $file_system
  ) {
    $this->configFactory = $config_factory;
    $this->httpClient = $http_client;
    $this->logger = $logger_factory;
    $this->logger = $logger_factory->get('pdf_services');
    $this->fileSystem = $file_system;
  }

  /**
   * Checks if valid API credentials are configured.
   *
   * @return bool
   *   TRUE if credentials are configured, FALSE otherwise.
   */
  public function hasValidCredentials() {
    $config = $this->configFactory->get('pdf_services.settings');
    $client_id = $config->get('client_id');
    $client_secret = $config->get('client_secret');

    return !empty($client_id) && !empty($client_secret);
  }

  /**
   * Gets an access token from Adobe API.
   *
   * @return string|null
   *   The access token or null if failed.
   */
  protected function getAccessToken() {
    try {
      $config = $this->configFactory->get('pdf_services.settings');

      // Create default credentials from configuration
      $credentials = [
        'grant_type' => 'client_credentials',
        'client_id' => $config->get('client_id'),
        'client_secret' => $config->get('client_secret'),
        'scope' => 'openid,AdobeID,DCAPI',
      ];

      // Allow other modules to alter the credentials
      \Drupal::moduleHandler()->alter('pdf_services_credentials', $credentials);

      $response = $this->httpClient->post('https://ims-na1.adobelogin.com/ims/token/v3', [
        'form_params' => $credentials,
      ]);

      $data = json_decode($response->getBody(), TRUE);
      return $data['access_token'] ?? NULL;
    }
    catch (GuzzleException $e) {
      $this->logger->error('Failed to get access token: @error', ['@error' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Gets request headers with authentication.
   *
   * @param string $token
   *   The access token.
   *
   * @return array
   *   Array of request headers.
   */
  protected function getHeaders($token) {
    $config = $this->configFactory->get('pdf_services.settings');
    return [
      'Authorization' => 'Bearer ' . $token,
      'x-api-key' => $config->get('client_id'),
      'Content-Type' => 'application/json',
      'Accept' => 'application/json',
    ];
  }

  /**
   * Logs API request details for debugging.
   */
  protected function logRequestDetails($method, $url, $headers, $response = NULL) {
    try {
      // Log request details
      $this->logger->debug('API Request Details:', [
        'method' => $method,
        'url' => $url,
        'headers' => json_encode(array_map(function($header) {
          // Safely handle null values and mask sensitive data
          if (!is_string($header)) {
            return $header;
          }
          return strpos($header, 'Bearer') === 0 ? 'Bearer [masked]' : $header;
        }, $headers)),
      ]);

      if ($response) {
        // Get response content
        $body = (string) $response->getBody();
        $response->getBody()->rewind();

        // Log response details
        $this->logger->debug('API Response Details:', [
          'status_code' => $response->getStatusCode(),
          'reason' => $response->getReasonPhrase(),
          'body' => $body,
        ]);

        try {
          // Try to decode JSON for prettier logging
          $jsonBody = json_decode($body, TRUE);
          if (json_last_error() === JSON_ERROR_NONE) {
            $this->logger->debug('Decoded Response:', [
              'data' => var_export($jsonBody, TRUE),
            ]);
          }
        } catch (\Exception $e) {
          // If JSON decode fails, log raw body
          $this->logger->debug('Raw Response Body: @body', [
            '@body' => $body,
          ]);
        }
      }
    } catch (\Exception $e) {
      $this->logger->error('Error logging request details: @error', [
        '@error' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Validates a file before processing.
   *
   * @param string $file_path
   *   Path to the file.
   *
   * @return bool
   *   TRUE if valid, FALSE otherwise.
   */
  protected function validateFile($file_path) {
    if (!file_exists($file_path)) {
      $this->logger->error('File does not exist: @path', [
        '@path' => $file_path,
      ]);
      return FALSE;
    }

    if (!is_readable($file_path)) {
      $this->logger->error('File is not readable: @path', [
        '@path' => $file_path,
      ]);
      return FALSE;
    }

    $fileInfo = finfo_open(FILEINFO_MIME_TYPE);
    $mimeType = finfo_file($fileInfo, $file_path);
    finfo_close($fileInfo);

    if ($mimeType !== 'application/pdf') {
      $this->logger->error('Invalid file type: @mime', [
        '@mime' => $mimeType,
      ]);
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Uploads a file to Adobe's service.
   *
   * @param string $file_path
   *   The path to the file.
   *
   * @return array|null
   *   Array containing assetID and uploadUri, or null if failed.
   */
  public function uploadFile($file_path) {
    try {
      $token = $this->getAccessToken();
      if (!$token) {
        return NULL;
      }

      // Validate file before proceeding
      if (!$this->validateFile($file_path)) {
        return NULL;
      }

      // Log file details before upload be sure to include filename
      $this->logger->debug('Uploading file: @path', ['@path' => $file_path]);

      $headers = $this->getHeaders($token);

      // Step 1: Get upload URL
      $response = $this->httpClient->post(self::API_BASE_URL . '/assets', [
        'headers' => $headers,
        'json' => ['mediaType' => 'application/pdf'],
      ]);

      $data = json_decode($response->getBody(), TRUE);

      $uploadUri = $data['uploadUri'] ?? NULL;
      $assetId = $data['assetID'] ?? NULL;

      if (!$uploadUri || !$assetId) {
        $this->logger->error('Missing upload URI or asset ID in response');
        return NULL;
      }

      // Step 2: Upload file
      $uploadResponse = $this->httpClient->put($uploadUri, [
        'body' => file_get_contents($file_path),
        'headers' => [
          'Content-Type' => 'application/pdf',
        ],
      ]);

      return [
        'assetId' => $assetId,
        'uploadUri' => $uploadUri,
      ];
    }
    catch (\Exception $e) {
      // Check if this is a quota exhaustion error
      if ($this->handleApiQuotaError($e, 'uploadFile')) {
        $this->logger->error('API quota exhausted during file upload. Processing has been disabled.');
      } else {
        $this->logger->error('File upload failed: @error', [
          '@error' => $e->getMessage(),
        ]);
      }
      return NULL;
    }
  }

  /**
   * Polls for job completion.
   *
   * @param string $statusUrl
   *   The status URL to poll.
   * @param string $token
   *   The access token.
   *
   * @return array|null
   *   The job result data or null if failed.
   */
  protected function pollJobStatus($statusUrl, $token) {
    $maxAttempts = 10;
    $attempt = 0;
    $delay = 2;

    while ($attempt < $maxAttempts) {
      try {
        $headers = $this->getHeaders($token);
        $this->logger->debug('Polling job status', [
          'attempt' => $attempt + 1,
          'status_url' => $statusUrl
        ]);

        $response = $this->httpClient->get(
          $statusUrl,
          ['headers' => $headers]
        );

        $this->logRequestDetails('GET', $statusUrl, $headers, $response);

        $status = json_decode($response->getBody(), TRUE);

        if ($status['status'] === 'done') {
          return $status;
        }
        elseif ($status['status'] === 'failed') {
          $this->logger->error('Job failed: @error', ['@error' => json_encode($status['error'])]);
          return NULL;
        }

        $attempt++;
        sleep($delay);
      }
      catch (GuzzleException $e) {
        $this->logger->error('Status check failed: @error', ['@error' => $e->getMessage()]);
        return NULL;
      }
    }

    return NULL;
  }

  /**
   * Downloads a result file from Adobe API.
   *
   * @param string $downloadUri
   *   The download URI provided by Adobe API.
   * @param string $destination
   *   The local destination path.
   *
   * @return bool
   *   TRUE if successful, FALSE otherwise.
   */
  public function downloadResult($downloadUri, $destination) {
    try {
      // Ensure the destination directory exists
      $directory = dirname($destination);
      if (!file_exists($directory)) {
        if (!$this->fileSystem->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY)) {
          throw new \Exception("Could not create directory: $directory");
        }
      }

      // Download the file
      $response = $this->httpClient->get($downloadUri);
      file_put_contents($destination, $response->getBody());

      $this->logger->debug('Downloaded result file to @path', [
        '@path' => $destination,
      ]);

      return TRUE;
    }
    catch (\Exception $e) {
      $this->logger->error('Download failed: @error', [
        '@error' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Optimizes a PDF file using Adobe's linearization service.
   *
   * @param string $file_path
   *   The path to the PDF file.
   *
   * @return string|null
   *   The path to the optimized file or null if failed.
   */
  public function optimizePdf(string $file_path) {
    try {
      // Get access token
      $token = $this->getAccessToken();
      if (!$token) {
        $this->logger->error('Failed to obtain access token');
        return NULL;
      }
      $this->logger->debug('Successfully obtained access token');

      // Upload file
      $this->logger->debug('Attempting to upload file: @path', ['@path' => $file_path]);
      $uploadResult = $this->uploadFile($file_path);
      if (!$uploadResult) {
        $this->logger->error('File upload failed');
        return NULL;
      }
      $this->logger->debug('File upload successful: @result', ['@result' => json_encode($uploadResult)]);

      // Start compression job
      $headers = $this->getHeaders($token);
      $jobUrl = self::API_BASE_URL . '/operation/compresspdf';

      // Get compression level from settings
      $compressionLevel = $this->configFactory->get('pdf_services.settings')->get('compression_level') ?? 'MEDIUM';

      $jobPayload = [
        'assetID' => $uploadResult['assetId'],
        'compressionLevel' => $compressionLevel,
      ];

      $this->logger->debug('Starting compression job', [
        'url' => $jobUrl,
        'payload' => json_encode($jobPayload)
      ]);

      try {
        $response = $this->httpClient->post(
          $jobUrl,
          [
            'headers' => $headers,
            'json' => $jobPayload,
          ]
        );
      } catch (GuzzleException $e) {
        $this->logger->error('Job creation failed: @error', [
          '@error' => $e->getMessage(),
          'response' => $e->hasResponse() ? (string) $e->getResponse()->getBody() : 'No response body'
        ]);
        return NULL;
      }

      // Get the location header from the 201 response to know where to poll for job status
      // which would be a url like "https://pdf-services-ue1.adobe.io/operation/compresspdf/Tcxt9J819XrMAv0Dx52DMSpva5DEzSlv/status"
      $jobLocation = $response->getHeader('Location')[0] ?? NULL;

      // Ensure the jobLocation is valid and use it to poll for completion
      if (!$jobLocation) {
        $this->logger->error('No job location found in response');
        return NULL;
      }

      $this->logger->debug('Job started, polling for completion');
      $result = $this->pollJobStatus($jobLocation, $token);
      if (!$result) {
        $this->logger->error('Job completion polling failed');
        return NULL;
      }

      // Download result
      $destination = 'temporary://pdf_services/optimized_' . basename($file_path);
      if (!$this->downloadResult($result['asset']['downloadUri'], $destination)) {
        return NULL;
      }

      // Drupal message showing downlaod link
      $this->logger->debug('PDF optimization completed successfully');


      $this->logger->notice('Download your optimized PDF at: @path', ['@path' => $destination]);


      return $destination;
    }
    catch (\Exception $e) {
      $this->logger->error('PDF optimization failed: @error', ['@error' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Starts an operation and returns status URL.
   *
   * @param string $service_id
   *   The service identifier.
   * @param string $asset_id
   *   The Adobe asset ID.
   *
   * @return array|null
   *   Operation details or null if failed.
   */
  public function startOperation($service_id, $asset_id) {
    try {
      $token = $this->getAccessToken();
      if (!$token) {
        $this->logger->error('Failed to get access token for operation @service', [
          '@service' => $service_id
        ]);
        return NULL;
      }

      $headers = $this->getHeaders($token);
      // Make sure we're using the correct operation endpoint
      $operationUrl = self::API_BASE_URL . '/operation/';
      switch ($service_id) {
        case 'pdfproperties':
          $operationUrl .= 'pdfproperties';
          break;
        case 'accessibilitychecker':
          $operationUrl .= 'accessibilitychecker'; // This might need to be adjusted based on Adobe's API
          break;
        case 'compresspdf':
          $operationUrl .= 'compresspdf';
          break;
        default:
          $this->logger->error('Unknown service ID: @service', [
            '@service' => $service_id
          ]);
          return NULL;
      }

      try {
        $response = $this->httpClient->post($operationUrl, [
          'headers' => $headers,
          'json' => ['assetID' => $asset_id],
        ]);

        // Get the location header for status polling
        $statusUrl = $response->getHeader('Location')[0] ?? NULL;
        if (!$statusUrl) {
          $this->logger->error('No status URL in response headers for @service', [
            '@service' => $service_id,
            'headers' => json_encode($response->getHeaders())
          ]);
          return NULL;
        }

        // Extract job ID from status URL
        preg_match('/\/([^\/]+)\/status$/', $statusUrl, $matches);
        $jobId = $matches[1] ?? NULL;

        return [
          'statusUrl' => $statusUrl,
          'jobId' => $jobId,
        ];
      }
      catch (GuzzleException $e) {
        // Check if this is a quota exhaustion error
        if ($this->handleApiQuotaError($e, $service_id)) {
          $this->logger->error('API quota exhausted while starting operation @service. Processing has been disabled.', [
            '@service' => $service_id
          ]);
        }
        else {
          $this->logger->error('Request failed for @service: @error', [
            '@service' => $service_id,
            '@error' => $e->getMessage(),
            'response' => $e->hasResponse() ? (string) $e->getResponse()->getBody() : 'No response'
          ]);
        }
        return NULL;
      }
    } catch (\Exception $e) {
      $this->logger->error('Error in startOperation: @error', [
        '@error' => $e->getMessage()
      ]);
      return NULL;
    }
  }

  /**
   * Checks operation status.
   *
   * @param string $statusUrl
   *   The status URL to check.
   *
   * @return array|bool|null
   *   - array: Operation completed successfully
   *   - FALSE: Operation failed
   *   - NULL: Still processing
   */
  public function checkOperationStatus($statusUrl) {
    try {
      $token = $this->getAccessToken();
      if (!$token) {
        return FALSE;
      }

      $headers = $this->getHeaders($token);
      $response = $this->httpClient->get($statusUrl, ['headers' => $headers]);
      $status = json_decode($response->getBody(), TRUE);

      switch ($status['status']) {
        case 'done':
          return $status;
        case 'failed':
          // Check if failure is due to quota exhaustion (could be either error code)
          if (!empty($status['error']) && !empty($status['error']['code'])) {
            $quotaErrorCodes = ['INSUFFICIENT_QUOTA', 'QUOTA_EXCEEDED'];
            if (in_array($status['error']['code'], $quotaErrorCodes)) {
              // Handle quota exhaustion using our helper method
              $exception = new \Exception(json_encode($status['error']));
              $this->handleApiQuotaError($exception, 'checkOperationStatus');
            }
          }

          // Log any error codes and messages received
          if (!empty($status['error'])) {
            $errorCode = $status['error']['code'] ?? 'UNKNOWN_ERROR';
            $errorMessage = $status['error']['message'] ?? 'No error message provided';
            $this->logger->error('PDF Services operation failed: @code - @message', [
              '@code' => $errorCode,
              '@message' => $errorMessage,
              '@details' => json_encode($status['error']),
            ]);
          }
          return FALSE;
        default:
          return NULL;
      }
    }
    catch (GuzzleException $e) {
      // Check for invalid job ID (expired after 24 hours)
      if ($e->hasResponse() && $e->getResponse()->getStatusCode() === 400) {
        $responseBody = json_decode((string) $e->getResponse()->getBody(), TRUE);
        if (isset($responseBody['error']['code']) && $responseBody['error']['code'] === 'INVALID_JOB_ID') {
          $this->logger->warning('Job ID has expired (jobs are removed after 24 hours): @url', [
            '@url' => $statusUrl,
          ]);
          // Return FALSE to indicate job failed, allowing queue workers to remove the item
          return FALSE;
        }
      }

      // Check if this is a quota exhaustion error
      if ($this->handleApiQuotaError($e, 'checkOperationStatus')) {
        $this->logger->error('API quota exhausted while checking operation status. Processing has been disabled.');
      } else {
        $this->logger->error('Failed to check operation status: @error', [
          '@error' => $e->getMessage(),
        ]);
        return FALSE;
      }
      return FALSE;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to check operation status: @error', [
        '@error' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  public function downloadFile($uri) {
    try {
        $response = $this->httpClient->get($uri);
        return (string) $response->getBody();
    }
    catch (\Exception $e) {
        $this->logger->error('Failed to download file: @error', [
            '@error' => $e->getMessage()
        ]);
        return FALSE;
    }
}

  /**
   * Checks PDF accessibility compliance.
   *
   * @param string $file_path
   *   The path to the PDF file.
   *
   * @return array|false
   *   Accessibility check results array or FALSE on failure.
   */
  public function checkAccessibility(Request $request) {
    $statusId = $request->query->get('status_id');

    // Load the processing status
    $status = $this->entityTypeManager
      ->getStorage('pdf_processing_status')
      ->load($statusId);

    if (!$status) {
      return new JsonResponse(['error' => 'Invalid status ID'], 404);
    }

    // Check callback URL and job status
    $result = $this->pdfClient->checkOperationStatus($status->get('callback_url')->value);

    if ($result === FALSE) {
      return new JsonResponse(['error' => 'Operation failed'], 500);
    }

    if ($result === NULL || empty($result['report'])) {
      \Drupal::logger('pdf_services')->debug('Accessibility check still processing: @data', [
        '@data' => json_encode($result)
      ]);
      return new JsonResponse(['status' => 'processing']);
    }

    try {
      // Download and process the report if we have a download URI
      if (!empty($result['report']['downloadUri'])) {
        // Download report content
        $report_content = $this->pdfClient->downloadFile($result['report']['downloadUri']);
        $accessibility_report = json_decode($report_content, TRUE);

        if (json_last_error() !== JSON_ERROR_NONE) {
          throw new \Exception('Failed to parse accessibility report JSON');
        }

        $summary = $accessibility_report['Summary'] ?? [];
        $passes = FALSE;

        if (!empty($summary)) {
          $total_failed = ($summary['Failed'] ?? 0) + ($summary['Failed manually'] ?? 0);
          $needs_manual = $summary['Needs manual check'] ?? 0;
          $passes = ($total_failed === 0 && $needs_manual <= 2);
        }

        // Save results
        $storage = $this->entityTypeManager->getStorage('pdf_accessibility_result');
        $accessibility = $storage->create([
          'fid' => $status->get('fid')->value,
          'passes' => $passes,
          'report' => json_encode([
            'summary' => $summary,
            'detailed_report' => $accessibility_report['Detailed Report'] ?? [],
            'timestamp' => \Drupal::time()->getRequestTime(),
          ]),
          'created' => \Drupal::time()->getRequestTime(),
        ]);
        $accessibility->save();

        return new JsonResponse([
          'status' => 'completed',
          'data' => [
            'passes' => $passes,
            'summary' => $summary,
            'details' => $accessibility_report['Detailed Report'] ?? []
          ]
        ]);
      } else {
        \Drupal::logger('pdf_services')->error('No download URI in report result');
        return new JsonResponse(['error' => 'Invalid report data'], 500);
      }
    }
    catch (\Exception $e) {
      \Drupal::logger('pdf_services')->error('Failed to save accessibility results: @error', [
        '@error' => $e->getMessage(),
      ]);
      return new JsonResponse(['error' => 'Failed to save accessibility results'], 500);
    }
}

/**
 * Deletes an asset from Adobe's storage.
 *
 * @param string $assetId
 *   The asset ID to delete.
 *
 * @return bool
 *   TRUE if deletion was successful, FALSE otherwise.
 */
public function deleteAsset($assetId) {
  try {
    $token = $this->getAccessToken();
    if (!$token) {
      $this->logger->error('Failed to get access token for asset deletion');
      return FALSE;
    }

    $headers = $this->getHeaders($token);
    $deleteUrl = self::API_BASE_URL . '/assets/' . $assetId;

    $this->logger->debug('Deleting asset @id', [
      '@id' => $assetId,
    ]);

    $response = $this->httpClient->delete($deleteUrl, [
      'headers' => $headers,
    ]);

    // 204 response indicates success with no content
    if ($response->getStatusCode() === 204) {
      $this->logger->notice('Successfully deleted asset @id', [
        '@id' => $assetId,
      ]);
      return TRUE;
    }

    $this->logger->error('Unexpected response status @status when deleting asset @id', [
      '@status' => $response->getStatusCode(),
      '@id' => $assetId,
    ]);

    return FALSE;
  }
  catch (\Exception $e) {
    $this->logger->error('Error deleting asset @id: @error', [
      '@id' => $assetId,
      '@error' => $e->getMessage(),
    ]);
    return FALSE;
  }
}

/**
 * Checks for API quota exhaustion and handles appropriate responses.
 *
 * @param \Exception $e
 *   The exception thrown during API communication.
 * @param string $operation
 *   The operation being performed.
 *
 * @return bool
 *   TRUE if quota is exhausted, FALSE otherwise.
 */
protected function handleApiQuotaError(\Exception $e, $operation = 'unknown') {
  // Default return value
  $quotaExhausted = FALSE;

  // Check if this is a GuzzleException with a response
  if (method_exists($e, 'hasResponse') && $e->hasResponse()) {
    $response = $e->getResponse();
    $statusCode = $response->getStatusCode();
    $body = (string) $response->getBody();

    // Try to decode the response body
    $errorData = json_decode($body, TRUE);

    // Log the full error for debugging
    $this->logger->error('API error in @operation: @status - @body', [
      '@operation' => $operation,
      '@status' => $statusCode,
      '@body' => $body,
    ]);

    // Check for 429 status code
    if ($statusCode === 429) {
      // Adobe can return multiple error codes for quota exhaustion
      $quotaErrorCodes = ['INSUFFICIENT_QUOTA', 'QUOTA_EXCEEDED'];

      if (isset($errorData['error']['code']) && in_array($errorData['error']['code'], $quotaErrorCodes)) {
        $this->logger->critical('Adobe PDF Services API quota has been exhausted. Processing will be automatically disabled until the quota resets on the first of next month.', [
          'error_message' => $errorData['error']['message'] ?? 'No detailed message available',
          'error_code' => $errorData['error']['code'] ?? 'Unknown error code'
        ]);

        // Disable PDF processing
        $config = \Drupal::configFactory()->getEditable('pdf_services.settings');
        $config->set('processing_enabled', FALSE);
        $config->save();

        // Set state flag to indicate this was an automatic disabling due to quota
        \Drupal::state()->set('pdf_services.auto_disabled', TRUE);

        // Set the flag to indicate quota exhaustion
        $quotaExhausted = TRUE;

        // Inform users via Drupal message system if in a UI context
        if (\Drupal::hasService('messenger')) {
          \Drupal::messenger()->addWarning(t('Adobe PDF Services API quota has been exhausted. PDF processing has been automatically disabled until the quota resets.'));
        }
      }
    }
  }

  return $quotaExhausted;
}

}
