<?php

namespace Drupal\Tests\pdf_services\Unit;

use Drupal\Tests\UnitTestCase;
use Drupal\pdf_services\Service\PdfServicesClient;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use GuzzleHttp\ClientInterface;
use Drupal\Core\File\FileSystemInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\StreamInterface;

/**
 * Tests the PDF Services Client.
 *
 * @group pdf_services
 * @coversDefaultClass \Drupal\pdf_services\Service\PdfServicesClient
 */
class PdfServicesClientTest extends UnitTestCase {

  /**
   * The mocked config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $configFactory;

  /**
   * The mocked HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $httpClient;

  /**
   * The mocked logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $loggerFactory;

  /**
   * The mocked file system.
   *
   * @var \Drupal\Core\File\FileSystemInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $fileSystem;

  /**
   * The PDF Services client being tested.
   *
   * @var \Drupal\pdf_services\Service\PdfServicesClient
   */
  protected $pdfServicesClient;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Mock the config factory
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);

    // Mock the HTTP client
    $this->httpClient = $this->createMock(ClientInterface::class);

    // Mock the logger factory
    $this->loggerFactory = $this->createMock(LoggerChannelFactoryInterface::class);
    $logger = $this->createMock(LoggerChannelInterface::class);
    $this->loggerFactory->expects($this->any())
      ->method('get')
      ->with('pdf_services')
      ->willReturn($logger);

    // Mock the file system
    $this->fileSystem = $this->createMock(FileSystemInterface::class);

    // Create the client
    $this->pdfServicesClient = new PdfServicesClient(
      $this->configFactory,
      $this->httpClient,
      $this->loggerFactory,
      $this->fileSystem
    );
  }

  /**
   * Tests hasValidCredentials method.
   *
   * @covers ::hasValidCredentials
   */
  public function testHasValidCredentials() {
    // Test with valid credentials
    $config = $this->createMock(ImmutableConfig::class);
    $config->expects($this->exactly(2))
      ->method('get')
      ->willReturnMap([
        ['client_id', 'test_client_id'],
        ['client_secret', 'test_client_secret'],
      ]);

    $this->configFactory->expects($this->once())
      ->method('get')
      ->with('pdf_services.settings')
      ->willReturn($config);

    $this->assertTrue($this->pdfServicesClient->hasValidCredentials());
  }

  /**
   * Tests hasValidCredentials with missing credentials.
   *
   * @covers ::hasValidCredentials
   */
  public function testHasValidCredentialsWithMissingCredentials() {
    // Test with missing client_id
    $config = $this->createMock(ImmutableConfig::class);
    $config->expects($this->exactly(2))
      ->method('get')
      ->willReturnMap([
        ['client_id', ''],
        ['client_secret', 'test_client_secret'],
      ]);

    $this->configFactory->expects($this->once())
      ->method('get')
      ->with('pdf_services.settings')
      ->willReturn($config);

    $this->assertFalse($this->pdfServicesClient->hasValidCredentials());
  }

  /**
   * Tests the downloadFile method.
   *
   * @covers ::downloadFile
   */
  public function testDownloadFile() {
    $uri = 'https://example.com/test.pdf';
    $expectedContent = 'PDF content';

    $stream = $this->createMock(StreamInterface::class);
    $stream->expects($this->once())
      ->method('__toString')
      ->willReturn($expectedContent);

    $response = $this->createMock(ResponseInterface::class);
    $response->expects($this->once())
      ->method('getBody')
      ->willReturn($stream);

    $this->httpClient->expects($this->once())
      ->method('request')
      ->with('GET', $uri)
      ->willReturn($response);

    $result = $this->pdfServicesClient->downloadFile($uri);
    $this->assertEquals($expectedContent, $result);
  }

  /**
   * Tests downloadFile with exception.
   *
   * @covers ::downloadFile
   */
  public function testDownloadFileException() {
    $uri = 'https://example.com/test.pdf';

    $this->httpClient->expects($this->once())
      ->method('request')
      ->with('GET', $uri)
      ->willThrowException(new \Exception('Download failed'));

    $result = $this->pdfServicesClient->downloadFile($uri);
    $this->assertFalse($result);
  }
}
