<?php

namespace Drupal\pdf_services\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\pdf_services\Service\PdfServicesClient;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

class PdfServicesScanController extends ControllerBase {
  protected $pdfClient;
  protected $entityTypeManager;
  protected $configFactory;

  public function __construct(PdfServicesClient $pdf_client, EntityTypeManagerInterface $entity_type_manager, ConfigFactoryInterface $config_factory) {
    $this->pdfClient = $pdf_client;
    $this->entityTypeManager = $entity_type_manager;
    $this->configFactory = $config_factory;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('pdf_services.client'),
      $container->get('entity_type.manager'),
      $container->get('config.factory')
    );
  }

  /**
   * Checks if PDF processing is enabled.
   *
   * @return bool
   *   TRUE if processing is enabled, FALSE otherwise.
   */
  protected function isProcessingEnabled() {
    return $this->configFactory->get('pdf_services.settings')->get('processing_enabled') ?? TRUE;
  }

  public function queueForScanning(Request $request) {
    $fid = $request->query->get('fid');
    $file = $this->entityTypeManager->getStorage('file')->load($fid);

    if (!$file || $file->getMimeType() !== 'application/pdf') {
      return new JsonResponse(['error' => 'Invalid file'], 400);
    }

    try {
      // Check for existing accessibility result - add accessCheck(FALSE)
      $accessibilityQuery = $this->entityTypeManager->getStorage('pdf_accessibility_result')
        ->getQuery()
        ->accessCheck(FALSE)  // Explicitly disable access checks
        ->condition('fid', $file->id())
        ->sort('created', 'DESC')
        ->range(0, 1);
      $accessibilityIds = $accessibilityQuery->execute();
      $existingAccessibility = !empty($accessibilityIds) ?
        $this->entityTypeManager->getStorage('pdf_accessibility_result')->load(reset($accessibilityIds)) :
        NULL;

      // Check for any existing pending or processing scan tasks
      $pendingStatusQuery = $this->entityTypeManager->getStorage('pdf_processing_status')
        ->getQuery()
        ->accessCheck(FALSE)
        ->condition('fid', $file->id())
        ->condition('operation', 'accessibilitychecker')
        ->condition('state', ['pending', 'processing'], 'IN')
        ->sort('created', 'DESC')
        ->range(0, 1);
      $pendingStatusIds = $pendingStatusQuery->execute();
      $existingPendingStatus = !empty($pendingStatusIds) ?
        $this->entityTypeManager->getStorage('pdf_processing_status')->load(reset($pendingStatusIds)) :
        NULL;

      // If there's an existing pending or processing task, return its status ID
      if ($existingPendingStatus) {
        \Drupal::logger('pdf_services')->info('Using existing pending scan for file @fid', ['@fid' => $file->id()]);
        return new JsonResponse([
          'status' => 'processing',
          'message' => 'PDF accessibility check already in queue',
          'accessibilityStatusId' => $existingPendingStatus->id(),
        ]);
      }

      // Get file modification time for comparison
      $file_uri = $file->getFileUri();
      $file_mtime = filemtime($file_uri);

      // Calculate a current checksum to detect file changes
      $current_checksum = md5_file($file_uri);

      // Variable to track if we need to process accessibility
      $need_accessibility = TRUE;
      $accessibilityStatusId = NULL;

      // Check if accessibility report exists and is current
      if ($existingAccessibility) {
        // Convert entity to array for easier property access
        $accessibilityValues = $existingAccessibility->toArray();

        // Get the accessibility timestamp
        $accessibilityTimestamp = $accessibilityValues['created'][0]['value'] ?? 0;

        // If accessibility check is newer than file modification time, we can use it
        if ($accessibilityTimestamp >= $file_mtime) {
          $need_accessibility = FALSE;
          \Drupal::logger('pdf_services')->info('Using existing accessibility report for file @fid', ['@fid' => $file->id()]);

          // Get the report JSON
          $reportJson = $accessibilityValues['report'][0]['value'] ?? '{}';
          $report = json_decode($reportJson, TRUE);

          // Apply the same enhancements as the preprocess function
          /** @var \Drupal\pdf_services\Service\AccessibilityEnhancementService $enhancement_service */
          $enhancement_service = \Drupal::service('pdf_services.accessibility_enhancement');
          $processed_report = pdf_services_enhance_accessibility_report($report);
          $processed_report = $enhancement_service->enhanceReport($processed_report);

          // Prepare the accessibility data for return
          $accessibilityData = [
            'passes' => !empty($accessibilityValues['passes'][0]['value']),
            'summary' => $processed_report['summary'] ?? [],
            'details' => $processed_report['detailed_report'] ?? [],
          ];
        }
      }

      // If we already have a valid report, return it immediately
      if (!$need_accessibility) {
        return new JsonResponse([
          'status' => 'completed',
          'existing' => TRUE,
          'accessibility' => $accessibilityData,
        ]);
      }

      // Check if processing is enabled before queuing new tasks
      if (!$this->isProcessingEnabled()) {
        return new JsonResponse([
          'status' => 'disabled',
          'message' => 'PDF processing is currently disabled by administrator',
        ]);
      }

      // Queue accessibility scan if needed
      // Upload for accessibility check
      $accessibilityUpload = $this->pdfClient->uploadFile($file->getFileUri());
      if (!$accessibilityUpload) {
        return new JsonResponse(['error' => 'Accessibility upload failed'], 500);
      }

      // Start accessibility check
      $accessibilityOperation = $this->pdfClient->startOperation('accessibilitychecker', $accessibilityUpload['assetId']);
      if (!$accessibilityOperation) {
        return new JsonResponse(['error' => 'Failed to start accessibility check'], 500);
      }

      // Create accessibility status
      $accessibilityStatus = $this->entityTypeManager
        ->getStorage('pdf_processing_status')
        ->create([
          'fid' => $file->id(),
          'filename' => $file->getFilename(),
          'checksum' => $current_checksum,
          'state' => 'pending',
          'operation' => 'accessibilitychecker',
          'asset_id' => $accessibilityUpload['assetId'],
          'job_id' => $accessibilityOperation['jobId'],
          'callback_url' => $accessibilityOperation['statusUrl'],
          'created' => \Drupal::time()->getRequestTime(),
          'changed' => \Drupal::time()->getRequestTime(),
        ]);
      $accessibilityStatus->save();
      $accessibilityStatusId = $accessibilityStatus->id();

      // Return response with accessibility status ID
      $response = [
        'status' => 'processing',
        'message' => 'PDF accessibility check queued',
        'accessibilityStatusId' => $accessibilityStatusId,
      ];

      return new JsonResponse($response);
    }
    catch (\Exception $e) {
      \Drupal::logger('pdf_services')->error('Scan queueing failed: @error', [
        '@error' => $e->getMessage()
      ]);
      return new JsonResponse(['error' => 'Failed to queue scans: ' . $e->getMessage()], 500);
    }
  }

  public function startScans(Request $request) {
    $fid = $request->query->get('fid');

    if (!$fid) {
      return new JsonResponse(['error' => 'No file ID provided'], 400);
    }

    $file = $this->entityTypeManager->getStorage('file')->load($fid);

    if (!$file) {
      return new JsonResponse(['error' => 'File not found'], 404);
    }

    // Verify it's a PDF
    if ($file->getMimeType() !== 'application/pdf') {
      return new JsonResponse(['error' => 'Not a PDF file'], 400);
    }

    // Check if processing is enabled
    if (!$this->isProcessingEnabled()) {
      return new JsonResponse([
        'status' => 'disabled',
        'message' => 'PDF processing is currently disabled by administrator',
      ]);
    }

    // Upload file and get asset ID
    $uploadResult = $this->pdfClient->uploadFile($file->getFileUri());
    if (!$uploadResult) {
      return new JsonResponse(['error' => 'Failed to upload to Adobe API'], 500);
    }

    return new JsonResponse([
      'asset_id' => $uploadResult['assetId'],
      'message' => 'File uploaded successfully',
    ]);
  }

  public function checkAccessibility(Request $request) {
    $statusId = $request->query->get('status_id');

    // Load the processing status
    $status = $this->entityTypeManager
      ->getStorage('pdf_processing_status')
      ->load($statusId);

    if (!$status) {
      return new JsonResponse(['error' => 'Invalid status ID'], 404);
    }

    // If processing is disabled, inform the client
    if (!$this->isProcessingEnabled()) {
      // If status is already in processing state, put it back to pending
      if ($status->get('state')->value === 'processing') {
        $status->set('state', 'pending');
        $status->save();

        \Drupal::logger('pdf_services')->notice('PDF processing is disabled. Setting @filename back to pending state.', [
          '@filename' => $status->get('filename')->value,
        ]);
      }

      return new JsonResponse([
        'status' => 'disabled',
        'message' => 'PDF processing is currently disabled by administrator',
      ]);
    }

    // Check callback URL and job status
    $result = $this->pdfClient->checkOperationStatus($status->get('callback_url')->value);

    if ($result === FALSE) {
      return new JsonResponse(['error' => 'Operation failed'], 500);
    }

    if ($result === NULL || empty($result['report'])) {
      //Accessibility check still processing
      return new JsonResponse(['status' => 'processing']);
    }

    try {
      // Download and process the report if we have a download URI
      if (!empty($result['report']['downloadUri'])) {
        // Download report content
        $report_content = $this->pdfClient->downloadFile($result['report']['downloadUri']);
        $accessibility_report = json_decode($report_content, TRUE);

        if (json_last_error() !== JSON_ERROR_NONE) {
          throw new \Exception('Failed to parse accessibility report JSON');
        }

        $summary = $accessibility_report['Summary'] ?? [];
        $passes = FALSE;

        if (!empty($summary)) {
          $total_failed = ($summary['Failed'] ?? 0) + ($summary['Failed manually'] ?? 0);
          $needs_manual = $summary['Needs manual check'] ?? 0;
          $passes = ($total_failed === 0 && $needs_manual <= 2);
        }

        // Save results
        $storage = $this->entityTypeManager->getStorage('pdf_accessibility_result');
        $accessibility = $storage->create([
          'fid' => $status->get('fid')->value,
          'passes' => $passes,
          'report' => json_encode([
            'summary' => $summary,
            'detailed_report' => $accessibility_report['Detailed Report'] ?? [],
            'timestamp' => \Drupal::time()->getRequestTime(),
          ]),
          'created' => \Drupal::time()->getRequestTime(),
        ]);
        $accessibility->save();

        // Mark the processing status as complete
        $status->set('state', 'completed');
        $status->set('changed', \Drupal::time()->getRequestTime());
        $status->save();

        // Apply the same enhancements as the preprocess function
        /** @var \Drupal\pdf_services\Service\AccessibilityEnhancementService $enhancement_service */
        $enhancement_service = \Drupal::service('pdf_services.accessibility_enhancement');
        $processed_report = pdf_services_enhance_accessibility_report([
          'summary' => $summary,
          'detailed_report' => $accessibility_report['Detailed Report'] ?? [],
        ]);
        $processed_report = $enhancement_service->enhanceReport($processed_report);

        // Return the enhanced report
        return new JsonResponse([
          'status' => 'completed',
          'data' => [
            'passes' => $passes,
            'summary' => $processed_report['summary'],
            'details' => $processed_report['detailed_report'],
          ]
        ]);
      } else {
        \Drupal::logger('pdf_services')->error('No download URI in report result');
        return new JsonResponse(['error' => 'Invalid report data'], 500);
      }
    }
    catch (\Exception $e) {
      \Drupal::logger('pdf_services')->error('Failed to save accessibility results: @error', [
        '@error' => $e->getMessage(),
      ]);
      return new JsonResponse(['error' => 'Failed to save accessibility results'], 500);
    }
}

/**
 * Process the raw accessibility report into a structured format.
 */
protected function processAccessibilityReport($report) {
    // If it's a string (JSON), decode it
    if (is_string($report)) {
        $report = json_decode($report, TRUE);
    }

    $summary = [];
    $detailed_report = [];
    $passes = TRUE;

    if (!empty($report['Summary'])) {
        $summary = $report['Summary'];
        // Check if there are any failures
        $total_failed = ($summary['Failed'] ?? 0) + ($summary['Failed manually'] ?? 0);
        $needs_manual = $summary['Needs manual check'] ?? 0;
        $passes = ($total_failed === 0 && $needs_manual <= 2);
    }

    if (!empty($report['Detailed Report'])) {
        $detailed_report = $report['Detailed Report'];
    }

    return [
        'passes' => $passes,
        'summary' => $summary,
        'detailed_report' => $detailed_report
    ];
}

}
