<?php

namespace Drupal\pdf_services\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;

/**
 * Defines the PDF accessibility result entity.
 *
 * @ContentEntityType(
 *   id = "pdf_accessibility_result",
 *   label = @Translation("PDF Accessibility Result"),
 *   base_table = "pdf_accessibility_result",
 *   entity_keys = {
 *     "id" = "id",
 *     "uuid" = "uuid"
 *   },
 *   handlers = {
 *     "views_data" = "Drupal\pdf_services\PdfAccessibilityResultViewsData",
 *     "access" = "Drupal\Core\Entity\EntityAccessControlHandler",
 *     "storage" = "Drupal\Core\Entity\Sql\SqlContentEntityStorage",
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *   }
 * )
 */
class PdfAccessibilityResult extends ContentEntityBase {

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['fid'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('File ID'))
      ->setDescription(t('The ID of the processed file.'))
      ->setRequired(TRUE);

    $fields['passes'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Passes'))
      ->setDescription(t('Whether the PDF passes accessibility checks.'))
      ->setDefaultValue(FALSE)
      ->setRequired(TRUE);

    // Change from text_long to string_long to match the schema
    $fields['report'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Report'))
      ->setDescription(t('The full JSON report from the accessibility check.'))
      ->setRequired(TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the result was created.'));

    return $fields;
  }

  /**
   * Loads the most recent accessibility result for a file.
   *
   * @param int $fid
   *   The file ID to load results for.
   *
   * @return \Drupal\pdf_services\Entity\PdfAccessibilityResult|null
   *   The most recent accessibility result, or NULL if none exists.
   */
  public static function loadByFileId($fid) {
    $results = \Drupal::entityTypeManager()
      ->getStorage('pdf_accessibility_result')
      ->loadByProperties([
        'fid' => $fid,
      ]);

    if (empty($results)) {
      return NULL;
    }

    // Sort by created timestamp to get the most recent
    usort($results, function ($a, $b) {
      return $b->get('created')->value - $a->get('created')->value;
    });

    return reset($results);
  }
}
