<?php

namespace Drupal\pdf_services\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure enhanced descriptions for accessibility rules.
 */
class AccessibilityDescriptionsForm extends ConfigFormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a new AccessibilityDescriptionsForm.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    EntityTypeManagerInterface $entity_type_manager
  ) {
    parent::__construct($config_factory);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['pdf_services.enhanced_descriptions'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'pdf_services_enhanced_descriptions';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('pdf_services.enhanced_descriptions');
    $enhancements = $config->get('enhancements') ?: [];

    // Create a container for all enhancements
    $form['enhancements'] = [
      '#type' => 'details',
      '#title' => $this->t('Accessibility Rule Enhancements'),
      '#open' => TRUE,
      '#tree' => TRUE,
      '#prefix' => '<div id="accessibility-enhancements-wrapper">',
      '#suffix' => '</div>',
    ];

    // Get all possible rule titles
    $rule_titles = $this->getRuleTitles();

    // Get number of enhancements to show
    $enhancement_count = $form_state->get('enhancement_count');
    if ($enhancement_count === NULL) {
      $enhancement_count = count($enhancements) ? count($enhancements) : 1;
      $form_state->set('enhancement_count', $enhancement_count);
    }

    // Add enhancement fields
    for ($i = 0; $i < $enhancement_count; $i++) {
      $form['enhancements'][$i] = [
        '#type' => 'details',
        '#title' => $this->t('Enhancement @num', ['@num' => $i + 1]),
        '#open' => TRUE,
      ];

      $form['enhancements'][$i]['rule_title'] = [
        '#type' => 'select',
        '#title' => $this->t('Rule title'),
        '#options' => $rule_titles,
        '#default_value' => $enhancements[$i]['rule_title'] ?? '',
        '#description' => $this->t('The title of the rule to enhance'),
        '#required' => TRUE,
      ];

      $form['enhancements'][$i]['enhanced_description'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Enhanced description'),
        '#default_value' => $enhancements[$i]['enhanced_description'] ?? '',
        '#description' => $this->t('Additional text or links to append to the original description. You can use HTML.'),
        '#required' => TRUE,
        '#rows' => 4,
      ];
    }

    // Add/Remove buttons
    $form['actions_container'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['container-inline']],
    ];

    $form['actions_container']['add_enhancement'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add another'),
      '#submit' => ['::addEnhancement'],
      '#ajax' => [
        'callback' => '::updateEnhancements',
        'wrapper' => 'accessibility-enhancements-wrapper',
      ],
    ];

    if ($enhancement_count > 1) {
      $form['actions_container']['remove_enhancement'] = [
        '#type' => 'submit',
        '#value' => $this->t('Remove last'),
        '#submit' => ['::removeEnhancement'],
        '#ajax' => [
          'callback' => '::updateEnhancements',
          'wrapper' => 'accessibility-enhancements-wrapper',
        ],
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * AJAX callback for updating enhancements.
   */
  public function updateEnhancements(array &$form, FormStateInterface $form_state) {
    return $form['enhancements'];
  }

  /**
   * Submit handler for adding an enhancement.
   */
  public function addEnhancement(array &$form, FormStateInterface $form_state) {
    $count = $form_state->get('enhancement_count');
    $form_state->set('enhancement_count', $count + 1);
    $form_state->setRebuild();
  }

  /**
   * Submit handler for removing an enhancement.
   */
  public function removeEnhancement(array &$form, FormStateInterface $form_state) {
    $count = $form_state->get('enhancement_count');
    if ($count > 1) {
      $form_state->set('enhancement_count', $count - 1);
    }
    $form_state->setRebuild();
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValues();
    $enhancements = array_values(array_filter($values['enhancements'], function ($item) {
      return !empty($item['rule_title']) && !empty($item['enhanced_description']);
    }));

    $this->config('pdf_services.enhanced_descriptions')
      ->set('enhancements', $enhancements)
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Gets available rule titles from the accessibility report.
   *
   * @return array
   *   An array of rule titles.
   */
  protected function getRuleTitles() {
    // Build rule titles from the accessibility report structure
    $titles = [
      'Accessibility permission flag' => 'Accessibility permission flag',
      'Image-only PDF' => 'Image-only PDF',
      'Tagged PDF' => 'Tagged PDF',
      'Logical Reading Order' => 'Logical Reading Order',
      'Primary language' => 'Primary language',
      'Title' => 'Title',
      'Bookmarks' => 'Bookmarks',
      'Color contrast' => 'Color contrast',
      'Tagged content' => 'Tagged content',
      'Tagged annotations' => 'Tagged annotations',
      'Tab order' => 'Tab order',
      'Character encoding' => 'Character encoding',
      'Tagged multimedia' => 'Tagged multimedia',
      'Screen flicker' => 'Screen flicker',
      'Scripts' => 'Scripts',
      'Timed responses' => 'Timed responses',
      'Navigation links' => 'Navigation links',
      'Tagged form fields' => 'Tagged form fields',
      'Field descriptions' => 'Field descriptions',
      'Figures alternate text' => 'Figures alternate text',
      'Nested alternate text' => 'Nested alternate text',
      'Associated with content' => 'Associated with content',
      'Hides annotation' => 'Hides annotation',
      'Other elements alternate text' => 'Other elements alternate text',
      'Rows' => 'Rows',
      'TH and TD' => 'TH and TD',
      'Headers' => 'Headers',
      'Regularity' => 'Regularity',
      'Summary' => 'Summary',
      'List items' => 'List items',
      'Lbl and LBody' => 'Lbl and LBody',
      'Appropriate nesting' => 'Appropriate nesting',
    ];

    return $titles;
  }
}
