<?php

namespace Drupal\pdf_services\Plugin\Action;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Action\ConfigurableActionBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\file\FileInterface;
use Drupal\pdf_services\Service\PdfServicesManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Queue PDF files for accessibility analysis.
 *
 * @Action(
 *   id = "pdf_services_queue_accessibility",
 *   label = @Translation("Queue PDF Accessibility Check"),
 *   type = "file",
 *   confirm = TRUE,
 * )
 */
class QueuePdfAccessibilityCheck extends ConfigurableActionBase implements ContainerFactoryPluginInterface {

  /**
   * The PDF services manager.
   *
   * @var \Drupal\pdf_services\Service\PdfServicesManager
   */
  protected $pdfServicesManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    PdfServicesManager $pdf_services_manager,
    EntityTypeManagerInterface $entity_type_manager
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->pdfServicesManager = $pdf_services_manager;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('pdf_services.manager'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'check_properties' => TRUE,
      'check_accessibility' => TRUE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['check_properties'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Analyze PDF Properties'),
      '#description' => $this->t('Required for accessibility check to determine if PDF is tagged.'),
      '#default_value' => $this->configuration['check_properties'],
      '#disabled' => TRUE,
      '#value' => TRUE,
    ];

    $form['check_accessibility'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Check Accessibility'),
      '#description' => $this->t('Analyze PDF/UA compliance and accessibility features.'),
      '#default_value' => $this->configuration['check_accessibility'],
      '#disabled' => TRUE,
      '#value' => TRUE,
    ];

    $form['info'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('This action will queue the selected PDF files for accessibility analysis. Only tagged PDF files will be processed for accessibility. PDF properties will be checked first to determine if the file is tagged.') . '</p>',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->configuration['check_properties'] = $form_state->getValue('check_properties');
    $this->configuration['check_accessibility'] = $form_state->getValue('check_accessibility');
  }

  /**
   * {@inheritdoc}
   */
  public function execute($entity = NULL) {
    // Only process file entities.
    if (!$entity instanceof FileInterface) {
      return;
    }

    // Only process PDF files.
    $mime_type = $entity->getMimeType();
    if ($mime_type !== 'application/pdf') {
      \Drupal::messenger()->addWarning($this->t('Skipped non-PDF file: @filename', [
        '@filename' => $entity->getFilename(),
      ]));
      return;
    }

    // Check if processing is enabled.
    $config = \Drupal::config('pdf_services.settings');
    $processing_enabled = $config->get('processing_enabled') ?? TRUE;

    if (!$processing_enabled) {
      \Drupal::messenger()->addError($this->t('PDF processing is currently disabled.'));
      return;
    }

    // Prepare field settings.
    $field_settings = [
      'check_properties' => TRUE,
      'check_accessibility' => TRUE,
      'compression_level' => 'NONE',
      'page_size_threshold' => 0,
    ];

    // Create processing status.
    $status_id = $this->pdfServicesManager->createProcessingStatus($entity, $field_settings);

    if ($status_id) {
      // \Drupal::messenger()->addStatus($this->t('Queued @filename for PDF accessibility check.', [
      //   '@filename' => $entity->getFilename(),
      // ]));
    }
    else {
      \Drupal::messenger()->addWarning($this->t('File @filename could not be queued for processing. It may already be queued, or the file may not be accessible.', [
        '@filename' => $entity->getFilename(),
      ]));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, AccountInterface $account = NULL, $return_as_object = FALSE) {
    /** @var \Drupal\file\FileInterface $object */
    // Check if user has permission for bulk operations.
    $access = $account->hasPermission('administer pdf services')
      || $account->hasPermission('bulk process pdf files');

    return $return_as_object ? AccessResult::allowedIf($access) : $access;
  }

}
