<?php

namespace Drupal\pdf_services\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Database\Connection;
use Drupal\file\FileInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Render\RenderContext;

/**
 * Service for sending PDF accessibility notification emails.
 */
class PdfAccessibilityNotificationService {
  use StringTranslationTrait;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The mail manager.
   *
   * @var \Drupal\Core\Mail\MailManagerInterface
   */
  protected MailManagerInterface $mailManager;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected RendererInterface $renderer;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;


  /**
   * Constructs a new PdfAccessibilityNotificationService.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Mail\MailManagerInterface $mail_manager
   *   The mail manager.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    EntityTypeManagerInterface $entity_type_manager,
    MailManagerInterface $mail_manager,
    RendererInterface $renderer,
    LoggerChannelFactoryInterface $logger_factory,
    Connection $database
  ) {
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->mailManager = $mail_manager;
    $this->renderer = $renderer;
    $this->logger = $logger_factory->get('pdf_services');
    $this->database = $database;
  }

  /**
   * Sends a single accessibility issue notification email to all previous editors.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file with accessibility issues.
   * @param array $report
   *   The accessibility report data.
   *
   * @return bool
   *   TRUE if email was sent successfully, FALSE otherwise.
   */
  public function sendNotification(FileInterface $file, array $report) {
    try {
      // Check if notifications are enabled
      $config = $this->configFactory->get('pdf_services.settings');
      if (!$config->get('email_notifications.enabled')) {
        return FALSE;
      }

      // Find the entity referencing this file
      $query = $this->database->select('file_usage', 'fu');
      $query->fields('fu', ['type', 'id'])
        ->condition('fid', $file->id())
        ->range(0, 1);

      $usage = $query->execute()->fetchAssoc();

      if (!$usage || $usage['type'] !== 'node') {
        $this->logger->warning('No nodes found referencing file @fid', [
          '@fid' => $file->id(),
        ]);
        return FALSE;
      }

      // Load the node
      $node = $this->entityTypeManager->getStorage('node')->load($usage['id']);

      if (!$node) {
        $this->logger->warning('Could not load node @nid referencing file @fid', [
          '@nid' => $usage['id'],
          '@fid' => $file->id(),
        ]);
        return FALSE;
      }

      // Get all revision UIDs for this node
      $revision_storage = $this->entityTypeManager->getStorage('node')->getQuery()
        ->accessCheck(FALSE)
        ->allRevisions()
        ->condition('nid', $node->id())
        ->execute();

      if (empty($revision_storage)) {
        $this->logger->warning('No revisions found for node @nid', [
          '@nid' => $node->id(),
        ]);
        return FALSE;
      }

      // Extract all revision IDs and load revision data
      $revision_ids = array_keys($revision_storage);
      $revisions = $this->entityTypeManager->getStorage('node')->loadMultipleRevisions($revision_ids);

      // Collect unique user IDs from revisions
      $editor_uids = [];
      foreach ($revisions as $revision) {
        $editor_uid = $revision->getRevisionUserId();
        if ($editor_uid && !in_array($editor_uid, $editor_uids)) {
          $editor_uids[] = $editor_uid;
        }
      }

      // Also include node owner if not already in the list
      $owner_uid = $node->getOwnerId();
      if ($owner_uid && !in_array($owner_uid, $editor_uids)) {
        $editor_uids[] = $owner_uid;
      }

      if (empty($editor_uids)) {
        $this->logger->warning('No editors found for node @nid', [
          '@nid' => $node->id(),
        ]);
        return FALSE;
      }

      // Load all user accounts
      $users = $this->entityTypeManager->getStorage('user')->loadMultiple($editor_uids);

      // Filter out users without email or blocked users
      $recipients = [];
      $recipient_emails = [];
      $recipient_names = [];

      foreach ($users as $user) {
        if ($user->isActive() && $user->getEmail()) {
          $recipients[$user->id()] = $user;
          $recipient_emails[] = $user->getEmail();
          $recipient_names[] = $user->getDisplayName() . ' (' . $user->getEmail() . ')';
        }
      }

      if (empty($recipients)) {
        $this->logger->warning('No active recipients found for node @nid accessibility notification', [
          '@nid' => $node->id(),
        ]);

        // Try fallback to admin if configured
        $fallback_uid = $config->get('notification_fallback_uid');
        if ($fallback_uid) {
          $fallback_user = $this->entityTypeManager->getStorage('user')->load($fallback_uid);
          if ($fallback_user && $fallback_user->isActive() && $fallback_user->getEmail()) {
            $recipients[$fallback_user->id()] = $fallback_user;
            $recipient_emails[] = $fallback_user->getEmail();
            $recipient_names[] = $fallback_user->getDisplayName() . ' (' . $fallback_user->getEmail() . ')';
          }
        }

        if (empty($recipients)) {
          return FALSE;
        }
      }

      // Build email params
      $params = [
        'node_title' => $node->label(),
        'node_url' => $node->toUrl('canonical', ['absolute' => TRUE])->toString(),
        'filename' => $file->getFilename(),
        'file_url' => $file->createFileUrl(FALSE),
        'summary' => $this->formatAccessibilitySummary($report),
        'editor_count' => count($recipients),
        'editors' => $recipient_names,
      ];

      if ($config->get('email_notifications.include_report')) {
        $params['full_report'] = $this->formatFullReport($report);
      }

      // Choose language based on site default
      $language = \Drupal::languageManager()->getDefaultLanguage()->getId();

      // Send a single email to all recipients
      $result = $this->mailManager->mail(
        'pdf_services',
        'accessibility_issues',
        implode(', ', $recipient_emails),  // All recipients in To: field
        $language,
        $params,
        NULL,
        TRUE  // Send as HTML
      );

      if (!empty($result['result'])) {
        $this->logger->notice('Sent accessibility notification email to @count recipients for file @filename (Node: @title)', [
          '@count' => count($recipient_emails),
          '@filename' => $file->getFilename(),
          '@title' => $node->label(),
        ]);
        return TRUE;
      }

      return FALSE;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to send accessibility notification: @error', [
        '@error' => $e->getMessage(),
        'trace' => $e->getTraceAsString(),
      ]);
      return FALSE;
    }
  }

  /**
   * Formats the accessibility summary for email.
   */
  protected function formatAccessibilitySummary(array $report) {
    $summary = $report['summary'] ?? [];
    $build = [
      '#theme' => 'pdf_accessibility_email_summary',
      '#summary' => $summary,
    ];

    // Create a render context to safely render the content
    $render_context = new RenderContext();
    $rendered_content = $this->renderer->executeInRenderContext($render_context, function () use ($build) {
      return $this->renderer->render($build);
    });

    return $rendered_content;
  }

  /**
   * Formats the full accessibility report for email.
   */
  protected function formatFullReport(array $report) {
    $build = [
      '#theme' => 'pdf_accessibility_email_report',
      '#report' => $report,
    ];

    // Create a render context to safely render the content
    $render_context = new RenderContext();
    $rendered_content = $this->renderer->executeInRenderContext($render_context, function () use ($build) {
      return $this->renderer->render($build);
    });

    return $rendered_content;
  }

}
