<?php

namespace Drupal\pdf_services\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\file\FileInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Database\Connection;

/**
 * Service for managing PDF processing services and their execution chain.
 */
class PdfServicesManager {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The queue factory.
   *
   * @var \Drupal\Core\Queue\QueueFactory
   */
  protected $queueFactory;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Constructs a new PdfServicesManager.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Queue\QueueFactory $queue_factory
   *   The queue factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger channel factory.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    EntityTypeManagerInterface $entity_type_manager,
    QueueFactory $queue_factory,
    LoggerChannelFactoryInterface $logger_factory,
    FileSystemInterface $file_system,
    Connection $database
  ) {
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->queueFactory = $queue_factory;
    $this->loggerFactory = $logger_factory->get('pdf_services');
    $this->fileSystem = $file_system;
    $this->database = $database;
  }
  /**
 * Creates a processing status for a PDF file.
 *
 * @param \Drupal\file\FileInterface $file
 *   The file to process.
 * @param array $field_settings
 *   The field-specific settings.
 *
 * @return int|false
 *   Status entity ID if successful, FALSE otherwise.
 */
public function createProcessingStatus(FileInterface $file, array $field_settings = []) {
  try {
    // Calculate file checksum
    $checksum = $this->getFileChecksum($file);

    // If checksum calculation fails, log error and return FALSE
    if ($checksum === FALSE) {
      \Drupal::logger('pdf_services')->error('Cannot queue file for processing - failed to calculate checksum: @filename', [
        '@filename' => $file->getFilename(),
      ]);
      return FALSE;
    }

    // Check if file is already being processed
    $existing = $this->entityTypeManager
      ->getStorage('pdf_processing_status')
      ->loadByProperties([
        'fid' => $file->id(),
        'checksum' => $checksum,
        'state' => ['pending', 'processing'],
        'operation' => 'pdfproperties',
      ]);

    if (!empty($existing)) {
       \Drupal::logger('pdf_services')->notice('File already has pending processing status: @filename', [
        '@filename' => $file->getFilename(),
      ]);
      return FALSE;
    }

     // JSON encode the settings for storage
     $encoded_settings = !empty($field_settings) ? json_encode($field_settings) : '';

    // Create new processing status entity
    $status = $this->entityTypeManager
      ->getStorage('pdf_processing_status')
      ->create([
        'fid' => $file->id(),
        'filename' => $file->getFilename(),
        'checksum' => $checksum,
        'state' => 'pending',
        'operation' => 'pdfproperties', // Always start with properties
        'created' => \Drupal::time()->getRequestTime(),
        'changed' => \Drupal::time()->getRequestTime(),
        'field_settings' => $encoded_settings,
      ]);

    $status->save();
    return $status->id();
  }
  catch (\Exception $e) {
     \Drupal::logger('pdf_services')->error('Failed to create processing status: @error', [
      '@error' => $e->getMessage(),
    ]);
    return FALSE;
  }
}

  /**
   * Records a completed service.
   *
   * @param \Drupal\pdf_services\Entity\PdfProcessingStatus $status
   *   The processing status entity.
   * @param string $service_id
   *   The completed service ID.
   * @param array $result
   *   The service result data.
   */
  public function recordServiceCompletion($status, $service_id, array $result = []) {
    $status->set('state', 'completed');
    // The changed field will be automatically updated when we save
    $status->save();

    // Check if we need to clean up the asset
    if ($status->get('asset_id')->value) {
      // Determine if this is the last service to run for this asset
      $pendingStatuses = $this->entityTypeManager
        ->getStorage('pdf_processing_status')
        ->loadByProperties([
          'asset_id' => $status->get('asset_id')->value,
          'state' => ['pending', 'processing'],
        ]);

      // If no more pending operations for this asset, delete it
      if (empty($pendingStatuses)) {
        $client = \Drupal::service('pdf_services.client');
        $assetId = $status->get('asset_id')->value;

        $this->loggerFactory->notice('All operations complete for @file, deleting asset @id', [
          '@file' => $status->get('filename')->value,
          '@id' => $assetId,
        ]);

        $client->deleteAsset($assetId);
      }
    }
  }

  /**
   * Records a service failure.
   *
   * @param \Drupal\pdf_services\Entity\PdfProcessingStatus $status
   *   The processing status entity.
   * @param string $service_id
   *   The failed service ID.
   * @param string $error
   *   The error message.
   */
  public function recordServiceFailure($status, $service_id, $error) {
    $this->loggerFactory->error('Service marked as failed: @service - @error', [
      '@service' => $service_id,
      '@error' => $error,
    ]);

    $status->set('state', 'failed');
    $status->save();
  }

  /**
   * Safely gets a checksum for any type of Drupal file stream.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file to get the checksum for.
   *
   * @return string|false
   *   The MD5 checksum of the file or FALSE on failure.
   */
  protected function getFileChecksum(FileInterface $file) {
    $uri = $file->getFileUri();
    $realpath = $this->fileSystem->realpath($uri);

    if (!$realpath) {
      // For streams that don't support realpath, try to read the file directly
      if ($handle = fopen($uri, 'rb')) {
        $ctx = hash_init('md5');
        while (!feof($handle)) {
          hash_update($ctx, fread($handle, 8192));
        }
        fclose($handle);
        return hash_final($ctx);
      }
      return FALSE;
    }

    return md5_file($realpath);
  }

  /**
   * Checks if a file needs processing.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file to check.
   *
   * @return bool
   *   TRUE if file needs processing, FALSE otherwise.
   */
  public function needsProcessing(FileInterface $file) {
    try {
      // Check if it's a PDF
      if ($file->getMimeType() !== 'application/pdf') {
        return FALSE;
      }

      // Skip processing for files with the skip flag
      if (!empty($file->pdf_services_skip_processing)) {
        return FALSE;
      }

      $current_checksum = $this->getFileChecksum($file);
      if (!$current_checksum) {
         \Drupal::logger('pdf_services')->error('Failed to calculate checksum for file: @filename', [
          '@filename' => $file->getFilename(),
        ]);
        return FALSE;
      }

      // Check for recent processing status with this checksum
      $status_storage = $this->entityTypeManager->getStorage('pdf_processing_status');
      $existing_statuses = $status_storage->getQuery()
        ->condition('fid', $file->id())
        ->condition('checksum', $current_checksum)
        ->condition('operation', 'pdfproperties')
        ->condition('created', \Drupal::time()->getRequestTime() - 3600, '>')  // Within last hour
        ->accessCheck(FALSE)
        ->execute();

      if (!empty($existing_statuses)) {
         \Drupal::logger('pdf_services')->info('Skipping processing for @filename (recently processed with same checksum)', [
          '@filename' => $file->getFilename(),
        ]);
        return FALSE;
      }

      // Check if we already have a completed analysis for this checksum
      $analysis_storage = $this->entityTypeManager->getStorage('pdf_analysis_result');
      $query = $analysis_storage->getQuery()
        ->condition('fid', $file->id())
        ->condition('timestamp', \Drupal::time()->getRequestTime() - 3600, '>')  // Within last hour
        ->accessCheck(FALSE);

      $analyses = $analysis_storage->loadMultiple($query->execute());

      foreach ($analyses as $analysis) {
        $results = json_decode($analysis->get('results')->value, TRUE);
        if (isset($results['document']['checksum']) && $results['document']['checksum'] === $current_checksum) {
           \Drupal::logger('pdf_services')->info('Skipping processing for @filename (recent analysis with same checksum exists)', [
            '@filename' => $file->getFilename(),
          ]);
          return FALSE;
        }
      }

      return TRUE;
    }
    catch (\Exception $e) {
       \Drupal::logger('pdf_services')->error('Error checking if file needs processing: @error', [
        '@error' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Gets the field settings for a file.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file to check.
   *
   * @return array|null
   *   Field settings if found, NULL otherwise.
   */
  public function getFieldSettingsForFile(FileInterface $file) {
    // Check file usage to find referencing entities
    try {
      // Get all usages of this file
      $usages = $this->database->select('file_usage', 'fu')
        ->fields('fu', ['type', 'id', 'module'])
        ->condition('fid', $file->id())
        ->execute()
        ->fetchAll();

      foreach ($usages as $usage) {
        // Skip non-field module usages
        if ($usage->module !== 'file') {
          continue;
        }

        // Load the entity
        $entity = $this->entityTypeManager
          ->getStorage($usage->type)
          ->load($usage->id);

        if ($entity) {
          // Check all fields to find which one references this file
          foreach ($entity->getFields() as $field_name => $field) {
            $field_definition = $field->getFieldDefinition();

            // Only check file fields
            if ($field_definition->getType() !== 'file') {
              continue;
            }

            // Check if this field contains our file
            foreach ($field as $item) {
              if ($item->target_id == $file->id()) {
                // This is our field, get its settings
                if ($field_definition instanceof \Drupal\field\FieldConfigInterface) {
                  $settings = $field_definition->getThirdPartySettings('pdf_services');
                  if (!empty($settings)) {
                     \Drupal::logger('pdf_services')->debug('Found PDF settings for file @fid in field @field: @settings', [
                      '@fid' => $file->id(),
                      '@field' => $field_name,
                      '@settings' => json_encode($settings),
                    ]);
                    return $settings;
                  }
                }
              }
            }
          }
        }
      }
    }
    catch (\Exception $e) {
       \Drupal::logger('pdf_services')->error('Failed to get field settings for file: @error', [
        '@error' => $e->getMessage(),
      ]);
    }

    // Return null if no field settings found
    return NULL;
  }

}
