<?php

namespace Drupal\Tests\pdf_services\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\field\Entity\FieldConfig;

/**
 * Tests basic functionality of the PDF Services module.
 *
 * @group pdf_services
 */
class PdfServicesBasicTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'file',
    'field',
    'field_ui',
    'system',
    'user',
    'pdf_services',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * A user with permission to administer PDF services.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create the article content type since it's not available by default in tests.
    $this->drupalCreateContentType(['type' => 'article', 'name' => 'Article']);

    // Create admin user
    $this->adminUser = $this->drupalCreateUser([
      'administer pdf services',
      'administer content types',
      'administer node fields',
      'administer node display',
      'create article content',
      'edit any article content',
    ]);
  }

  /**
   * Tests the PDF Services settings form.
   */
  public function testSettingsForm() {
    $this->drupalLogin($this->adminUser);

    // Visit the settings page
    $this->drupalGet('/admin/config/content/pdf-services');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('PDF Services Settings');

    // Check for form elements
    $this->assertSession()->fieldExists('processing_enabled');
    $this->assertSession()->fieldExists('client_id');
    $this->assertSession()->fieldExists('client_secret');
    $this->assertSession()->fieldExists('batch_size');
    $this->assertSession()->fieldExists('retry_limit');

    // Test form submission
    $edit = [
      'processing_enabled' => TRUE,
      'client_id' => 'test_client_id',
      'client_secret' => 'test_client_secret',
      'batch_size' => 10,
      'retry_limit' => 5,
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Verify the values were saved
    $config = $this->config('pdf_services.settings');
    $this->assertEquals('test_client_id', $config->get('client_id'));
    $this->assertEquals('test_client_secret', $config->get('client_secret'));
    $this->assertEquals(10, $config->get('batch_size'));
    $this->assertEquals(5, $config->get('retry_limit'));
  }

  /**
   * Tests PDF field configuration.
   */
  public function testPdfFieldConfiguration() {
    $this->drupalLogin($this->adminUser);

    // Create a file field
    $field_name = 'field_test_pdf';
    FieldStorageConfig::create([
      'field_name' => $field_name,
      'entity_type' => 'node',
      'type' => 'file',
      'settings' => [
        'file_extensions' => 'pdf',
      ],
    ])->save();

    FieldConfig::create([
      'field_name' => $field_name,
      'entity_type' => 'node',
      'bundle' => 'article',
      'label' => 'Test PDF Field',
    ])->save();

    // Visit field configuration
    $this->drupalGet('/admin/structure/types/manage/article/fields/node.article.' . $field_name);
    $this->assertSession()->statusCodeEquals(200);

    // Check for PDF Services settings
    $this->assertSession()->pageTextContains('PDF Services Settings');
    $this->assertSession()->fieldExists('pdf_services[analysis][check_properties]');
    $this->assertSession()->fieldExists('pdf_services[analysis][check_accessibility]');
    $this->assertSession()->fieldExists('pdf_services[optimization][compression_level]');
  }

  /**
   * Tests the queue status page.
   */
  public function testQueueStatusPage() {
    $this->drupalLogin($this->adminUser);

    // Visit the queue status page
    $this->drupalGet('/admin/config/content/pdf-services/queue');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('PDF Processing Queue');

    // Check for status summary
    $this->assertSession()->pageTextContains('Pending:');
    $this->assertSession()->pageTextContains('Processing:');
    $this->assertSession()->pageTextContains('Completed:');
    $this->assertSession()->pageTextContains('Failed:');
  }

  /**
   * Tests the Adobe PDF Embed formatter.
   */
  public function testAdobePdfEmbedFormatter() {
    $this->drupalLogin($this->adminUser);

    // Create a file field with PDF embed formatter
    $field_name = 'field_pdf_embed';
    FieldStorageConfig::create([
      'field_name' => $field_name,
      'entity_type' => 'node',
      'type' => 'file',
    ])->save();

    FieldConfig::create([
      'field_name' => $field_name,
      'entity_type' => 'node',
      'bundle' => 'article',
      'label' => 'PDF Embed Field',
    ])->save();

    // Configure the display to use Adobe PDF Embed formatter
    $display = \Drupal::entityTypeManager()
      ->getStorage('entity_view_display')
      ->load('node.article.default');

    if (!$display) {
      $display = \Drupal::entityTypeManager()
        ->getStorage('entity_view_display')
        ->create([
          'targetEntityType' => 'node',
          'bundle' => 'article',
          'mode' => 'default',
          'status' => TRUE,
        ]);
    }

    $display->setComponent($field_name, [
      'type' => 'adobe_pdf_embed',
      'settings' => [
        'embed_mode' => 'FULL_WINDOW',
        'show_download' => TRUE,
        'show_print' => TRUE,
      ],
    ])->save();

    // Create a node with a PDF file (we'll mock this since we can't upload in tests)
    $node = $this->drupalCreateNode([
      'type' => 'article',
      'title' => 'Test PDF Article',
    ]);

    // Visit the node page
    $this->drupalGet('/node/' . $node->id());
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Tests accessibility report page access.
   */
  public function testAccessibilityReportPage() {
    $this->drupalLogin($this->adminUser);

    // Visit the accessibility report page
    $this->drupalGet('/admin/config/content/pdf-services/accessibility');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('PDF Accessibility Report');

    // Check for filter form
    $this->assertSession()->fieldExists('status');
  }
}
