<?php

namespace Drupal\Tests\pdf_services\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\file\Entity\File;
use Drupal\pdf_services\Entity\PdfProcessingStatus;

/**
 * Tests the PDF Services Manager functionality.
 *
 * @group pdf_services
 */
class PdfServicesManagerTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'file',
    'user',
    'pdf_services',
  ];

  /**
   * The PDF services manager.
   *
   * @var \Drupal\pdf_services\Service\PdfServicesManager
   */
  protected $pdfServicesManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('file');
    $this->installEntitySchema('user');
    $this->installEntitySchema('pdf_processing_status');
    $this->installEntitySchema('pdf_analysis_result');
    $this->installEntitySchema('pdf_accessibility_result');
    $this->installSchema('file', ['file_usage']);

    $this->pdfServicesManager = \Drupal::service('pdf_services.manager');
  }

  /**
   * Tests needsProcessing method.
   */
  public function testNeedsProcessing() {
    // Create a test PDF file entity
    $file = File::create([
      'uri' => 'public://test.pdf',
      'filename' => 'test.pdf',
      'filemime' => 'application/pdf',
      'status' => 1,
    ]);
    $file->save();

    // Mock the file content
    $file_path = \Drupal::service('file_system')->realpath($file->getFileUri());
    file_put_contents($file_path, 'PDF test content');

    // Test that a new PDF needs processing
    $this->assertTrue($this->pdfServicesManager->needsProcessing($file));

    // Create a non-PDF file
    $non_pdf = File::create([
      'uri' => 'public://test.txt',
      'filename' => 'test.txt',
      'filemime' => 'text/plain',
      'status' => 1,
    ]);
    $non_pdf->save();

    // Test that non-PDF files don't need processing
    $this->assertFalse($this->pdfServicesManager->needsProcessing($non_pdf));
  }

  /**
   * Tests createProcessingStatus method.
   */
  public function testCreateProcessingStatus() {
    // Create a test PDF file
    $file = File::create([
      'uri' => 'public://test.pdf',
      'filename' => 'test.pdf',
      'filemime' => 'application/pdf',
      'status' => 1,
    ]);
    $file->save();

    // Mock the file content
    $file_path = \Drupal::service('file_system')->realpath($file->getFileUri());
    file_put_contents($file_path, 'PDF test content');

    // Test creating a processing status
    $field_settings = [
      'check_properties' => TRUE,
      'check_accessibility' => TRUE,
      'compression_level' => 'MEDIUM',
    ];

    $status_id = $this->pdfServicesManager->createProcessingStatus($file, $field_settings);
    $this->assertNotFalse($status_id);

    // Load and verify the status entity
    $status = PdfProcessingStatus::load($status_id);
    $this->assertNotNull($status);
    $this->assertEquals($file->id(), $status->get('fid')->value);
    $this->assertEquals('pending', $status->get('state')->value);
    $this->assertEquals('pdfproperties', $status->get('operation')->value);

    // Verify field settings were saved
    $saved_settings = json_decode($status->get('field_settings')->value, TRUE);
    $this->assertEquals($field_settings, $saved_settings);
  }

  /**
   * Tests that duplicate processing statuses are not created.
   */
  public function testPreventDuplicateProcessing() {
    // Create a test PDF file
    $file = File::create([
      'uri' => 'public://test.pdf',
      'filename' => 'test.pdf',
      'filemime' => 'application/pdf',
      'status' => 1,
    ]);
    $file->save();

    // Mock the file content
    $file_path = \Drupal::service('file_system')->realpath($file->getFileUri());
    file_put_contents($file_path, 'PDF test content');

    // Create first processing status
    $status_id1 = $this->pdfServicesManager->createProcessingStatus($file);
    $this->assertNotFalse($status_id1);

    // Try to create another - should return FALSE
    $status_id2 = $this->pdfServicesManager->createProcessingStatus($file);
    $this->assertFalse($status_id2);
  }
}
