<?php

namespace Drupal\Tests\pdf_services\Unit;

use Drupal\Tests\UnitTestCase;
use Drupal\pdf_services\Service\AccessibilityEnhancementService;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;

/**
 * Tests the accessibility enhancement service.
 *
 * @group pdf_services
 * @coversDefaultClass \Drupal\pdf_services\Service\AccessibilityEnhancementService
 */
class AccessibilityEnhancementServiceTest extends UnitTestCase {

  /**
   * The accessibility enhancement service.
   *
   * @var \Drupal\pdf_services\Service\AccessibilityEnhancementService
   */
  protected $enhancementService;

  /**
   * The mocked config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $configFactory;

  /**
   * The mocked config.
   *
   * @var \Drupal\Core\Config\ImmutableConfig|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $config;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a mock configuration object
    $this->config = $this->createMock(ImmutableConfig::class);

    // Create a mock config factory that returns our mock config
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->configFactory->expects($this->any())
      ->method('get')
      ->with('pdf_services.enhanced_descriptions')
      ->willReturn($this->config);

    // Create the service with our mock dependencies
    $this->enhancementService = new AccessibilityEnhancementService(
      $this->configFactory
    );
  }

  /**
   * Tests that a report without enhancements is returned unchanged.
   *
   * @covers ::enhanceReport
   */
  public function testEnhanceReportNoEnhancements() {
    // Configure the mock config to return empty enhancements
    $this->config->expects($this->once())
      ->method('get')
      ->with('enhancements')
      ->willReturn([]);

    // Create a test report
    $report = [
      'detailed_report' => [
        'Section 1' => [
          [
            'Rule' => 'Test Rule',
            'Status' => 'Passed',
            'Description' => 'Original description',
          ],
        ],
      ],
    ];

    // Call the method
    $result = $this->enhancementService->enhanceReport($report);

    // Assert the report is unchanged
    $this->assertEquals($report, $result);
  }

  /**
   * Tests enhancement of a report with configured enhancements.
   *
   * @covers ::enhanceReport
   */
  public function testEnhanceReportWithEnhancements() {
    // Configure the mock config to return enhancements
    $enhancements = [
      [
        'rule_title' => 'Test Rule',
        'enhanced_description' => 'Enhanced test description',
      ],
    ];
    $this->config->expects($this->once())
      ->method('get')
      ->with('enhancements')
      ->willReturn($enhancements);

    // Create a test report
    $report = [
      'detailed_report' => [
        'Section 1' => [
          [
            'Rule' => 'Test Rule',
            'Status' => 'Passed',
            'Description' => 'Original description',
          ],
        ],
      ],
    ];

    // Call the method
    $result = $this->enhancementService->enhanceReport($report);

    // Assert the report has been enhanced
    $this->assertEquals('Original description', $result['detailed_report']['Section 1'][0]['OriginalDescription']);
    $this->assertEquals('Enhanced test description', $result['detailed_report']['Section 1'][0]['EnhancedDescription']);
  }

  /**
   * Tests enhancement of a report without a detailed report section.
   *
   * @covers ::enhanceReport
   */
  public function testEnhanceReportWithoutDetailedReport() {
    // Configure the mock config to return enhancements
    $enhancements = [
      [
        'rule_title' => 'Test Rule',
        'enhanced_description' => 'Enhanced test description',
      ],
    ];
    $this->config->expects($this->once())
      ->method('get')
      ->with('enhancements')
      ->willReturn($enhancements);

    // Create a test report without detailed_report
    $report = [
      'summary' => [
        'Passed' => 5,
        'Failed' => 2,
      ],
    ];

    // Call the method
    $result = $this->enhancementService->enhanceReport($report);

    // Assert the report is unchanged
    $this->assertEquals($report, $result);
  }

}
