<?php

namespace Drupal\pelias_field\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\pelias_field\Service\PeliasFieldInterface;
use Drupal\pelias_field\Service\PeliasResponseParserService;
use Drupal\search_api\Plugin\search_api\processor\Resources\Pe;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for Pelias autocomplete functionality.
 */
class PeliasAutocompleteController extends ControllerBase {

  /**
   * The Pelias geocoder service.
   *
   * @var \Drupal\pelias_field\Service\PeliasFieldInterface
   */
  protected $peliasFieldInterface;

  /**
   * The Pelias response parser service.
   *
   * @var \Drupal\pelias_field\Service\PeliasResponseParserService
   */
  protected $responseParser;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * Constructs a new PeliasAutocompleteController object.
   *
   * @param \Drupal\pelias_field\Service\PeliasFieldInterface $peliasFieldInterface
   *   The Pelias geocoder service.
   * @param \Drupal\pelias_field\Service\PeliasResponseParserService $response_parser
   *   The Pelias response parser service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager.
   */
  public function __construct(PeliasFieldInterface $peliasFieldInterface, PeliasResponseParserService $response_parser, EntityTypeManagerInterface $entity_type_manager, EntityFieldManagerInterface $entity_field_manager) {
    $this->peliasFieldInterface = $peliasFieldInterface;
    $this->responseParser = $response_parser;
    $this->entityTypeManager = $entity_type_manager;
    $this->entityFieldManager = $entity_field_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('pelias_field.pelias_field_service'),
      $container->get('pelias_field.response_parser'),
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager')
    );
  }

  /**
   * Handles autocomplete requests.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   * @param string $field_name
   *   The field name.
   * @param string $entity_type
   *   The entity type.
   * @param string $bundle
   *   The bundle name.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response containing the autocomplete suggestions.
   */
  public function autocomplete(Request $request, string $field_name, string $entity_type, string $bundle): JsonResponse {
    $results = [];

    // Get the search query from the request.
    $query = $request->query->get('q');
    if (empty($query) || strlen($query) < 2) {
      return new JsonResponse($results);
    }

    try {
      // Get field definition to access settings.
      $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_type, $bundle);
      
      if (!isset($field_definitions[$field_name])) {
        $this->getLogger('pelias_field')->error('Field @field_name not found for entity type @entity_type, bundle @bundle', [
          '@field_name' => $field_name,
          '@entity_type' => $entity_type,
          '@bundle' => $bundle,
        ]);
        return new JsonResponse($results);
      }

      $field_definition = $field_definitions[$field_name];
      $field_settings = $field_definition->getSettings();

      // Check minimum length requirement.
      $min_length = $field_settings['min_length'] ?? 2;
      if (strlen($query) < $min_length) {
        return new JsonResponse($results);
      }

      // Build options from field settings.
      $options = [
        'max_suggestions' => $field_settings['max_suggestions'] ?? 10,
      ];

      // Add focus point if configured.
      if (!empty($field_settings['focus_point']['lat']) && !empty($field_settings['focus_point']['lon'])) {
        $options['focus_point'] = [
          'lat' => (float) $field_settings['focus_point']['lat'],
          'lon' => (float) $field_settings['focus_point']['lon'],
        ];
      }

      // Add boundary country if configured.
      if (!empty($field_settings['boundary_country'])) {
        $options['boundary_country'] = $field_settings['boundary_country'];
      }

      // Add sources if configured.
      if (!empty($field_settings['sources'])) {
        $sources = array_filter($field_settings['sources']);
        if (!empty($sources)) {
          $options['sources'] = array_values($sources);
        }
      }

      // Add layers if configured.
      if (!empty($field_settings['layers'])) {
        $layers = array_filter($field_settings['layers']);
        if (!empty($layers)) {
          $options['layers'] = array_values($layers);
        }
      }

      // Perform the autocomplete search using your original method.
      $features = $this->peliasFieldInterface->autocomplete($query, $options);

      if (empty($features)) {
        return new JsonResponse($results);
      }

      // Parse the features and extract suggestions using your response parser.
      $parsed_fields = $field_settings['parsed_fields'] ?? [];
      $suggestions = $this->responseParser->extractAutocompleteSuggestions($features, $parsed_fields);

      // Format suggestions for the JavaScript autocomplete.
      foreach ($suggestions as $suggestion) {
        $results[] = [
          'value' => $suggestion['value'] ?? '',
          'label' => $suggestion['label'] ?? $suggestion['value'] ?? '',
          'raw_data' => json_encode($suggestion['data']['raw'] ?? [], JSON_UNESCAPED_UNICODE),
          'parsed_data' => json_encode($suggestion['data']['parsed'] ?? [], JSON_UNESCAPED_UNICODE),
        ];
      }

    }
    catch (\Exception $e) {
      $this->getLogger('pelias_field')->error('Error in autocomplete request: @error', [
        '@error' => $e->getMessage(),
      ]);
      
      // Return empty results on error, don't expose error details to frontend
      return new JsonResponse($results);
    }

    return new JsonResponse($results);
  }

}