<?php

namespace Drupal\pelias_field\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Defines the 'pelias_field_formatter' field formatter.
 *
 * @FieldFormatter(
 *   id = "pelias_field_formatter",
 *   label = @Translation("Pelias Field Formatter"),
 *   field_types = {"pelias_field"}
 * )
 */
class PeliasFieldFormatter extends FormatterBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'display_format' => 'label_only',
      'show_coordinates' => FALSE,
      'data_source' => '',
      'data_layer' => '',
      'custom_template' => '',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form['display_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Display format'),
      '#default_value' => $this->getSetting('display_format'),
      '#options' => [
        'label_only' => $this->t('Label only'),
        'label_with_context' => $this->t('Label with context'),
        'full_address' => $this->t('Full address'),
        'coordinates_only' => $this->t('Coordinates only'),
        'custom' => $this->t('Custom template'),
      ],
      '#required' => TRUE,
    ];

    $form['show_coordinates'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show coordinates'),
      '#default_value' => $this->getSetting('show_coordinates'),
      '#description' => $this->t('Whether to display latitude and longitude coordinates.'),
      '#states' => [
        'invisible' => [
          ':input[name*="display_format"]' => ['value' => 'coordinates_only'],
        ],
      ],
    ];

    $form['data_source'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Filter by data source'),
      '#default_value' => $this->getSetting('data_source'),
      '#description' => $this->t('Only display items from this data source (e.g., "osm", "gn"). Leave empty to show all sources.'),
    ];

    $form['data_layer'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Filter by data layer'),
      '#default_value' => $this->getSetting('data_layer'),
      '#description' => $this->t('Only display items from this data layer (e.g., "address", "venue"). Leave empty to show all layers.'),
    ];

    $form['custom_template'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Custom template'),
      '#default_value' => $this->getSetting('custom_template'),
      '#description' => $this->t('Custom template for displaying the location. Use tokens like [label], [name], [locality], [region], [country], [latitude], [longitude]. Example: "[name], [locality], [region] ([latitude], [longitude])".'),
      '#states' => [
        'visible' => [
          ':input[name*="display_format"]' => ['value' => 'custom'],
        ],
        'required' => [
          ':input[name*="display_format"]' => ['value' => 'custom'],
        ],
      ],
    ];

    $form['help'] = [
      '#type' => 'details',
      '#title' => $this->t('Available tokens'),
      '#description' => $this->t('The following tokens are available from the parsed data: [label], [name], [source], [layer], [country], [country_code], [region], [county], [locality], [neighbourhood], [street], [postalcode], [confidence], [latitude], [longitude]'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];

    $display_format_options = [
      'label_only' => $this->t('Label only'),
      'label_with_context' => $this->t('Label with context'),
      'full_address' => $this->t('Full address'),
      'coordinates_only' => $this->t('Coordinates only'),
      'custom' => $this->t('Custom template'),
    ];

    $summary[] = $this->t('Display format: @format', [
      '@format' => $display_format_options[$this->getSetting('display_format')] ?? $this->getSetting('display_format'),
    ]);

    if ($this->getSetting('show_coordinates')) {
      $summary[] = $this->t('Show coordinates: Yes');
    }

    if (!empty($this->getSetting('data_source'))) {
      $summary[] = $this->t('Source filter: @source', ['@source' => $this->getSetting('data_source')]);
    }

    if (!empty($this->getSetting('data_layer'))) {
      $summary[] = $this->t('Layer filter: @layer', ['@layer' => $this->getSetting('data_layer')]);
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];

    foreach ($items as $delta => $item) {
      if ($item->isEmpty()) {
        continue;
      }

      $parsed_data = $item->getParsedData();
      $raw_data = $item->getRawData();

      // Apply source and layer filters.
      if (!$this->matchesFilters($parsed_data)) {
        continue;
      }

      $display_value = $this->buildDisplayValue($parsed_data, $item->value);

      $elements[$delta] = [
        '#markup' => $display_value,
        '#cache' => [
          'contexts' => ['url.query_args'],
        ],
      ];
    }

    return $elements;
  }

  /**
   * Check if the item matches the configured filters.
   *
   * @param array $parsed_data
   *   The parsed data array.
   *
   * @return bool
   *   TRUE if the item matches the filters.
   */
  protected function matchesFilters(array $parsed_data): bool {
    $data_source = $this->getSetting('data_source');
    if (!empty($data_source) && ($parsed_data['source'] ?? '') !== $data_source) {
      return FALSE;
    }

    $data_layer = $this->getSetting('data_layer');
    if (!empty($data_layer) && ($parsed_data['layer'] ?? '') !== $data_layer) {
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Build the display value based on the configured format.
   *
   * @param array $parsed_data
   *   The parsed data array.
   * @param string $fallback_value
   *   The fallback display value.
   *
   * @return string
   *   The formatted display value.
   */
  protected function buildDisplayValue(array $parsed_data, string $fallback_value): string {
    $format = $this->getSetting('display_format');

    switch ($format) {
      case 'label_only':
        return $this->buildLabelOnly($parsed_data, $fallback_value);

      case 'label_with_context':
        return $this->buildLabelWithContext($parsed_data, $fallback_value);

      case 'full_address':
        return $this->buildFullAddress($parsed_data, $fallback_value);

      case 'coordinates_only':
        return $this->buildCoordinatesOnly($parsed_data);

      case 'custom':
        return $this->buildCustomTemplate($parsed_data, $fallback_value);

      default:
        return $fallback_value;
    }
  }

  /**
   * Build label-only display.
   */
  protected function buildLabelOnly(array $parsed_data, string $fallback_value): string {
    $value = $parsed_data['label'] ?? $parsed_data['name'] ?? $fallback_value;
    
    if ($this->getSetting('show_coordinates') && !empty($parsed_data['latitude']) && !empty($parsed_data['longitude'])) {
      $value .= ' (' . $parsed_data['latitude'] . ', ' . $parsed_data['longitude'] . ')';
    }

    return $value;
  }

  /**
   * Build label with context display.
   */
  protected function buildLabelWithContext(array $parsed_data, string $fallback_value): string {
    $parts = [];
    $parts[] = $parsed_data['label'] ?? $parsed_data['name'] ?? $fallback_value;

    $context_parts = [];
    if (!empty($parsed_data['locality'])) {
      $context_parts[] = $parsed_data['locality'];
    }
    if (!empty($parsed_data['region'])) {
      $context_parts[] = $parsed_data['region'];
    }
    if (!empty($parsed_data['country'])) {
      $context_parts[] = $parsed_data['country'];
    }

    if (!empty($context_parts)) {
      $parts[] = '(' . implode(', ', $context_parts) . ')';
    }

    $value = implode(' ', $parts);

    if ($this->getSetting('show_coordinates') && !empty($parsed_data['latitude']) && !empty($parsed_data['longitude'])) {
      $value .= ' [' . $parsed_data['latitude'] . ', ' . $parsed_data['longitude'] . ']';
    }

    return $value;
  }

  /**
   * Build full address display.
   */
  protected function buildFullAddress(array $parsed_data, string $fallback_value): string {
    $parts = [];

    // Build address components in logical order.
    if (!empty($parsed_data['name']) && $parsed_data['name'] !== ($parsed_data['street'] ?? '')) {
      $parts[] = $parsed_data['name'];
    }
    if (!empty($parsed_data['street'])) {
      $parts[] = $parsed_data['street'];
    }
    if (!empty($parsed_data['neighbourhood'])) {
      $parts[] = $parsed_data['neighbourhood'];
    }
    if (!empty($parsed_data['locality'])) {
      $parts[] = $parsed_data['locality'];
    }
    if (!empty($parsed_data['county'])) {
      $parts[] = $parsed_data['county'];
    }
    if (!empty($parsed_data['region'])) {
      $parts[] = $parsed_data['region'];
    }
    if (!empty($parsed_data['postalcode'])) {
      $parts[] = $parsed_data['postalcode'];
    }
    if (!empty($parsed_data['country'])) {
      $parts[] = $parsed_data['country'];
    }

    $value = !empty($parts) ? implode(', ', $parts) : $fallback_value;

    if ($this->getSetting('show_coordinates') && !empty($parsed_data['latitude']) && !empty($parsed_data['longitude'])) {
      $value .= ' (' . $parsed_data['latitude'] . ', ' . $parsed_data['longitude'] . ')';
    }

    return $value;
  }

  /**
   * Build coordinates-only display.
   */
  protected function buildCoordinatesOnly(array $parsed_data): string {
    if (!empty($parsed_data['latitude']) && !empty($parsed_data['longitude'])) {
      return $parsed_data['latitude'] . ', ' . $parsed_data['longitude'];
    }
    return '';
  }

  /**
   * Build custom template display.
   */
  protected function buildCustomTemplate(array $parsed_data, string $fallback_value): string {
    $template = $this->getSetting('custom_template');
    if (empty($template)) {
      return $fallback_value;
    }

    // Replace tokens with actual values.
    foreach ($parsed_data as $key => $value) {
      if (is_scalar($value)) {
        $template = str_replace('[' . $key . ']', (string) $value, $template);
      }
    }

    // Remove any remaining tokens that weren't replaced.
    $template = preg_replace('/\[[^\]]+\]/', '', $template);

    return trim($template) ?: $fallback_value;
  }

}