<?php

namespace Drupal\pelias_field\Plugin\Field\FieldType;

use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\TypedData\DataDefinition;

/**
 * Defines the 'pelias_field' field type.
 *
 * @FieldType(
 *   id = "pelias_field",
 *   label = @Translation("Pelias Field"),
 *   description = @Translation("A field for geocoding using Pelias API."),
 *   category = @Translation("Geographic"),
 *   default_widget = "pelias_field_autocomplete_widget",
 *   default_formatter = "pelias_field_formatter"
 * )
 */
class PeliasFieldType extends FieldItemBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultStorageSettings() {
    return [
      'max_length' => 255,
    ] + parent::defaultStorageSettings();
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultFieldSettings() {
    return [
      'max_suggestions' => 10,
      'min_length' => 2,
      'focus_point' => [
        'lat' => '',
        'lon' => '',
      ],
      'boundary_country' => '',
      'sources' => [],
      'layers' => [],
      'parsed_fields' => [
        'label' => 'properties.label',
        'latitude' => 'geometry.coordinates.1',
        'longitude' => 'geometry.coordinates.0',
      ],
    ] + parent::defaultFieldSettings();
  }

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition) {
    $properties['value'] = DataDefinition::create('string')
      ->setLabel(t('Display value'))
      ->setRequired(FALSE);

    $properties['raw_data'] = DataDefinition::create('string')
      ->setLabel(t('Raw JSON data'))
      ->setRequired(FALSE);

    $properties['parsed_data'] = DataDefinition::create('string')
      ->setLabel(t('Parsed data'))
      ->setRequired(FALSE);

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition) {
    $schema = [
      'columns' => [
        'value' => [
          'type' => 'varchar',
          'length' => (int) $field_definition->getSetting('max_length'),
          'not null' => FALSE,
        ],
        'raw_data' => [
          'type' => 'blob',
          'size' => 'big',
          'not null' => FALSE,
        ],
        'parsed_data' => [
          'type' => 'text',
          'size' => 'medium',
          'not null' => FALSE,
        ],
      ],
    ];

    return $schema;
  }

  /**
   * {@inheritdoc}
   */
  public function fieldSettingsForm(array $form, FormStateInterface $form_state) {
    $element = [];
    $config = \Drupal::config('pelias_field.settings');
    $available_sources = $config->get('sources') ?? [];
    $available_layers = $config->get('layers') ?? [];

    $element['max_suggestions'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum suggestions'),
      '#default_value' => $this->getSetting('max_suggestions'),
      '#required' => TRUE,
      '#min' => 1,
      '#max' => 50,
      '#description' => $this->t('Maximum number of autocomplete suggestions to display.'),
    ];

    $element['min_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum characters'),
      '#default_value' => $this->getSetting('min_length'),
      '#required' => TRUE,
      '#min' => 1,
      '#max' => 10,
      '#description' => $this->t('Minimum number of characters before triggering autocomplete.'),
    ];

    $element['focus_point'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Focus Point'),
      '#description' => $this->t('Bias results around a specific point (optional).'),
      '#collapsible' => TRUE,
      '#collapsed' => TRUE,
    ];

    $element['focus_point']['lat'] = [
      '#type' => 'number',
      '#title' => $this->t('Latitude'),
      '#default_value' => $this->getSetting('focus_point')['lat'] ?? '',
      '#step' => 'any',
      '#min' => -90,
      '#max' => 90,
    ];

    $element['focus_point']['lon'] = [
      '#type' => 'number',
      '#title' => $this->t('Longitude'),
      '#default_value' => $this->getSetting('focus_point')['lon'] ?? '',
      '#step' => 'any',
      '#min' => -180,
      '#max' => 180,
    ];

    $element['boundary_country'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Boundary Country'),
      '#default_value' => $this->getSetting('boundary_country'),
      '#maxlength' => 3,
      '#description' => $this->t('ISO country code to restrict results (e.g., "US", "GB", "IN").'),
    ];

    if (!empty($available_sources)) {
      $element['sources'] = [
        '#type' => 'checkboxes',
        '#title' => $this->t('Data Sources'),
        '#default_value' => $this->getSetting('sources'),
        '#options' => array_combine($available_sources, $available_sources),
        '#description' => $this->t('Select which data sources to query. Leave empty to use all available sources.'),
      ];
    }

    if (!empty($available_layers)) {
      $element['layers'] = [
        '#type' => 'checkboxes',
        '#title' => $this->t('Place Layers'),
        '#default_value' => $this->getSetting('layers'),
        '#options' => array_combine($available_layers, $available_layers),
        '#description' => $this->t('Select which types of places to search for. Leave empty to search all types.'),
      ];
    }

    // Simplified parsed fields configuration
    $element['parsed_fields_config'] = [
      '#type' => 'details',
      '#title' => $this->t('Parsed Field Mappings'),
      '#description' => $this->t('Configure which fields to extract from the raw JSON data for easier access.'),
      '#open' => FALSE,
    ];

    $parsed_fields = $this->getSetting('parsed_fields') ?? [];
    
    // Convert old array format to new simplified format if needed
    if (isset($parsed_fields[0]) && is_array($parsed_fields[0]) && isset($parsed_fields[0]['field_name'])) {
      $old_format = $parsed_fields;
      $parsed_fields = [];
      foreach ($old_format as $mapping) {
        if (!empty($mapping['field_name']) && !empty($mapping['json_path'])) {
          $parsed_fields[$mapping['field_name']] = $mapping['json_path'];
        }
      }
    }

    // Add some common fields for easy configuration
    $common_fields = [
      'label' => 'properties.label',
      'name' => 'properties.name',
      'locality' => 'properties.locality',
      'region' => 'properties.region',
      'country' => 'properties.country',
      'postalcode' => 'properties.postalcode',
      'latitude' => 'geometry.coordinates.1',
      'longitude' => 'geometry.coordinates.0',
    ];

    foreach ($common_fields as $field_name => $default_path) {
      $element['parsed_fields_config'][$field_name] = [
        '#type' => 'textfield',
        '#title' => $this->t('@field JSON Path', ['@field' => ucfirst($field_name)]),
        '#default_value' => $parsed_fields[$field_name] ?? $default_path,
        '#description' => $this->t('JSON path to extract @field data from the API response.', ['@field' => $field_name]),
        '#size' => 40,
      ];
    }

    // Custom fields textarea for additional mappings
    $custom_fields = [];
    foreach ($parsed_fields as $field_name => $path) {
      if (!isset($common_fields[$field_name])) {
        $custom_fields[] = $field_name . '|' . $path;
      }
    }

    $element['parsed_fields_config']['custom_fields'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Custom Field Mappings'),
      '#default_value' => implode("\n", $custom_fields),
      '#description' => $this->t('Additional field mappings, one per line. Format: field_name|json.path.to.value'),
      '#rows' => 5,
    ];

    $element['parsed_fields_config']['help'] = [
      '#type' => 'markup',
      '#markup' => '<p><strong>' . $this->t('Common JSON paths:') . '</strong></p>' .
        '<ul>' .
        '<li><code>properties.label</code> - Full display name</li>' .
        '<li><code>properties.name</code> - Place name</li>' .
        '<li><code>properties.locality</code> - City/town</li>' .
        '<li><code>properties.region</code> - State/province</li>' .
        '<li><code>properties.country</code> - Country name</li>' .
        '<li><code>properties.postalcode</code> - Postal/ZIP code</li>' .
        '<li><code>geometry.coordinates.0</code> - Longitude</li>' .
        '<li><code>geometry.coordinates.1</code> - Latitude</li>' .
        '</ul>',
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function fieldSettingsFormValidate(array $form, FormStateInterface $form_state) {
    // Process and validate the parsed fields configuration
    $values = $form_state->getValues();
    $parsed_fields = [];
    
    // Get common field mappings
    $common_fields = [
      'label', 'name', 'locality', 'region', 'country', 'postalcode', 'latitude', 'longitude'
    ];
    
    foreach ($common_fields as $field_name) {
      $path = $values['settings']['parsed_fields_config'][$field_name] ?? '';
      if (!empty($path)) {
        $parsed_fields[$field_name] = $path;
      }
    }
    
    // Process custom fields
    $custom_fields = $values['settings']['parsed_fields_config']['custom_fields'] ?? '';
    if (!empty($custom_fields)) {
      $lines = explode("\n", $custom_fields);
      foreach ($lines as $line) {
        $line = trim($line);
        if (!empty($line) && strpos($line, '|') !== FALSE) {
          list($field_name, $path) = explode('|', $line, 2);
          $field_name = trim($field_name);
          $path = trim($path);
          if (!empty($field_name) && !empty($path)) {
            $parsed_fields[$field_name] = $path;
          }
        }
      }
    }
    
    // Update the form state with processed parsed_fields
    $form_state->setValue(['settings', 'parsed_fields'], $parsed_fields);
  }

  /**
   * {@inheritdoc}
   */
  public function storageSettingsForm(array &$form, FormStateInterface $form_state, $has_data) {
    $form['max_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum length'),
      '#default_value' => $this->getSetting('max_length'),
      '#required' => TRUE,
      '#description' => $this->t('The maximum length of the display value in characters.'),
      '#min' => 1,
      '#disabled' => $has_data,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty() {
    $value = $this->get('value')->getValue();
    return $value === NULL || $value === '';
  }

  /**
   * {@inheritdoc}
   */
  public function setValue($values, $notify = TRUE) {
    // Ensure parsed_data is properly JSON encoded if it's an array.
    if (isset($values['parsed_data']) && is_array($values['parsed_data'])) {
      $values['parsed_data'] = json_encode($values['parsed_data'], JSON_UNESCAPED_UNICODE);
    }

    // Ensure raw_data is properly JSON encoded if it's an array.
    if (isset($values['raw_data']) && is_array($values['raw_data'])) {
      $values['raw_data'] = json_encode($values['raw_data'], JSON_UNESCAPED_UNICODE);
    }

    parent::setValue($values, $notify);
  }

  /**
   * Get the parsed data as an array.
   *
   * @return array
   *   The parsed data array.
   */
  public function getParsedData(): array {
    $parsed_data = $this->get('parsed_data')->getValue();
    if (empty($parsed_data)) {
      return [];
    }

    $decoded = json_decode($parsed_data, TRUE);
    return is_array($decoded) ? $decoded : [];
  }

  /**
   * Get the raw data as an array.
   *
   * @return array
   *   The raw data array.
   */
  public function getRawData(): array {
    $raw_data = $this->get('raw_data')->getValue();
    if (empty($raw_data)) {
      return [];
    }

    $decoded = json_decode($raw_data, TRUE);
    return is_array($decoded) ? $decoded : [];
  }

}