<?php

namespace Drupal\pelias_field\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for Pelias Field settings.
 */
class PeliasFieldSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['pelias_field.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'pelias_field_admin_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('pelias_field.settings');

    $form['api_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('API Settings'),
      '#description' => $this->t('Configure the Pelias API endpoint and authentication.'),
    ];

    $form['api_settings']['api_endpoint'] = [
      '#type' => 'url',
      '#title' => $this->t('API Endpoint'),
      '#default_value' => $config->get('api_endpoint'),
      '#required' => TRUE,
      '#description' => $this->t('The base URL for the Pelias API. For Geocode Earth, use: https://api.geocode.earth/v1. For self-hosted, use your server URL.'),
    ];

    $form['api_settings']['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Key'),
      '#default_value' => $config->get('api_key'),
      '#description' => $this->t('API key for authentication (required for Geocode Earth, optional for self-hosted).'),
    ];

    $form['performance_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Performance Settings'),
      '#description' => $this->t('Configure performance and rate limiting options.'),
    ];

    $form['performance_settings']['timeout'] = [
      '#type' => 'number',
      '#title' => $this->t('Request Timeout'),
      '#default_value' => $config->get('timeout'),
      '#required' => TRUE,
      '#min' => 1,
      '#max' => 120,
      '#field_suffix' => $this->t('seconds'),
      '#description' => $this->t('Timeout for HTTP requests to the Pelias API.'),
    ];

    $form['performance_settings']['debounce_delay'] = [
      '#type' => 'number',
      '#title' => $this->t('Debounce Delay'),
      '#default_value' => $config->get('debounce_delay'),
      '#required' => TRUE,
      '#min' => 100,
      '#max' => 2000,
      '#step' => 100,
      '#field_suffix' => $this->t('milliseconds'),
      '#description' => $this->t('Delay before triggering autocomplete search after user stops typing.'),
    ];

    $form['performance_settings']['cache_lifetime'] = [
      '#type' => 'number',
      '#title' => $this->t('Cache Lifetime'),
      '#default_value' => $config->get('cache_lifetime'),
      '#required' => TRUE,
      '#min' => 0,
      '#max' => 86400,
      '#field_suffix' => $this->t('seconds'),
      '#description' => $this->t('How long to cache API responses. Set to 0 to disable caching.'),
    ];

    $form['data_sources'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Data Sources'),
      '#description' => $this->t('Configure available data sources and layers.'),
    ];

    $form['data_sources']['layers'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Available Layers'),
      '#default_value' => $config->get('layers') ?? [],
      '#options' => [
        'venue' => $this->t('Venue (points of interest, businesses, things with walls)'),
        'address' => $this->t('Address (places with a street address)'),
        'street' => $this->t('Street (streets, roads, highways)'),
        'country' => $this->t('Country (places that issue passports, nations, nation-states)'),
        'macroregion' => $this->t('Macroregion (states and provinces)'),
        'region' => $this->t('Region (states and provinces)'),
        'macrocounty' => $this->t('Macrocounty (a related group of counties)'),
        'county' => $this->t('County (administrative divisions between localities and regions)'),
        'locality' => $this->t('Locality (towns, hamlets, cities)'),
        'localadmin' => $this->t('Local admin (local administrative boundaries)'),
        'borough' => $this->t('Borough (a local administrative boundary)'),
        'neighbourhood' => $this->t('Neighbourhood (social communities, neighbourhoods)'),
        'microhood' => $this->t('Microhood (a micro-neighbourhood)'),
        'disputed' => $this->t('Disputed (a disputed territory)'),
        'postalcode' => $this->t('Postalcode (postal codes used by mail services)'),
      ],
      '#description' => $this->t('Select which place types should be available for fields to use.'),
    ];

    $form['test_api'] = [
      '#type' => 'details',
      '#title' => $this->t('Test API Connection'),
      '#description' => $this->t('Test your API configuration.'),
      '#open' => FALSE,
    ];

    $form['test_api']['test_query'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Test Query'),
      '#default_value' => 'New York',
      '#description' => $this->t('Enter a location to test the API connection.'),
    ];

    $form['test_api']['test_button'] = [
      '#type' => 'button',
      '#value' => $this->t('Test Connection'),
      '#ajax' => [
        'callback' => '::testApiConnection',
        'wrapper' => 'test-results',
      ],
    ];

    $form['test_api']['test_results'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'test-results'],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Ajax callback to test API connection.
   */
  public function testApiConnection(array &$form, FormStateInterface $form_state) {
    $test_query = $form_state->getValue('test_query');
    $api_endpoint = $form_state->getValue('api_endpoint');
    $api_key = $form_state->getValue('api_key');

    $result_element = [
      '#type' => 'container',
      '#attributes' => ['id' => 'test-results'],
    ];

    if (empty($test_query) || empty($api_endpoint)) {
      $result_element['message'] = [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--error">' . $this->t('Please enter a test query and API endpoint.') . '</div>',
      ];
      return $result_element;
    }

    try {
      // Build test URL.
      $url = rtrim($api_endpoint, '/') . '/autocomplete';
      $query_params = ['text' => $test_query, 'size' => 1];
      
      if (!empty($api_key)) {
        $query_params['api_key'] = $api_key;
      }

      $client = \Drupal::httpClient();
      $response = $client->request('GET', $url, [
        'query' => $query_params,
        'timeout' => 10,
        'headers' => [
          'Accept' => 'application/json',
          'User-Agent' => 'Drupal Pelias Field Module (Test)',
        ],
      ]);

      $body = $response->getBody()->getContents();
      $data = json_decode($body, TRUE);

      if ($data && isset($data['features']) && is_array($data['features'])) {
        $count = count($data['features']);
        $result_element['message'] = [
          '#type' => 'markup',
          '#markup' => '<div class="messages messages--status">' . 
            $this->t('API connection successful! Found @count results for "@query".', [
              '@count' => $count,
              '@query' => $test_query,
            ]) . '</div>',
        ];

        if ($count > 0 && isset($data['features'][0]['properties']['label'])) {
          $first_result = $data['features'][0]['properties']['label'];
          $result_element['first_result'] = [
            '#type' => 'markup',
            '#markup' => '<p><strong>' . $this->t('First result:') . '</strong> ' . htmlspecialchars($first_result) . '</p>',
          ];
        }
      }
      else {
        $result_element['message'] = [
          '#type' => 'markup',
          '#markup' => '<div class="messages messages--warning">' . 
            $this->t('API responded but returned unexpected data format.') . '</div>',
        ];
      }
    }
    catch (\Exception $e) {
      $result_element['message'] = [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--error">' . 
          $this->t('API connection failed: @error', ['@error' => $e->getMessage()]) . '</div>',
      ];
    }

    return $result_element;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $api_endpoint = $form_state->getValue('api_endpoint');
    $api_key = $form_state->getValue('api_key');

    // Validate API endpoint format.
    if (!filter_var($api_endpoint, FILTER_VALIDATE_URL)) {
      $form_state->setErrorByName('api_endpoint', $this->t('Please enter a valid URL for the API endpoint.'));
    }

    // Warn if using geocode.earth without API key.
    if (strpos($api_endpoint, 'geocode.earth') !== FALSE && empty($api_key)) {
      $form_state->setErrorByName('api_key', $this->t('API key is required when using Geocode Earth service.'));
    }

    // Validate API key format for geocode.earth.
    if (!empty($api_key) && strpos($api_endpoint, 'geocode.earth') !== FALSE) {
      if (!preg_match('/^ge-[a-f0-9]{16}$/', $api_key)) {
        $form_state->setErrorByName('api_key', $this->t('Geocode Earth API keys should start with "ge-" followed by 16 hexadecimal characters.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Filter out unchecked options.
    $sources = array_filter($form_state->getValue('sources'));
    $layers = array_filter($form_state->getValue('layers'));

    $this->config('pelias_field.settings')
      ->set('api_endpoint', $form_state->getValue('api_endpoint'))
      ->set('api_key', $form_state->getValue('api_key'))
      ->set('timeout', $form_state->getValue('timeout'))
      ->set('debounce_delay', $form_state->getValue('debounce_delay'))
      ->set('cache_lifetime', $form_state->getValue('cache_lifetime'))
      ->set('sources', array_values($sources))
      ->set('layers', array_values($layers))
      ->save();

    parent::submitForm($form, $form_state);
  }

}