<?php

namespace Drupal\pelias_field\Plugin\Field\FieldType;

use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\TypedData\DataDefinition;

/**
 * Defines the 'pelias_field' field type.
 *
 * @FieldType(
 *   id = "pelias_field",
 *   label = @Translation("Pelias Field"),
 *   description = @Translation("A field for geocoding using Pelias API."),
 *   category = @Translation("Geographic"),
 *   default_widget = "pelias_field_autocomplete_widget",
 *   default_formatter = "pelias_field_formatter"
 * )
 */
class PeliasFieldType extends FieldItemBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultStorageSettings() {
    return [
      'max_length' => 255,
    ] + parent::defaultStorageSettings();
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultFieldSettings() {
    return [
      'max_suggestions' => 10,
      'min_length' => 2,
      'focus_point' => [
        'lat' => '',
        'lon' => '',
      ],
      'boundary_country' => '',
      'sources' => [],
      'layers' => [],
    ] + parent::defaultFieldSettings();
  }

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition) {
    $properties['value'] = DataDefinition::create('string')
      ->setLabel(t('Display value'))
      ->setRequired(FALSE);

    $properties['raw_data'] = DataDefinition::create('string')
      ->setLabel(t('Raw/Processed JSON data'))
      ->setDescription(t('Contains either original API response or processed data from hooks'))
      ->setRequired(FALSE);

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition) {
    $schema = [
      'columns' => [
        'value' => [
          'type' => 'varchar',
          'length' => (int) $field_definition->getSetting('max_length'),
          'not null' => FALSE,
        ],
        'raw_data' => [
          'type' => 'blob',
          'size' => 'big',
          'not null' => FALSE,
        ],
      ],
    ];

    return $schema;
  }

  /**
   * {@inheritdoc}
   */
  public function preSave() {
    parent::preSave();
    
    // Prepare the current field values
    $values = [
      'value' => $this->get('value')->getValue(),
      'raw_data' => $this->get('raw_data')->getValue(),
    ];
    
    // Create context for the hook
    $context = [
      'field_definition' => $this->getFieldDefinition(),
      'entity' => $this->getEntity(),
      'field_name' => $this->getFieldDefinition()->getName(),
      'entity_type' => $this->getEntity()->getEntityTypeId(),
      'bundle' => $this->getEntity()->bundle(),
      'delta' => $this->getName(),
      'entity_id' => $this->getEntity()->id(),
      'is_new' => $this->getEntity()->isNew(),
      'original_entity' => $this->getEntity()->original ?? NULL,
    ];
    
    // Decode raw_data to array for easier manipulation in the hook
    $original_raw_data = NULL;
    if (!empty($values['raw_data']) && is_string($values['raw_data'])) {
      $decoded = json_decode($values['raw_data'], TRUE);
      if (json_last_error() === JSON_ERROR_NONE) {
        $original_raw_data = $values['raw_data']; // Keep original for comparison
        $values['raw_data'] = $decoded;
      }
    }
    
    // Allow other modules to alter the data before saving
    \Drupal::moduleHandler()->alter('pelias_field_presave', $values, $context);
    
    // Apply the modified values back to the field
    if (isset($values['value'])) {
      $this->set('value', $values['value']);
    }
    
    if (isset($values['raw_data'])) {
      // If raw_data was manipulated (changed from original), encode and store
      if (is_array($values['raw_data'])) {
        $encoded_data = json_encode($values['raw_data'], JSON_UNESCAPED_UNICODE);
        
        // Check if the data was actually modified
        if ($original_raw_data !== NULL && $encoded_data !== $original_raw_data) {
          // Store the manipulated data
          $this->set('raw_data', $encoded_data);
          
          // Optional: Log manipulation for debugging/auditing
          \Drupal::logger('pelias_field')->info('Data was manipulated for field @field on @entity_type @bundle @id', [
            '@field' => $context['field_name'],
            '@entity_type' => $context['entity_type'],
            '@bundle' => $context['bundle'],
            '@id' => $context['entity_id'] ?? 'new',
          ]);
        } else {
          // Data wasn't changed, store original or encode array
          $this->set('raw_data', $original_raw_data ?? $encoded_data);
        }
      } else {
        // Data is already a string, use as-is
        $this->set('raw_data', $values['raw_data']);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function fieldSettingsForm(array $form, FormStateInterface $form_state) {
    $element = [];
    $config = \Drupal::config('pelias_field.settings');
    $available_sources = $config->get('sources') ?? [];
    $available_layers = $config->get('layers') ?? [];

    $element['max_suggestions'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum suggestions'),
      '#default_value' => $this->getSetting('max_suggestions'),
      '#required' => TRUE,
      '#min' => 1,
      '#max' => 50,
      '#description' => $this->t('Maximum number of autocomplete suggestions to display.'),
    ];

    $element['min_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum characters'),
      '#default_value' => $this->getSetting('min_length'),
      '#required' => TRUE,
      '#min' => 1,
      '#max' => 10,
      '#description' => $this->t('Minimum number of characters before triggering autocomplete.'),
    ];

    $element['focus_point'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Focus Point'),
      '#description' => $this->t('Bias results around a specific point (optional).'),
      '#collapsible' => TRUE,
      '#collapsed' => TRUE,
    ];

    $element['focus_point']['lat'] = [
      '#type' => 'number',
      '#title' => $this->t('Latitude'),
      '#default_value' => $this->getSetting('focus_point')['lat'] ?? '',
      '#step' => 'any',
      '#min' => -90,
      '#max' => 90,
    ];

    $element['focus_point']['lon'] = [
      '#type' => 'number',
      '#title' => $this->t('Longitude'),
      '#default_value' => $this->getSetting('focus_point')['lon'] ?? '',
      '#step' => 'any',
      '#min' => -180,
      '#max' => 180,
    ];

    $element['boundary_country'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Boundary Country'),
      '#default_value' => $this->getSetting('boundary_country'),
      '#maxlength' => 3,
      '#description' => $this->t('ISO country code to restrict results (e.g., "US", "GB", "IN").'),
    ];

    if (!empty($available_sources)) {
      $element['sources'] = [
        '#type' => 'checkboxes',
        '#title' => $this->t('Data Sources'),
        '#default_value' => $this->getSetting('sources'),
        '#options' => array_combine($available_sources, $available_sources),
        '#description' => $this->t('Select which data sources to query. Leave empty to use all available sources.'),
      ];
    }

    if (!empty($available_layers)) {
      $element['layers'] = [
        '#type' => 'checkboxes',
        '#title' => $this->t('Place Layers'),
        '#default_value' => $this->getSetting('layers'),
        '#options' => array_combine($available_layers, $available_layers),
        '#description' => $this->t('Select which types of places to search for. Leave empty to search all types.'),
      ];
    }

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function fieldSettingsFormValidate(array $form, FormStateInterface $form_state) {
    // Add any field settings validation if needed
    $values = $form_state->getValues();
    
    // Validate focus point coordinates
    if (!empty($values['settings']['focus_point']['lat'])) {
      $lat = $values['settings']['focus_point']['lat'];
      if ($lat < -90 || $lat > 90) {
        $form_state->setErrorByName('focus_point][lat', $this->t('Latitude must be between -90 and 90.'));
      }
    }
    
    if (!empty($values['settings']['focus_point']['lon'])) {
      $lon = $values['settings']['focus_point']['lon'];
      if ($lon < -180 || $lon > 180) {
        $form_state->setErrorByName('focus_point][lon', $this->t('Longitude must be between -180 and 180.'));
      }
    }
    
    // Validate boundary country format
    if (!empty($values['settings']['boundary_country'])) {
      $country = $values['settings']['boundary_country'];
      if (!preg_match('/^[A-Z]{2,3}$/', strtoupper($country))) {
        $form_state->setErrorByName('boundary_country', $this->t('Boundary country must be a 2 or 3 letter ISO country code.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function storageSettingsForm(array &$form, FormStateInterface $form_state, $has_data) {
    $form['max_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum length'),
      '#default_value' => $this->getSetting('max_length'),
      '#required' => TRUE,
      '#description' => $this->t('The maximum length of the display value in characters.'),
      '#min' => 1,
      '#disabled' => $has_data,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty() {
    $value = $this->get('value')->getValue();
    return $value === NULL || $value === '';
  }

  /**
   * {@inheritdoc}
   */
  public function setValue($values, $notify = TRUE) {
    // Ensure raw_data is properly JSON encoded if it's an array.
    if (isset($values['raw_data']) && is_array($values['raw_data'])) {
      $values['raw_data'] = json_encode($values['raw_data'], JSON_UNESCAPED_UNICODE);
    }

    parent::setValue($values, $notify);
  }

  /**
   * Get the raw/processed data as an array.
   *
   * @return array
   *   The data array (either original raw data or processed data from hooks).
   */
  public function getData(): array {
    $raw_data = $this->get('raw_data')->getValue();
    if (empty($raw_data)) {
      return [];
    }

    $decoded = json_decode($raw_data, TRUE);
    return is_array($decoded) ? $decoded : [];
  }

  /**
   * Helper method to extract specific values from the stored data.
   *
   * @param string $path
   *   Dot notation path (e.g., 'properties.latitude', 'geometry.coordinates.0').
   * @param mixed $default
   *   Default value if path is not found.
   *
   * @return mixed
   *   The extracted value or default.
   */
  public function extractValue($path, $default = NULL) {
    $data = $this->getData();
    if (empty($data)) {
      return $default;
    }

    $keys = explode('.', $path);
    $current = $data;

    foreach ($keys as $key) {
      if (is_array($current) && isset($current[$key])) {
        $current = $current[$key];
      } else {
        return $default;
      }
    }

    return $current;
  }

  /**
   * Check if the stored data contains processed/manipulated data.
   *
   * @return bool
   *   TRUE if data appears to be processed, FALSE if it looks like original API data.
   */
  public function isProcessedData(): bool {
    $data = $this->getData();
    
    if (empty($data)) {
      return FALSE;
    }
    
    // Check for common indicators of processed data
    $processed_indicators = [
      '_processed',
      '_manipulated', 
      '_timestamp',
      '_format_version',
      'processed_timestamp',
      'custom_fields',
      'computed_fields',
    ];
    
    foreach ($processed_indicators as $indicator) {
      if (isset($data[$indicator])) {
        return TRUE;
      }
    }
    
    // Check if it lacks typical Pelias API structure
    $typical_pelias_keys = ['type', 'geometry', 'properties'];
    $has_pelias_structure = TRUE;
    
    foreach ($typical_pelias_keys as $key) {
      if (!isset($data[$key])) {
        $has_pelias_structure = FALSE;
        break;
      }
    }
    
    // If it doesn't have typical Pelias structure, it's likely processed
    return !$has_pelias_structure;
  }

}