<?php

namespace Drupal\pelias_field\Service;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;

/**
 * Service for interacting with Pelias APIs.
 */
class PeliasFieldService implements PeliasFieldInterface {

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Constructs a new PeliasFieldService object.
   *
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(ClientInterface $http_client, ConfigFactoryInterface $config_factory, CacheBackendInterface $cache, LoggerChannelFactoryInterface $logger_factory) {
    $this->httpClient = $http_client;
    $this->configFactory = $config_factory;
    $this->cache = $cache;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * {@inheritdoc}
   */
  public function autocomplete(string $text, array $options = []): array {
    if (empty($text) || strlen($text) < $this->getMinLength()) {
      return [];
    }

    $cache_key = $this->getCacheKey('autocomplete', $text, $options);
    $cached = $this->cache->get($cache_key);
    
    if ($cached) {
      return $cached->data;
    }

    try {
      $config = $this->configFactory->get('pelias_field.settings');
      $endpoint = rtrim($config->get('api_endpoint'), '/');
      $url = $endpoint . '/autocomplete';
      
      $query_params = array_merge([
        'text' => $text,
        'size' => $options['max_suggestions'] ?? 10,
      ], $this->buildQueryParams($options));

      // Add API key if configured (for geocode.earth).
      $api_key = $config->get('api_key');
      if (!empty($api_key)) {
        $query_params['api_key'] = $api_key;
      }

      $response = $this->httpClient->request('GET', $url, [
        'query' => $query_params,
        'timeout' => $config->get('timeout') ?? 30,
        'headers' => [
          'Accept' => 'application/json',
          'User-Agent' => 'Drupal Pelias Field Module',
        ],
      ]);

      $body = $response->getBody()->getContents();
      $data = json_decode($body, TRUE);

      if (!$data || !isset($data['features'])) {
        $this->loggerFactory->get('pelias_field')->error('Invalid response from Pelias API: @response', ['@response' => $body]);
        return [];
      }

      $results = $data['features'];
      
      // Cache the results.
      $cache_lifetime = $config->get('cache_lifetime') ?? 3600;
      $this->cache->set($cache_key, $results, time() + $cache_lifetime);

      return $results;

    }
    catch (RequestException $e) {
      $this->loggerFactory->get('pelias_field')->error('Error calling Pelias autocomplete API: @error', ['@error' => $e->getMessage()]);
      return [];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function search(string $text, array $options = []): array {
    if (empty($text)) {
      return [];
    }

    $cache_key = $this->getCacheKey('search', $text, $options);
    $cached = $this->cache->get($cache_key);
    
    if ($cached) {
      return $cached->data;
    }

    try {
      $config = $this->configFactory->get('pelias_field.settings');
      $endpoint = rtrim($config->get('api_endpoint'), '/');
      $url = $endpoint . '/search';
      
      $query_params = array_merge([
        'text' => $text,
        'size' => $options['max_results'] ?? 10,
      ], $this->buildQueryParams($options));

      // Add API key if configured.
      $api_key = $config->get('api_key');
      if (!empty($api_key)) {
        $query_params['api_key'] = $api_key;
      }

      $response = $this->httpClient->request('GET', $url, [
        'query' => $query_params,
        'timeout' => $config->get('timeout') ?? 30,
        'headers' => [
          'Accept' => 'application/json',
          'User-Agent' => 'Drupal Pelias Field Module',
        ],
      ]);

      $body = $response->getBody()->getContents();
      $data = json_decode($body, TRUE);

      if (!$data || !isset($data['features'])) {
        $this->loggerFactory->get('pelias_field')->error('Invalid response from Pelias search API: @response', ['@response' => $body]);
        return [];
      }

      $results = $data['features'];
      
      // Cache the results.
      $cache_lifetime = $config->get('cache_lifetime') ?? 3600;
      $this->cache->set($cache_key, $results, time() + $cache_lifetime);

      return $results;

    }
    catch (RequestException $e) {
      $this->loggerFactory->get('pelias_field')->error('Error calling Pelias search API: @error', ['@error' => $e->getMessage()]);
      return [];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function reverse(float $lat, float $lon, array $options = []): array {
    $cache_key = $this->getCacheKey('reverse', "{$lat},{$lon}", $options);
    $cached = $this->cache->get($cache_key);
    
    if ($cached) {
      return $cached->data;
    }

    try {
      $config = $this->configFactory->get('pelias_field.settings');
      $endpoint = rtrim($config->get('api_endpoint'), '/');
      $url = $endpoint . '/reverse';
      
      $query_params = array_merge([
        'point.lat' => $lat,
        'point.lon' => $lon,
        'size' => $options['max_results'] ?? 1,
      ], $this->buildQueryParams($options));

      // Add API key if configured.
      $api_key = $config->get('api_key');
      if (!empty($api_key)) {
        $query_params['api_key'] = $api_key;
      }

      $response = $this->httpClient->request('GET', $url, [
        'query' => $query_params,
        'timeout' => $config->get('timeout') ?? 30,
        'headers' => [
          'Accept' => 'application/json',
          'User-Agent' => 'Drupal Pelias Field Module',
        ],
      ]);

      $body = $response->getBody()->getContents();
      $data = json_decode($body, TRUE);

      if (!$data || !isset($data['features'])) {
        $this->loggerFactory->get('pelias_field')->error('Invalid response from Pelias reverse API: @response', ['@response' => $body]);
        return [];
      }

      $results = $data['features'];
      
      // Cache the results.
      $cache_lifetime = $config->get('cache_lifetime') ?? 3600;
      $this->cache->set($cache_key, $results, time() + $cache_lifetime);

      return $results;

    }
    catch (RequestException $e) {
      $this->loggerFactory->get('pelias_field')->error('Error calling Pelias reverse API: @error', ['@error' => $e->getMessage()]);
      return [];
    }
  }

  /**
   * Build query parameters based on options.
   *
   * @param array $options
   *   The options array.
   *
   * @return array
   *   The query parameters.
   */
  protected function buildQueryParams(array $options): array {
    $params = [];

    // Focus point (bias results around a point).
    if (!empty($options['focus_point'])) {
      $params['focus.point.lat'] = $options['focus_point']['lat'];
      $params['focus.point.lon'] = $options['focus_point']['lon'];
    }

    // Boundary country.
    if (!empty($options['boundary_country'])) {
      $params['boundary.country'] = $options['boundary_country'];
    }

    // Sources (data sources to query).
    if (!empty($options['sources']) && is_array($options['sources'])) {
      $params['sources'] = implode(',', $options['sources']);
    }

    // Layers (types of places to search for).
    if (!empty($options['layers']) && is_array($options['layers'])) {
      $params['layers'] = implode(',', $options['layers']);
    }

    return $params;
  }

  /**
   * Generate a cache key.
   *
   * @param string $type
   *   The type of request (autocomplete, search, reverse).
   * @param string $query
   *   The query string.
   * @param array $options
   *   The options array.
   *
   * @return string
   *   The cache key.
   */
  protected function getCacheKey(string $type, string $query, array $options): string {
    $key_parts = [
      'pelias_field',
      $type,
      md5($query),
      md5(serialize($options)),
    ];
    return implode(':', $key_parts);
  }

  /**
   * Get the minimum length for search queries.
   *
   * @return int
   *   The minimum length.
   */
  protected function getMinLength(): int {
    return 2;
  }

}