/**
 * @file
 * Pelias Field autocomplete functionality.
 */

(function ($, Drupal, once, debounce) {
  'use strict';

  /**
   * Pelias autocomplete behavior.
   */
  Drupal.behaviors.peliasAutocomplete = {
    attach: function (context, settings) {
      once('pelias-autocomplete', '.pelias-field-autocomplete', context).forEach(function (element) {
        var $input = $(element);
        var $wrapper = $input.closest('.pelias-field-wrapper');
        var $rawDataField = $wrapper.find('.pelias-field-raw-data');

        var autocompleteUrl = $input.data('pelias-autocomplete-url');
        var debounceDelay = parseInt($input.data('pelias-debounce-delay')) || 300;
        var minLength = parseInt($input.data('pelias-min-length')) || 2;

        var $suggestionsList = $('<ul class="pelias-suggestions"></ul>');
        var $suggestionsContainer = $('<div class="pelias-suggestions-container"></div>');

        $suggestionsContainer.append($suggestionsList);
        $wrapper.append($suggestionsContainer);

        var currentRequest = NULL;
        var selectedIndex = -1;

        // Hide suggestions initially
        $suggestionsContainer.hide();

        // Debounced search function
        var performSearch = debounce(function (query) {
          if (query.length < minLength) {
            hideSuggestions();
            return;
          }

          // Cancel previous request
          if (currentRequest) {
            currentRequest.abort();
          }

          // Show loading state
          $suggestionsList.html('<li class="pelias-suggestion-loading">' + Drupal.t('Searching...') + '</li>');
          $suggestionsContainer.show();

          currentRequest = $.ajax({
            url: autocompleteUrl,
            method: 'GET',
            data: { q: query },
            dataType: 'json'
          })
          .done(function (data) {
            displaySuggestions(data);
          })
          .fail(function (xhr) {
            if (xhr.statusText !== 'abort') {
              $suggestionsList.html('<li class="pelias-suggestion-error">' + Drupal.t('Search failed. Please try again.') + '</li>');
            }
          })
          .always(function () {
            currentRequest = NULL;
          });
        }, debounceDelay);

        // Input event handler
        $input.on('input keyup', function (e) {
          var query = $(this).val().trim();

          // Handle special keys
          if (e.type === 'keyup') {
            switch (e.keyCode) {
              case 38: // Up arrow
                e.preventDefault();
                navigateSuggestions(-1);
                return;

              case 40: // Down arrow
                e.preventDefault();
                navigateSuggestions(1);
                return;

              case 13: // Enter
                e.preventDefault();
                if (selectedIndex >= 0) {
                  selectSuggestion(selectedIndex);
                }
                return;

              case 27: // Escape
                hideSuggestions();
                return;
            }
          }

          selectedIndex = -1;

          if (query === '') {
            clearFieldData();
            hideSuggestions();
            return;
          }

          performSearch(query);
        });

        // Click outside to hide suggestions
        $(document).on('click', function (e) {
          if (!$wrapper.is(e.target) && $wrapper.has(e.target).length === 0) {
            hideSuggestions();
          }
        });

        function displaySuggestions(suggestions) {
          $suggestionsList.empty();
          selectedIndex = -1;

          if (!suggestions || suggestions.length === 0) {
            $suggestionsList.html('<li class="pelias-suggestion-empty">' + Drupal.t('No results found') + '</li>');
            $suggestionsContainer.show();
            return;
          }

          suggestions.forEach(function (suggestion, index) {
            var $item = $('<li class="pelias-suggestion-item"></li>')
              .html('<strong>' + escapeHtml(suggestion.value) + '</strong><br><small>' + escapeHtml(suggestion.label) + '</small>')
              .data('suggestion', suggestion)
              .data('index', index);

            $item.on('click', function () {
              selectSuggestion(index);
            });

            $item.on('mouseenter', function () {
              highlightSuggestion(index);
            });

            $suggestionsList.append($item);
          });

          $suggestionsContainer.show();
        }

        function navigateSuggestions(direction) {
          var $items = $suggestionsList.find('.pelias-suggestion-item');
          if ($items.length === 0) { return;
          }

          selectedIndex += direction;

          if (selectedIndex < 0) {
            selectedIndex = $items.length - 1;
          } else if (selectedIndex >= $items.length) {
            selectedIndex = 0;
          }

          highlightSuggestion(selectedIndex);
        }

        function highlightSuggestion(index) {
          var $items = $suggestionsList.find('.pelias-suggestion-item');
          $items.removeClass('pelias-suggestion-highlighted');

          if (index >= 0 && index < $items.length) {
            $items.eq(index).addClass('pelias-suggestion-highlighted');
            selectedIndex = index;
          }
        }

        function selectSuggestion(index) {
          var $items = $suggestionsList.find('.pelias-suggestion-item');
          if (index < 0 || index >= $items.length) { return;
          }

          var suggestion = $items.eq(index).data('suggestion');
          if (!suggestion) { return;
          }

          // Update form fields with proper data
          $input.val(suggestion.value || '');
          $rawDataField.val(suggestion.raw_data || '');
          // Mark fields as changed to ensure they're processed during form submission
          $rawDataField.trigger('change');
          $input.trigger('change');

          // Set a flag to indicate this field has been populated via autocomplete
          $input.attr('data-pelias-selected', 'true');

          hideSuggestions();
        }

        function hideSuggestions() {
          $suggestionsContainer.hide();
          selectedIndex = -1;
        }

        function clearFieldData() {
          $rawDataField.val('');

          // Trigger change events to ensure form state is updated
          $rawDataField.trigger('change');

          // Remove the autocomplete selection flag
          $input.removeAttr('data-pelias-selected');
        }

        function escapeHtml(text) {
          var div = document.createElement('div');
          div.textContent = text;
          return div.innerHTML;
        }
      });
    }
  };

})(jQuery, Drupal, once, Drupal.debounce);