<?php

namespace Drupal\pelias_field\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Defines the 'pelias_field_formatter' field formatter.
 *
 * @FieldFormatter(
 *   id = "pelias_field_formatter",
 *   label = @Translation("Pelias Field Formatter"),
 *   field_types = {"pelias_field"}
 * )
 */
class PeliasFieldFormatter extends FormatterBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'display_format' => 'label_only',
      'show_coordinates' => FALSE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form['display_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Display format'),
      '#default_value' => $this->getSetting('display_format'),
      '#options' => [
        'label_only' => $this->t('Label only'),
        'label_with_context' => $this->t('Label with context'),
        'full_address' => $this->t('Full address'),
        'coordinates_only' => $this->t('Coordinates only'),
        'custom' => $this->t('Custom template'),
      ],
      '#required' => TRUE,
    ];

    $form['show_coordinates'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show coordinates'),
      '#default_value' => $this->getSetting('show_coordinates'),
      '#description' => $this->t('Whether to display latitude and longitude coordinates.'),
      '#states' => [
        'invisible' => [
          ':input[name*="display_format"]' => ['value' => 'coordinates_only'],
        ],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];

    $display_format_options = [
      'label_only' => $this->t('Label only'),
    ];

    $summary[] = $this->t('Display format: @format', [
      '@format' => $display_format_options[$this->getSetting('display_format')] ?? $this->getSetting('display_format'),
    ]);

    if ($this->getSetting('show_coordinates')) {
      $summary[] = $this->t('Show coordinates: Yes');
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];

    foreach ($items as $delta => $item) {
      if ($item->isEmpty()) {
        continue;
      }

      $data = $item->getData();
      $display_value = $item->get('value')->getValue();

      $content = $this->formatContent($display_value, $data);

      if (!empty($content)) {
        $elements[$delta] = [
          '#markup' => $content,
          '#cache' => [
            'tags' => $item->getEntity()->getCacheTags(),
          ],
        ];
      }
    }

    return $elements;
  }

  /**
   * Format the content based on display settings.
   *
   * @param string $display_value
   *   The display value.
   * @param array $data
   *   The field data array.
   *
   * @return string
   *   The formatted content.
   */
  protected function formatContent(string $display_value, array $data): string {
    $format = $this->getSetting('display_format');
    $content = '';

    switch ($format) {
      case 'label_only':
        $content = $this->formatLabelOnly($display_value, $data);
        break;

      default:
        $content = $display_value;
    }

    // Add coordinates if requested.
    if ($this->getSetting('show_coordinates') && $format !== 'coordinates_only') {
      $coordinates = $this->getCoordinates($data);
      if (!empty($coordinates)) {
        $content .= ' <span class="pelias-coordinates">(' . $coordinates . ')</span>';
      }
    }
    return $content;
  }

  /**
   * Format as label only.
   */
  protected function formatLabelOnly(string $display_value, array $data): string {
    return htmlspecialchars($display_value);
  }

  /**
   * Get formatted coordinates from data.
   */
  protected function getCoordinates(array $data): string {
    if (isset($data['geometry']['coordinates']) && is_array($data['geometry']['coordinates'])) {
      $coordinates = $data['geometry']['coordinates'];
      if (count($coordinates) >= 2) {
        $lat = round($coordinates[1], 6);
        $lng = round($coordinates[0], 6);
        return "{$lat}, {$lng}";
      }
    }

    // Check for processed data format.
    if (isset($data['coordinates']['lat']) && isset($data['coordinates']['lng'])) {
      $lat = round($data['coordinates']['lat'], 6);
      $lng = round($data['coordinates']['lng'], 6);
      return "{$lat}, {$lng}";
    }

    return '';
  }

}
