<?php

/**
 * @file
 * Hooks provided by the Pelias Field module.
 */

/**
 * Alter Pelias field data before it's saved to the database.
 *
 * This hook allows modules to process and modify the field data before it's
 * stored in the field table. You can modify the raw data, parsed data, or
 * display value based on your requirements.
 *
 * @param array &$values
 *   An associative array containing the field values:
 *   - 'value': The display value (string)
 *   - 'raw_data': The raw JSON data from Pelias API (string or array)
 * @param array $context
 *   An associative array containing context information:
 *   - 'field_definition': The field definition object
 *   - 'entity': The entity being saved
 *   - 'field_name': The machine name of the field
 *   - 'entity_type': The entity type ID
 *   - 'bundle': The entity bundle
 *   - 'delta': The field delta (for multi-value fields)
 *   - 'entity_id': The entity ID (NULL for new entities)
 *   - 'is_new': Boolean indicating if this is a new entity
 *   - 'original_entity': The original entity object (for updates)
 *   - 'changed_fields': Array of field names that have changed.
 *
 * @see \Drupal\pelias_field\Plugin\Field\FieldType\PeliasGeoFieldType::setValue()
 */
function hook_pelias_field_presave_alter(array &$values, array $context) {
  // Example 1: Transform raw Pelias data into a custom format.
  if (isset($values['raw_data']) && is_array($values['raw_data'])) {
    $raw = $values['raw_data'];

    // Create a simplified, custom data structure.
    $custom_data = [
      '_processed' => TRUE,
      '_timestamp' => time(),
      'display_name' => $raw['properties']['label'] ?? '',
      'coordinates' => [
        'lat' => $raw['geometry']['coordinates'][1] ?? 0,
        'lng' => $raw['geometry']['coordinates'][0] ?? 0,
      ],
      'address_components' => [
        'street' => trim(($raw['properties']['housenumber'] ?? '') . ' ' . ($raw['properties']['street'] ?? '')),
        'city' => $raw['properties']['locality'] ?? '',
        'state' => $raw['properties']['region'] ?? '',
        'country' => $raw['properties']['country'] ?? '',
        'postal_code' => $raw['properties']['postalcode'] ?? '',
      ],
      'source' => $raw['properties']['source'] ?? 'pelias',
    ];

    // Replace the raw data with processed data.
    $values['raw_data'] = $custom_data;
  }
}
