<?php

namespace Drupal\pelias_field\Service;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Service for parsing Pelias API responses.
 */
class PeliasResponseParserService {

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Constructs a new PeliasResponseParserService object.
   *
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(LoggerChannelFactoryInterface $logger_factory) {
    $this->loggerFactory = $logger_factory;
  }

  /**
   * Parse a GeoJSON feature into structured data.
   *
   * @param array $feature
   *   The GeoJSON feature.
   * @param array $field_mappings
   *   Array of field mappings with 'field_name' and 'json_path' keys.
   *
   * @return array
   *   Array with 'raw' and 'parsed' keys.
   */
  public function parseFeature(array $feature, array $field_mappings = []): array {
    $result = [
      'raw' => $feature,
      'parsed' => [],
    ];

    // Extract common fields by default.
    $default_mappings = [
      'label' => 'properties.label',
      'name' => 'properties.name',
      'source' => 'properties.source',
      'layer' => 'properties.layer',
      'country' => 'properties.country',
      'country_code' => 'properties.country_a',
      'region' => 'properties.region',
      'county' => 'properties.county',
      'locality' => 'properties.locality',
      'neighbourhood' => 'properties.neighbourhood',
      'street' => 'properties.street',
      'postalcode' => 'properties.postalcode',
      'confidence' => 'properties.confidence',
      'latitude' => 'geometry.coordinates.1',
      'longitude' => 'geometry.coordinates.0',
    ];

    // Merge custom mappings with default mappings.
    $all_mappings = [];
    foreach ($field_mappings as $mapping) {
      if (!empty($mapping['field_name']) && !empty($mapping['json_path'])) {
        $all_mappings[$mapping['field_name']] = $mapping['json_path'];
      }
    }
    $all_mappings = array_merge($default_mappings, $all_mappings);

    // Extract values using JSON paths.
    foreach ($all_mappings as $field_name => $json_path) {
      $value = $this->extractValueByPath($feature, $json_path);
      if ($value !== NULL) {
        $result['parsed'][$field_name] = $value;
      }
    }

    return $result;
  }

  /**
   * Extract autocomplete suggestions from API response.
   *
   * @param array $features
   *   Array of GeoJSON features.
   * @param array $field_mappings
   *   Array of field mappings.
   *
   * @return array
   *   Array of suggestions for autocomplete.
   */
  public function extractAutocompleteSuggestions(array $features, array $field_mappings = []): array {
    $suggestions = [];

    foreach ($features as $feature) {
      $parsed = $this->parseFeature($feature, $field_mappings);

      $suggestion = [
        'value' => $parsed['parsed']['label'] ?? $parsed['parsed']['name'] ?? 'Unknown',
        'label' => $this->buildDisplayLabel($parsed['parsed']),
        'data' => $parsed,
      ];

      $suggestions[] = $suggestion;
    }

    return $suggestions;
  }

  /**
   * Build a display label for a location.
   *
   * @param array $parsed_data
   *   The parsed location data.
   *
   * @return string
   *   The display label.
   */
  protected function buildDisplayLabel(array $parsed_data): string {
    $parts = [];

    // Primary label or name.
    if (!empty($parsed_data['label'])) {
      $parts[] = $parsed_data['label'];
    }
    elseif (!empty($parsed_data['name'])) {
      $parts[] = $parsed_data['name'];
    }

    // Add contextual information.
    $context_parts = [];

    if (!empty($parsed_data['locality']) && $parsed_data['locality'] !== ($parsed_data['name'] ?? '')) {
      $context_parts[] = $parsed_data['locality'];
    }

    if (!empty($parsed_data['region']) && !in_array($parsed_data['region'], $context_parts)) {
      $context_parts[] = $parsed_data['region'];
    }

    if (!empty($parsed_data['country']) && !in_array($parsed_data['country'], $context_parts)) {
      $context_parts[] = $parsed_data['country'];
    }

    if (!empty($context_parts)) {
      $parts[] = '(' . implode(', ', $context_parts) . ')';
    }

    return implode(' ', $parts);
  }

  /**
   * Extract a value from nested array using dot notation path.
   *
   * @param array $data
   *   The data array.
   * @param string $path
   *   The dot notation path (e.g., 'properties.label').
   *
   * @return mixed|null
   *   The extracted value or NULL if not found.
   */
  protected function extractValueByPath(array $data, string $path) {
    $keys = explode('.', $path);
    $current = $data;

    foreach ($keys as $key) {
      if (!is_array($current) || !isset($current[$key])) {
        return NULL;
      }
      $current = $current[$key];
    }

    return $current;
  }

}
