<?php

namespace Drupal\pelias_field\Plugin\Field\FieldWidget;

use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;

/**
 * Defines the 'pelias_field_autocomplete_widget' field widget.
 *
 * @FieldWidget(
 *   id = "pelias_field_autocomplete_widget",
 *   label = @Translation("Pelias Autocomplete Field"),
 *   field_types = {"pelias_field"}
 * )
 */
class PeliasFieldAutoCompleteWidget extends WidgetBase {

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a PeliasFieldAutoCompleteWidget object.
   *
   * @param string $plugin_id
   *   The plugin_id for the widget.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition.
   * @param array $settings
   *   The widget settings.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, array $third_party_settings, ConfigFactoryInterface $config_factory) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
    $plugin_id,
    $plugin_definition,
    $configuration['field_definition'],
    $configuration['settings'],
    $configuration['third_party_settings'],
    $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'placeholder' => '',
      'size' => 60,
      'show_raw_data_field' => FALSE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form['placeholder'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Placeholder text'),
      '#default_value' => $this->getSetting('placeholder'),
      '#description' => $this->t('Text that will be shown inside the field until a value is entered. This hint is usually a sample value or a brief description of the expected format.'),
    ];

    $form['size'] = [
      '#type' => 'number',
      '#title' => $this->t('Size of textfield'),
      '#default_value' => $this->getSetting('size'),
      '#required' => TRUE,
      '#min' => 1,
    ];

    $form['show_raw_data_field'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show raw data field'),
      '#default_value' => $this->getSetting('show_raw_data_field'),
      '#description' => $this->t('Show the raw API response data field for debugging purposes.'),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];

    if (!empty($this->getSetting('placeholder'))) {
      $summary[] = $this->t('Placeholder: @placeholder', ['@placeholder' => $this->getSetting('placeholder')]);
    }

    $summary[] = $this->t('Textfield size: @size', ['@size' => $this->getSetting('size')]);

    if ($this->getSetting('show_raw_data_field')) {
      $summary[] = $this->t('Raw data field: Visible');
    }
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $field_definition = $items->getFieldDefinition();
    $field_name = $field_definition->getName();
    $entity_type = $field_definition->getTargetEntityTypeId();
    $bundle = $field_definition->getTargetBundle();

    $config = $this->configFactory->get('pelias_field.settings');
    $debounce_delay = $config->get('debounce_delay') ?? 300;

    // Main container - this is important for proper form structure.
    $element += [
      '#type' => 'container',
      '#attributes' => ['class' => ['pelias-field-wrapper']],
    ];

    $element['value'] = [
      '#type' => 'textfield',
      '#title' => $element['#title'],
      '#default_value' => $items[$delta]->value ?? '',
      '#size' => $this->getSetting('size'),
      '#placeholder' => $this->getSetting('placeholder'),
      '#maxlength' => $field_definition->getSetting('max_length') ?? 255,
      '#attributes' => [
        'class' => ['pelias-field-autocomplete'],
        'data-pelias-autocomplete-url' => "/pelias-field/autocomplete/{$field_name}/{$entity_type}/{$bundle}",
        'data-pelias-debounce-delay' => $debounce_delay,
        'data-pelias-min-length' => $field_definition->getSetting('min_length') ?? 2,
      ],
      '#attached' => [
        'library' => ['pelias_field/autocomplete'],
      ],
    ];

    // Hidden fields to store the raw data - these need to be form elements.
    $element['raw_data'] = [
      '#type' => $this->getSetting('show_raw_data_field') ? 'textarea' : 'hidden',
      '#title' => $this->getSetting('show_raw_data_field') ? $this->t('Raw Data (JSON)') : '',
      '#default_value' => $items[$delta]->raw_data ?? '',
      '#attributes' => ['class' => ['pelias-field-raw-data']],
      '#rows' => $this->getSetting('show_raw_data_field') ? 5 : 1,
    // Ensure these fields are always processed.
      '#access' => TRUE,
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    foreach ($values as $delta => &$item) {
      // Handle completely empty fields.
      if (empty($item['value']) && empty($item['raw_data'])) {
        $item['value'] = '';
        $item['raw_data'] = '';
        continue;
      }

      // Validate and clean up JSON data.
      if (!empty($item['raw_data'])) {
        $decoded = json_decode($item['raw_data'], TRUE);
        if (json_last_error() !== JSON_ERROR_NONE) {
          $this->messenger()->addWarning($this->t('Invalid raw data JSON for field item @delta: @error', [
            '@delta' => $delta + 1,
            '@error' => json_last_error_msg(),
          ]));
          $item['raw_data'] = '';
        }
        else {
          // Re-encode to ensure consistent formatting.
          $item['raw_data'] = json_encode($decoded, JSON_UNESCAPED_UNICODE);
        }
      }

      // If we have raw data but no display value, try to extract it.
      if (empty($item['value']) && !empty($item['raw_data'])) {
        $raw_decoded = json_decode($item['raw_data'], TRUE);
        if (is_array($raw_decoded)) {
          $item['value'] = $raw_decoded['properties']['label'] ??
                          $raw_decoded['properties']['name'] ??
                          '';
        }
      }
    }

    return $values;
  }

  /**
   * {@inheritdoc}
   */
  public static function validateElement(array $element, FormStateInterface $form_state) {
    $value = $element['value']['#value'];
    $raw_data = $element['raw_data']['#value'];

    // If we have a display value but no raw data,that's problematic.
    if (!empty($value) && empty($raw_data)) {
      // This might be manual entry, which is fine, but we should note it.
      $form_state->setValueForElement($element['raw_data'], '');
    }

    // Validate JSON if present.
    if (!empty($raw_data)) {
      if (json_last_error() !== JSON_ERROR_NONE) {
        $form_state->setError($element['raw_data'], t('Raw data must be valid JSON.'));
      }
    }
  }

}
