<?php

namespace Drupal\per_domain_fields\Plugin\Field\FieldType;

use Drupal\Component\Assertion\Inspector;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\per_domain_fields\LoadsOverridableActiveDomainIdTrait;

/**
 * Shared code for field types that convert an ordinary type to be per-domain.
 */
trait PerDomainFieldTypeTrait {
  use LoadsOverridableActiveDomainIdTrait;

  /**
   * Privately log when a DB save is in process.
   *
   * @var bool
   */
  private bool $saveIsInFlight = FALSE;

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition) {
    $properties = parent::propertyDefinitions($field_definition);

    // Convert our parent field type's usual properties to per_domain version.
    foreach ($properties as $property) {
      $property->setDataType('per_domain_any');
    }

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition) {
    $schema = [];

    // Convert our parent field type's usual columns to serialized blob storage.
    foreach (array_keys(parent::schema($field_definition)['columns'] ?? []) as $name) {
      $schema['columns'][$name] = [
        'type' => 'blob',
        'size' => 'big',
        'serialize' => TRUE,
        'not null' => FALSE,
        'description' => sprintf('Serialized array of Domain IDs and their %s.', $name),
      ];
    }

    // @todo Add support for unique keys, indexes, and foreign keys.
    return $schema;
  }

  /**
   * {@inheritdoc}
   */
  public function getConstraints() {
    // @todo Convert parent constraints to new format.
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function preSave() {
    $this->ensureAllPropertiesAreDomainValueArrays();
    $this->saveIsInFlight = TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function postSave($update) {
    $this->saveIsInFlight = FALSE;
    return parent::postSave($update);
  }

  /**
   * Helper that casts all properties to arrays.
   */
  private function ensureAllPropertiesAreDomainValueArrays() {
    foreach ($this->getProperties() as $property) {
      $property_name = $property->getName();
      $property_value = $this->get($property_name)->getValueForAllDomains();

      if (!is_array($property_value)) {
        $this->get($property_name)->setValueForAllDomains((array) $property_value);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty() {
    $domain_values = $this->get('value')->getValueForAllDomains();
    return empty($domain_values) || !is_array($domain_values);
  }

  /**
   * {@inheritdoc}
   */
  public function setValue($values, $notify = TRUE) {
    // Treat the values as property value of the first property, if no array is
    // given. See \Drupal\Core\Field\FieldItemBase::setValue().
    if (isset($values) && !is_array($values)) {
      $keys = array_keys($this->definition->getPropertyDefinitions());
      $values = [$keys[0] => $values];
    }

    // Translate non-domain values to domain values.
    if (isset($values) && !is_array($values)) {
      throw new \InvalidArgumentException("Invalid values given. Values must be represented as an associative array.");
    }

    foreach ($values as $name => $value) {
      if (is_array($value) || substr($name, 0, 1) === '_') {
        continue;
      }

      $values[$name] = [$this->getActiveDomainId() => $value];
    }

    parent::setValue($values, $notify);
  }

  /**
   * {@inheritdoc}
   */
  public function getValue() {
    $value = parent::getValue();

    if (!is_array($value) || Inspector::assertStrictArray($value)) {
      return $value;
    }

    $valid_property_names = array_keys($this->getProperties());
    $is_property_array = count(array_intersect($valid_property_names, array_keys($value))) > 0;

    if (!$is_property_array) {
      return $value[$this->getActiveDomainId()] ?? reset($value);
    }

    foreach ($value as $property_name => $property_value) {
      if (!is_array($property_value) || Inspector::assertStrictArray($property_value)) {
        continue;
      }

      $value[$property_name] = $property_value[$this->getActiveDomainId()] ?? reset($property_value);
    }

    return $value;
  }

  /**
   * {@inheritdoc}
   */
  public function __get($name) {
    // Drupal core gives field types zero control over mapping values to the DB,
    // and requests the value via $item->$field_name when making the DB record.
    // We set this flag so we know to break encapsulation and return the full
    // domain-value mapping when DB insert/update queries are being built.
    if ($this->saveIsInFlight) {
      return $this->properties[$name]->getValueForAllDomains();
    }

    $value = parent::__get($name);

    if (!is_array($value) || Inspector::assertStrictArray($value)) {
      return $value;
    }

    return $value[$this->getActiveDomainId()] ?? reset($value);
  }

}
