<?php

namespace Drupal\Tests\per_domain_fields\Traits;

use Drupal\Core\Entity\Display\EntityFormDisplayInterface;
use Drupal\Core\Entity\Display\EntityViewDisplayInterface;
use Drupal\domain\DomainInterface;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\Tests\BrowserTestBase;

/**
 * Shared code for testing Per-domain fields.
 */
trait PerDomainTestTrait {

  /**
   * The generated field name.
   *
   * @var ?string
   */
  protected ?string $fieldName;

  /**
   * The generated entity display.
   *
   * @var ?\Drupal\Core\Entity\Display\EntityViewDisplayInterface
   */
  protected ?EntityViewDisplayInterface $display;

  /**
   * The generated entity form display.
   *
   * @var ?\Drupal\Core\Entity\Display\EntityFormDisplayInterface
   */
  protected ?EntityFormDisplayInterface $formDisplay;

  /**
   * Useful as a data provider for tests.
   */
  public static function providePerDomainFieldData(): array {
    $data = [];

    // Decimal.
    $data[] = [
      'per_domain_decimal',
      'value',
      743.0,
      999.0,
      'per_domain_number',
      'per_domain_number_decimal',
      [],
      '[0][value]',
      [static::class, 'assertFormFieldHasValue'],
      [static::class, 'assertPageTextContainsValue'],
    ];

    // Boolean.
    $data[] = [
      'per_domain_boolean',
      'value',
      1,
      0,
      'per_domain_boolean_checkbox',
      'per_domain_boolean',
      [
        'format' => 'custom',
        'format_custom_false' => 'Unique string meaning false',
        'format_custom_true' => 'Unique string meaning true',
      ],
      '[value]',
      [static::class, 'assertCheckboxCheckedIfValueIsTruthy'],
      [static::class, 'assertPageTextContainsBooleanValue'],
    ];

    return $data;
  }

  /**
   * Callable to assert that a number of a given value appears on the form.
   */
  public static function assertFormFieldHasValue(BrowserTestBase $test_class, string $value_key, mixed $expected_value, string $message = '') {
    $field = $test_class->assertSession()->fieldExists($value_key);
    $test_class->assertNotNull($field);
    $test_class->assertSame((string) $expected_value, $field->getValue(), $message);
  }

  /**
   * Callable to assert that a number of a given value appears on the form.
   */
  public static function assertCheckboxCheckedIfValueIsTruthy(BrowserTestBase $test_class, string $value_key, mixed $expected_value, string $message = '') {
    if ($expected_value) {
      $test_class->assertSession()->checkboxChecked($value_key);
    }
    else {
      $test_class->assertSession()->checkboxNotChecked($value_key);
    }
  }

  /**
   * Callable to assert that a number of a given value appears on the view page.
   */
  public static function assertPageTextContainsValue(BrowserTestBase $test_class, string $value_key, mixed $expected_value) {
    $test_class->assertSession()->pageTextContains((string) $expected_value);
  }

  /**
   * Callable to assert that a number of a given value appears on the view page.
   */
  public static function assertPageTextContainsBooleanValue(BrowserTestBase $test_class, string $value_key, mixed $expected_value) {
    $expected_value = $expected_value ? 'Unique string meaning true' : 'Unique string meaning false';
    $test_class->assertSession()->pageTextContains((string) $expected_value);
  }

  /**
   * Helper to create all configuration for a field.
   */
  protected function createField(
    string $entity_type_id,
    string $bundle_id,
    string $field_type,
    int $cardinality = 1,
    array $storage_settings = [],
    array $instance_settings = [],
    ?string $widget_id = NULL,
    array $widget_settings = [],
    ?string $formatter_id = NULL,
    array $formatter_settings = [],
  ) {
    $field_storage = FieldStorageConfig::create([
      'field_name' => $this->fieldName,
      'entity_type' => $entity_type_id,
      'type' => $field_type,
      'settings' => $storage_settings,
      'cardinality' => $cardinality,
    ]);
    $field_storage->save();

    $field = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => $bundle_id,
      'label' => $this->randomMachineName(),
      'settings' => $instance_settings,
    ]);
    $field->save();

    /** @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface $entityDisplayRepository */
    $entityDisplayRepository = $this->container->get('entity_display.repository');

    if (isset($widget_id)) {
      $this->formDisplay = $entityDisplayRepository
        ->getFormDisplay($entity_type_id, $bundle_id, 'default');
      $this->formDisplay->setComponent($this->fieldName, [
        'type' => $widget_id,
        'settings' => $widget_settings,
      ]);
      $this->formDisplay->save();
    }

    if (isset($formatter_id)) {
      $this->display = $entityDisplayRepository
        ->getViewDisplay($entity_type_id, $bundle_id, 'default');
      $this->display->setComponent($this->fieldName, [
        'type' => $formatter_id,
        'settings' => $formatter_settings,
      ]);
      $this->display->save();
    }
  }

  /**
   * Create a new domain record.
   */
  protected function createDomain(string $subdomain, string $name): DomainInterface {
    /** @var \Drupal\domain\DomainStorageInterface $storage */
    $storage = \Drupal::entityTypeManager()->getStorage('domain');

    /** @var \Drupal\domain\DomainInterface $domain */
    $hostname = $storage->createHostname();
    $domain = $storage->create([
      'hostname' => $subdomain . '.' . $hostname,
      'name' => $name,
      'id' => $storage->createMachineName($subdomain . '.' . $hostname),
    ]);

    $domain->save();

    return $domain;
  }

}
