<?php

namespace Drupal\Tests\per_domain_fields\Kernel;

use Drupal\domain\DomainInterface;
use Drupal\entity_test\Entity\EntityTestMul;
use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\per_domain_fields\Traits\PerDomainTestTrait;

/**
 * Test Per-Domain fields.
 */
class PerDomainFieldsTest extends KernelTestBase {
  use PerDomainTestTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'content_translation',
    'domain',
    'entity_test',
    'field',
    'language',
    'per_domain_fields',
    'system',
    'text',
    'user',
  ];

  /**
   * {@inheritdoc}
   */
  protected $strictConfigSchema = FALSE;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    if (\Drupal::entityTypeManager()->hasDefinition('path_alias')) {
      $this->installEntitySchema('path_alias');
    }
    $this->installConfig(['system']);
    $this->installConfig(['field']);
    $this->installConfig(['text']);
    $this->installEntitySchema('entity_test');
    $this->installEntitySchema('entity_test_mul');

    $this->fieldName = mb_strtolower($this->randomMachineName());
    $this->container->get('content_translation.manager')->setEnabled('entity_test_mul', 'entity_test_mul', TRUE);
  }

  /**
   * Test.
   *
   * @dataProvider providePerDomainFieldData
   */
  public function testOrdinarySettingAndGetting(string $field_type, string $main_property_name, mixed $value) {
    $one = $this->createDomain('one', 'One');
    $this->createDomain('two', 'Two');
    $this->container->get('domain.negotiator')->setActiveDomain($one);

    $this->createField('entity_test_mul', 'entity_test_mul', $field_type);

    $construction_strategies = $this->getOrdinaryConstructionStrategyCallbacks($main_property_name, $value);

    // Changing the order of these types of property lookups changes the code
    // paths used to fetch them, so repeat with all permutations.
    $expected_actual_lookups = $this->getExpectedActualLookupCallbacks($main_property_name);

    foreach ($this->permuteItems(array_keys($expected_actual_lookups)) as $ordered_keys) {
      $ordered_keys_string = implode(' ', $ordered_keys);
      foreach ($construction_strategies as $construction_label => $construction_callback) {
        $entity = $construction_callback();
        foreach ($ordered_keys as $lookup_key) {
          [$expected, $actual] = $expected_actual_lookups[$lookup_key]($value, $entity);
          $this->assertSame($expected, $actual, sprintf(
            'Construction "%s" set value "%s" on %s field property %s. Lookup assertion %d in order "%s".',
            $construction_label,
            $value,
            $field_type,
            $main_property_name,
            $lookup_key,
            $ordered_keys_string,
          ));
        }
      }
    }
  }

  /**
   * Test.
   *
   * @dataProvider providePerDomainFieldData
   */
  public function testDomainSettingAndGetting(
    string $field_type,
    string $main_property_name,
    mixed $domain_one_value,
    mixed $domain_two_value,
  ) {
    /** @var \Drupal\domain\DomainNegotiatorInterface $domain_negotiator */
    $domain_negotiator = $this->container->get('domain.negotiator');

    $one = $this->createDomain('one', 'One');
    $two = $this->createDomain('two', 'Two');

    $this->createField('entity_test_mul', 'entity_test_mul', $field_type);

    $construction_strategies = $this->getDomainAwareConstructionStrategyCallbacks($main_property_name, $one, $domain_one_value, $two, $domain_two_value);

    // Changing the order of these types of property lookups changes the code
    // paths used to fetch them, so repeat with all permutations.
    $expected_actual_lookups = $this->getExpectedActualLookupCallbacks($main_property_name);

    foreach ($this->permuteItems(array_keys($expected_actual_lookups)) as $ordered_keys) {
      $ordered_keys_string = implode(' ', $ordered_keys);
      foreach ($construction_strategies as $construction_label => $construction_callback) {
        $entity = $construction_callback();
        foreach ($ordered_keys as $lookup_key) {
          foreach ([[$one, $domain_one_value], [$two, $domain_two_value]] as $index => [$domain, $domain_value]) {
            $domain_negotiator->setActiveDomain($domain);
            [$expected, $actual] = $expected_actual_lookups[$lookup_key]($domain_value, $entity);
            $this->assertSame($expected, $actual, sprintf(
              'Construction "%s" set value "%s" for domain %d on %s field property %s. Lookup assertion %d in order "%s".',
              $construction_label,
              $domain_one_value,
              $index + 1,
              $field_type,
              $main_property_name,
              $lookup_key,
              $ordered_keys_string,
            ));
          }
        }
      }
    }
  }

  /**
   * Callbacks implementing various techniques for creating an entity w/ field.
   */
  private function getOrdinaryConstructionStrategyCallbacks(string $main_property_name, mixed $value): array {
    return [
      'Inline Single' => (fn() => EntityTestMul::create([$this->fieldName => $value])),
      'Inline Delta Single' => (fn() => EntityTestMul::create([$this->fieldName => [0 => $value]])),
      'Inline Prop' => (fn() => EntityTestMul::create([$this->fieldName => [$main_property_name => $value]])),
      'Inline Delta Prop' => (fn() => EntityTestMul::create([$this->fieldName => [0 => [$main_property_name => $value]]])),
      'Setter Single' => (function () use ($value) {
        $entity = EntityTestMul::create();
        $entity->set($this->fieldName, $value);
        return $entity;
      }),
      'Setter Delta Single' => (function () use ($value) {
        $entity = EntityTestMul::create();
        $entity->set($this->fieldName, [0 => $value]);
        return $entity;
      }),
      'Setter Prop' => (function () use ($main_property_name, $value) {
        $entity = EntityTestMul::create();
        $entity->set($this->fieldName, [$main_property_name => $value]);
        return $entity;
      }),
      'Setter Delta Prop' => (function () use ($main_property_name, $value) {
        $entity = EntityTestMul::create();
        $entity->set($this->fieldName, [0 => [$main_property_name => $value]]);
        return $entity;
      }),
    ];
  }

  /**
   * Callbacks implementing various techniques for creating an entity w/ field.
   */
  private function getDomainAwareConstructionStrategyCallbacks(
    string $main_property_name,
    DomainInterface $one,
    mixed $domain_one_value,
    DomainInterface $two,
    mixed $domain_two_value,
  ): array {
    return [
      'Inline Full' => (fn() => EntityTestMul::create([
        $this->fieldName => [
          0 => [$main_property_name => [$one->id() => $domain_one_value, $two->id() => $domain_two_value]],
        ],
      ])),
      'Inline Single' => (fn() => EntityTestMul::create([
        $this->fieldName => [$main_property_name => [$one->id() => $domain_one_value, $two->id() => $domain_two_value]],
      ])),
      'Setter Full' => (function () use ($main_property_name, $one, $domain_one_value, $two, $domain_two_value) {
        $entity = EntityTestMul::create();
        $entity->set($this->fieldName, [
          0 => [$main_property_name => [$one->id() => $domain_one_value, $two->id() => $domain_two_value]],
        ]);
        return $entity;
      }),
      'Setter Single' => (function () use ($main_property_name, $one, $domain_one_value, $two, $domain_two_value) {
        $entity = EntityTestMul::create();
        $entity->set($this->fieldName, [
          $main_property_name => [$one->id() => $domain_one_value, $two->id() => $domain_two_value],
        ]);
        return $entity;
      }),
    ];
  }

  /**
   * Callbacks implementing various techniques for looking-up a field value.
   *
   * @param string $main_property_name
   *   I.e. 'value'.
   *
   * @return \Closure[]
   *   Accessor callbacks.
   */
  private function getExpectedActualLookupCallbacks(string $main_property_name): array {
    return [
      (fn($value, $entity) => [[0 => [$main_property_name => $value]], $entity->{$this->fieldName}->getValue()]),
      (fn($value, $entity) => [$value, $entity->{$this->fieldName}->{$main_property_name}]),
      (fn($value, $entity) => [[$main_property_name => $value], $entity->{$this->fieldName}->get(0)->getValue()]),
      (fn($value, $entity) => [$value, $entity->get($this->fieldName)->get(0)->get($main_property_name)->getValue()]),
      (fn($value, $entity) => [$value, $entity->{$this->fieldName}->{$main_property_name}]),
    ];
  }

  /**
   * Returns an array that contains all permutations of the input array.
   */
  private function permuteItems(array $items, array $perms = []) {
    if (empty($items)) {
      return [$perms];
    }

    $result = [];

    for ($i = count($items) - 1; $i >= 0; --$i) {
      $new_items = $items;
      $new_perms = $perms;
      [$foo] = array_splice($new_items, $i, 1);
      array_unshift($new_perms, $foo);
      $result = array_merge($result, $this->permuteItems($new_items, $new_perms));
    }

    return $result;
  }

}
