<?php

namespace Drupal\per_domain_fields\Plugin\Field\FieldType;

use Drupal\Core\Field\Attribute\FieldType;
use Drupal\Core\Field\Plugin\Field\FieldType\DecimalItem;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Defines the `per_domain_decimal` type.
 */
#[FieldType(
  id: "per_domain_decimal",
  label: new TranslatableMarkup("Number (decimal) [Per-domain]"),
  description: [
    new TranslatableMarkup("Ideal for exact counts and measures (prices, temperatures, distances, volumes, etc.)"),
    new TranslatableMarkup("Stores a number in the database in a fixed decimal format"),
    new TranslatableMarkup("For example, 12.34 km or € when used for further detailed calculations (such as summing many of these)"),
    new TranslatableMarkup("Secretly stores a value per-domain, but hides this on the front-end, showing the value that corresponds to the Active domain only except on edit forms."),
  ],
  category: "number",
  weight: -30,
  default_widget: "per_domain_number",
  default_formatter: "per_domain_number_decimal"
)]
class PerDomainDecimalItem extends DecimalItem {
  use PerDomainFieldTypeTrait{
    PerDomainFieldTypeTrait::preSave as preSaveGeneral;
    PerDomainFieldTypeTrait::isEmpty as isEmptyGeneral;
  }

  /**
   * {@inheritdoc}
   */
  public function preSave() {
    $this->preSaveGeneral();

    $value = $this->get('value')->getValueForAllDomains();

    foreach ($value as $domain_id => $domain_value) {
      $value[$domain_id] = round($domain_value ?: 0, $this->getSetting('scale'));
    }

    $this->get('value')->setValueForAllDomains($value);
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty() {
    if ($this->isEmptyGeneral()) {
      return TRUE;
    }

    $domain_values = $this->get('value')->getValueForAllDomains();

    if (empty($domain_values) || !is_array($domain_values)) {
      return TRUE;
    }

    $domain_values = array_filter($domain_values, fn($v) => !empty($v) && (string) $v !== '0');

    return empty($domain_values);
  }

}
