<?php

namespace Drupal\per_domain_fields;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Shared code for loading all domain records with static caching.
 */
trait LoadsDomainsTrait {

  /**
   * Internal static cache of loaded domains.
   */
  private static ?array $domainsCache = NULL;

  /**
   * Entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Loads all domain records.
   *
   * @return array|\Drupal\Core\Entity\EntityInterface[]
   *   Array of Domain config entities.
   */
  public function loadDomains() {
    if (self::$domainsCache !== NULL) {
      return self::$domainsCache;
    }

    try {
      self::$domainsCache = $this->entityTypeManager->getStorage('domain')->loadMultiple();
      ksort(self::$domainsCache);
    }
    catch (PluginNotFoundException | InvalidPluginDefinitionException $e) {
      self::$domainsCache = [];
    }

    return self::$domainsCache;
  }

  /**
   * Resets our internal static cache of loaded domains.
   */
  public static function resetDomainsCache(): void {
    self::$domainsCache = NULL;
  }

}
