<?php

namespace Drupal\per_domain_fields\Plugin\DataType;

use Drupal\Component\Assertion\Inspector;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\Attribute\DataType;
use Drupal\Core\TypedData\Plugin\DataType\Any;
use Drupal\per_domain_fields\LoadsOverridableActiveDomainIdTrait;

/**
 * The "per_domain_any" data type.
 */
#[DataType(
  id: "per_domain_any",
  label: new TranslatableMarkup("Any data (per-domain)")
)]
class PerDomainAny extends Any implements PerDomainTypedDataInterface {
  use LoadsOverridableActiveDomainIdTrait;

  /**
   * The data value.
   *
   * @var mixed
   */
  protected $value;

  /**
   * {@inheritDoc}
   */
  public function getValueForAllDomains(): array {
    return parent::getValue() ?? [];
  }

  /**
   * {@inheritDoc}
   */
  public function getValue() {
    $value = parent::getValue();

    if (!is_array($value) || Inspector::assertStrictArray($value)) {
      return $value;
    }

    return $value[$this->getActiveDomainId()] ?? reset($value);
  }

  /**
   * {@inheritDoc}
   */
  public function setValueForAllDomains(array $value, $notify = TRUE) {
    parent::setValue($value, $notify);
  }

  /**
   * {@inheritDoc}
   */
  public function setValue($value, $notify = TRUE) {
    // Support setting all domain values in one go.
    if (is_array($value)) {
      parent::setValue($value, $notify);
      return;
    }

    $activeDomainId = $this->getActiveDomainId();

    if (!isset($activeDomainId)) {
      throw new \LogicException(sprintf(
        'Could not set value (%s) on per-domain field %s, as no active domain could be determined.',
        is_scalar($value) ? $value : gettype($value),
        $this->getPropertyPath(),
      ));
    }

    $domain_values = parent::getValue() ?? [];
    $domain_values[$activeDomainId] = $domain_values;

    parent::setValue($domain_values, $notify);
  }

}
