<?php

declare(strict_types=1);

namespace Drupal\Tests\per_domain_fields\Functional;

use Drupal\node\NodeInterface;
use Drupal\Tests\domain\Functional\DomainTestBase;
use Drupal\Tests\per_domain_fields\Traits\PerDomainTestTrait;

/**
 * Kernel tests for Per-domain fields.
 */
class PerDomainFieldTest extends DomainTestBase {
  use PerDomainTestTrait;

  /**
   * {@inheritdoc}
   */
  protected $profile = 'standard';

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'block',
    'datetime',
    'domain',
    'field',
    'node',
    'per_domain_fields',
  ];

  /**
   * {@inheritdoc}
   */
  protected $strictConfigSchema = FALSE;

  /**
   * A normal logged in user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $webUser;

  /**
   * A user with permission to bypass content access checks.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * The node storage.
   *
   * @var \Drupal\node\NodeStorageInterface
   */
  protected $nodeStorage;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->webUser = $this->drupalCreateUser([
      'edit own page content',
      'create page content',
    ]);
    $this->adminUser = $this->drupalCreateUser([
      'bypass node access',
      'administer nodes',
    ]);

    $this->nodeStorage = $this->container->get('entity_type.manager')->getStorage('node');

    $this->fieldName = mb_strtolower($this->randomMachineName());
  }

  /**
   * Test.
   *
   * @dataProvider providePerDomainFieldData
   */
  public function testEditAndViewPerDomainField(
    string $field_type,
    string $main_property_name,
    mixed $domain_one_value,
    mixed $domain_two_value,
    string $widget_id,
    string $formatter_id,
    array $formatter_settings,
    string $widget_main_property_key,
    callable $assertOnForm,
    callable $assertOnView,
  ) {
    $one = $this->createDomain('one', 'One');
    $two = $this->createDomain('two', 'Two');

    $this->createField(
      'node',
      'page',
      $field_type,
      widget_id: $widget_id,
      formatter_id: $formatter_id,
      formatter_settings: $formatter_settings,
    );

    $this->drupalLogin($this->webUser);

    $title_key = 'title[0][value]';
    $domain_one_value_key = "{$this->fieldName}{$widget_main_property_key}[{$one->id()}]";
    $domain_two_value_key = "{$this->fieldName}{$widget_main_property_key}[{$two->id()}]";

    // Create node to edit.
    $edit = [];
    $edit[$title_key] = $this->randomMachineName(8);
    $edit[$domain_one_value_key] = $domain_one_value;
    $edit[$domain_two_value_key] = $domain_two_value;
    $this->drupalGet('node/add/page');
    $this->submitForm($edit, 'Save');

    $node = $this->drupalGetNodeByTitle($edit[$title_key]);
    $this->assertInstanceOf(NodeInterface::class, $node);

    // Verify value was created in the database correctly.
    $query = $this->database->select("node__{$this->fieldName}", 'f');
    $query->fields('f', ["{$this->fieldName}_{$main_property_name}"]);
    $query->condition('f.entity_id', $node->id());
    $results = $query->execute()->fetchAll(\PDO::FETCH_ASSOC);
    $this->assertCount(1, $results);
    $this->assertSame(
      [
        0 => [
          "{$this->fieldName}_{$main_property_name}" => serialize(
            [
              $one->id() => $domain_one_value,
              $two->id() => $domain_two_value,
            ],
          ),
        ],
      ],
      $results,
      'Domain values stored in database in correct format.',
    );

    // Reload edit form and verify values appear on it.
    $this->drupalGet('node/' . $node->id() . '/edit');
    $assertOnForm($this, $domain_one_value_key, $domain_one_value, 'Domain One value was saved.');
    $assertOnForm($this, $domain_two_value_key, $domain_two_value, 'Domain Two value was saved.');

    // Verify active value only is displayed on the front-end.
    $this->drupalGet($one->getPath() . 'node/' . $node->id());
    $assertOnView($this, $domain_one_value_key, $domain_one_value);
    $this->drupalGet($two->getPath() . 'node/' . $node->id());
    $assertOnView($this, $domain_two_value_key, $domain_two_value);
  }

}
