<?php

namespace Drupal\permission_search\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\user\PermissionHandlerInterface;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

class PermissionSearchForm extends FormBase {

  public function getFormId(): string {
    return 'permission_search_form';
  }

  public function __construct(
    protected PermissionHandlerInterface $permissionHandler,
    protected ModuleExtensionList $extensionList,
    protected ModuleHandlerInterface $moduleHandler,
  ) {}

  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('user.permissions'),
      $container->get('extension.list.module'),
      $container->get('module_handler'),
    );
  }

  public function buildForm(array $form, FormStateInterface $form_state): array {
    $options = $this->buildModuleOptions();
    $matches = $form_state->get('perm_matches') ?? [];

    $form['intro'] = [
      '#markup' => '<p>Type to search permissions or select a module to open its permissions.</p>',
    ];

    $form['q'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Search permissions'),
      '#default_value' => (string) ($form_state->getValue('q') ?? ''),
      '#attributes' => ['placeholder' => $this->t('e.g. administer *')],
      '#description' => $this->t('Use a mask (supports * and ?) to match permission ID, title or description. Examples: <code>administer *</code>, <code>*content?</code>, <code>delete</code>.'),
    ];

    $form['module'] = [
      '#type' => 'select',
      '#title' => $this->t('Module'),
      '#options' => $options,
      '#description' => $this->t('If you select a module you will be redirected to this module permissions page.'),
      '#empty_option' => $this->t('- Select -'),
    ];

    $form['actions']['go'] = [
      '#type' => 'submit',
      '#value' => $this->t('Find permissions'),
    ];

    if ($matches) {
      $form['results'] = [
        '#type' => 'details',
        '#title' => $this->t('Search results (@count)', ['@count' => count($matches)]),
        '#open' => TRUE,
      ];
      $form['results']['table'] = $this->buildResultsTable($matches);
    }

    $form['#cache'] = [
      'tags' => ['extension:list'],
      'max-age' => 0,
    ];

    return $form;
  }

  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $mask = trim((string) $form_state->getValue('q'));
    $module = (string) $form_state->getValue('module');

    if ($mask !== '') {
      // Search mode.
      $matches = $this->filterPermissionsByMask($mask);
      if (count($matches) === 0) {
        $this->messenger()->addStatus($this->t('No permissions matched your query.'));
      }
      $form_state->set('perm_matches', $matches);
      $form_state->setRebuild();
      return;
    }

    if ($module && $module !== '__orphans__') {
      // Redirect to module page.
      $form_state->setRedirect('user.admin_permissions.module', ['modules' => $module]);
      return;
    }

    // Nothing entered or selected.
    $this->messenger()->addWarning($this->t('Please enter a search term or select a module.'));
    $form_state->setRebuild();
  }

  // ------------------------------------------------------------------------
  // Helpers
  // ------------------------------------------------------------------------

  protected function buildResultsTable(array $matches): array {
    $header = [
      $this->t('Permission ID'),
      $this->t('Title'),
      $this->t('Description'),
      $this->t('Provider'),
      $this->t('Actions'),
    ];

    $rows = [];
    foreach ($matches as $perm_id => $def) {
      $idStr = (string) $perm_id;
      $title = $this->toText($def['title'] ?? '');
      $desc  = $this->toText($def['description'] ?? '');
      $prov  = $this->toText($def['provider'] ?? '');

      $url = $prov
        ? Url::fromRoute('user.admin_permissions.module', ['modules' => $prov])
        : Url::fromRoute('user.admin_permissions');

      $rows[] = [
        ['data' => ['#plain_text' => $idStr]],
        ['data' => ['#markup' => $title !== '' ? $title : '—']],
        ['data' => ['#markup' => $desc !== '' ? $desc : '—']],
        ['data' => ['#plain_text' => $prov !== '' ? $prov : '—']],
        [
          'data' => [
            '#type' => 'link',
            '#title' => $this->t('Open module'),
            '#url' => $url,
            '#attributes' => ['class' => ['button', 'button--small']],
          ],
        ],
      ];
    }

    return [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No permissions matched your query.'),
    ];
  }

  protected function buildModuleOptions(): array {
    $providers = [];
    foreach ($this->permissionHandler->getPermissions() as $def) {
      if (!empty($def['provider'])) {
        $providers[(string) $def['provider']] = TRUE;
      }
    }

    $core = [];
    $other = [];
    foreach ($this->extensionList->getList() as $name => $ext) {
      if (!$this->moduleHandler->moduleExists($name) || empty($providers[$name])) {
        continue;
      }
      $label = sprintf('%s (%s)', (string) ($ext->info['name'] ?? $name), $name);
      $path = (string) $ext->getPath();
      if (str_starts_with($path, 'core/modules')) {
        $core[$name] = $label;
      } else {
        $other[$name] = $label;
      }
    }
    natcasesort($core);
    natcasesort($other);

    $options = [];
    if ($core) {
      $options[(string) $this->t('Core modules')] = $core;
    }
    if ($other) {
      $options[(string) $this->t('Contrib & custom modules')] = $other;
    }

    return $options;
  }

  protected function filterPermissionsByMask(string $mask): array {
    $all = $this->permissionHandler->getPermissions();
    $mask = trim($mask);
    $regex = NULL;
    if (strpbrk($mask, '*?') !== FALSE) {
      $q = preg_quote($mask, '/');
      $regex = '/^' . str_replace(['\*', '\?'], ['.*', '.'], $q) . '$/i';
    }

    $matches = [];
    foreach ($all as $id => $def) {
      $title = $this->toText($def['title'] ?? '');
      $desc  = $this->toText($def['description'] ?? '');
      $hay   = [$id, $title, $desc];
      $found = FALSE;

      foreach ($hay as $h) {
        if ($regex ? preg_match($regex, $h) : mb_stripos($h, $mask) !== FALSE) {
          $found = TRUE;
          break;
        }
      }
      if ($found) {
        $matches[$id] = $def;
      }
    }

    uasort($matches, fn($a, $b) =>
    strcmp($a['provider'] ?? '', $b['provider'] ?? '') ?: strcmp($a['title'] ?? '', $b['title'] ?? '')
    );

    return $matches;
  }

  protected function toText(mixed $v): string {
    if ($v instanceof \Drupal\Component\Render\MarkupInterface) {
      return (string) $v;
    }
    if (is_scalar($v) || (is_object($v) && method_exists($v, '__toString'))) {
      return (string) $v;
    }
    if (is_array($v)) {
      return implode(' ', array_map(fn($x) => (string) $x, $v));
    }
    return '';
  }

}
