/**
 * @file
 * Permissions Turbo - Advanced permissions management interface.
 *
 * Provides a high-performance, AJAX-powered permissions UI with:
 * - Lazy-loaded provider permissions
 * - Real-time search filtering
 * - Delta-based change tracking
 * - Optimistic UI updates
 * - Keyboard shortcuts (Ctrl+S to save)
 * - Accessible ARIA patterns
 */

(function (Drupal, drupalSettings, once) {
  'use strict';

  /**
   * Permissions Turbo State Manager.
   */
  const PermissionTurbo = {
    /**
     * Permission labels from API.
     * @type {Object|null}
     */
    labels: null,

    /**
     * Cached provider permission data.
     * @type {Map<string, Object>}
     */
    loadedProviders: new Map(),

    /**
     * Tracked permission changes.
     * @type {Map<string, boolean>} Key format: "role:permission"
     */
    changes: new Map(),

    /**
     * Current search query.
     * @type {string}
     */
    searchQuery: '',

    /**
     * Providers manually expanded by user (not by search).
     * @type {Set<string>}
     */
    manuallyExpanded: new Set(),

    /**
     * Providers auto-expanded by search.
     * @type {Set<string>}
     */
    searchExpanded: new Set(),

    /**
     * Main container element.
     * @type {HTMLElement|null}
     */
    container: null,

    /**
     * Save button element.
     * @type {HTMLElement|null}
     */
    saveButton: null,

    /**
     * Search input element.
     * @type {HTMLElement|null}
     */
    searchInput: null,

    /**
     * Initialize the Permissions Turbo interface.
     *
     * @param {HTMLElement} container - Main container element.
     */
    init: function (container) {
      this.container = container;
      this.setupDOM();
      this.bindEvents();
      this.loadLabels();
    },

    /**
     * Setup DOM elements and initial structure.
     */
    setupDOM: function () {
      // Find or create control elements
      this.saveButton = this.container.querySelector('.permission-turbo-save');
      this.searchInput = this.container.querySelector('.permission-turbo-search');
      this.searchButton = this.container.querySelector('.permission-turbo-search-submit');
      this.searchResetButton = this.container.querySelector('.permission-turbo-search-reset');
      this.collapseAllButton = this.container.querySelector('.permission-turbo-collapse-all');
      this.header = this.container.querySelector('.permission-turbo-header');

      // Add ARIA attributes to container
      this.container.setAttribute('role', 'application');
      this.container.setAttribute('aria-label', Drupal.t('Permission management interface'));

      // Setup sticky header detection
      this.setupStickyHeader();
    },

    /**
     * Setup sticky header with JavaScript fallback.
     */
    setupStickyHeader: function () {
      const self = this;
      if (!this.header) {
        return;
      }

      // Get the header's initial position
      const headerRect = this.header.getBoundingClientRect();
      const headerTop = headerRect.top + window.scrollY;
      let isStuck = false;

      // Create placeholder for when header becomes fixed
      const placeholder = document.createElement('div');
      placeholder.className = 'permission-turbo-header-placeholder';
      placeholder.style.display = 'none';
      this.header.parentNode.insertBefore(placeholder, this.header.nextSibling);

      function onScroll() {
        const scrollY = window.scrollY;
        const toolbarOffset = self.getToolbarOffset();

        if (scrollY > headerTop - toolbarOffset) {
          if (!isStuck) {
            isStuck = true;
            placeholder.style.height = self.header.offsetHeight + 'px';
            placeholder.style.display = 'block';
            self.header.classList.add('is-stuck');
            self.header.style.position = 'fixed';
            self.header.style.top = toolbarOffset + 'px';
            self.header.style.left = '0';
            self.header.style.right = '0';
            self.header.style.zIndex = '500';
          }
        } else {
          if (isStuck) {
            isStuck = false;
            placeholder.style.display = 'none';
            self.header.classList.remove('is-stuck');
            self.header.style.position = '';
            self.header.style.top = '';
            self.header.style.left = '';
            self.header.style.right = '';
            self.header.style.zIndex = '';
          }
        }
      }

      window.addEventListener('scroll', onScroll, { passive: true });
      window.addEventListener('resize', onScroll, { passive: true });

      // Initial check
      onScroll();
    },

    /**
     * Get the Drupal toolbar offset.
     */
    getToolbarOffset: function () {
      // Check for Drupal toolbar
      const toolbar = document.getElementById('toolbar-bar');
      const toolbarTray = document.querySelector('.toolbar-tray-horizontal.is-active');
      let offset = 0;

      if (toolbar) {
        offset += toolbar.offsetHeight || 0;
      }
      if (toolbarTray) {
        offset += toolbarTray.offsetHeight || 0;
      }

      return offset;
    },

    /**
     * Bind event handlers.
     */
    bindEvents: function () {
      const self = this;

      // Save button click
      if (this.saveButton) {
        this.saveButton.addEventListener('click', function (e) {
          e.preventDefault();
          self.saveChanges();
        });
      }

      // Collapse all button click
      if (this.collapseAllButton) {
        this.collapseAllButton.addEventListener('click', function (e) {
          e.preventDefault();
          self.collapseAll();
        });
      }

      // Search input with debounce
      if (this.searchInput) {
        let searchTimeout;
        this.searchInput.addEventListener('input', function (e) {
          clearTimeout(searchTimeout);
          searchTimeout = setTimeout(function () {
            self.searchQuery = e.target.value.toLowerCase();
            self.filterProviders();
          }, 300);
        });

        // Enter key triggers search immediately
        this.searchInput.addEventListener('keydown', function (e) {
          if (e.key === 'Enter') {
            e.preventDefault();
            clearTimeout(searchTimeout);
            self.searchQuery = e.target.value.toLowerCase();
            self.filterProviders();
          }
        });
      }

      // Search button click
      if (this.searchButton) {
        this.searchButton.addEventListener('click', function (e) {
          e.preventDefault();
          if (self.searchInput) {
            self.searchQuery = self.searchInput.value.toLowerCase();
            self.filterProviders();
            self.updateResetButtonVisibility();
          }
        });
      }

      // Search reset button click
      if (this.searchResetButton) {
        this.searchResetButton.addEventListener('click', function (e) {
          e.preventDefault();
          self.resetSearch();
        });
      }

      // Keyboard shortcut: Ctrl+S to save
      document.addEventListener('keydown', function (e) {
        if ((e.ctrlKey || e.metaKey) && e.key === 's') {
          e.preventDefault();
          if (self.changes.size > 0) {
            self.saveChanges();
          }
        }
      });

      // Warn on unsaved changes
      window.addEventListener('beforeunload', function (e) {
        if (self.changes.size > 0) {
          const message = Drupal.t('You have unsaved permission changes. Are you sure you want to leave?');
          e.returnValue = message;
          return message;
        }
      });

      // Provider accordion toggle
      this.container.addEventListener('click', function (e) {
        const header = e.target.closest('.permission-turbo-provider-header');
        if (header) {
          e.preventDefault();
          self.toggleProvider(header);
        }
      });

      // Checkbox change tracking
      this.container.addEventListener('change', function (e) {
        if (e.target.classList.contains('permission-turbo-checkbox')) {
          self.handleCheckboxChange(e.target);
        }
      });
    },

    /**
     * Load permission labels from API.
     */
    loadLabels: async function () {
      const self = this;
      const endpoint = drupalSettings.permissionTurbo?.endpoints?.labels || '/api/permission-turbo/labels';
      const url = drupalSettings.path.baseUrl + endpoint.replace(/^\//, '');

      try {
        const response = await fetch(url, {
          method: 'GET',
          headers: {
            'Content-Type': 'application/json',
          },
        });

        if (!response.ok) {
          throw new Error('Failed to load permission labels');
        }

        const result = await response.json();
        if (result.success) {
          self.labels = result.data || {};
          // Render providers after labels are loaded
          self.renderProviders();
        } else {
          throw new Error(result.error || 'Unknown error');
        }
      } catch (error) {
        console.error('Permissions Turbo: Error loading labels', error);
        self.showError(Drupal.t('Failed to load permission labels. Please refresh the page.'));
      }
    },

    /**
     * Render provider accordion structure.
     */
    renderProviders: function () {
      const labels = this.labels || {};
      const roles = drupalSettings.permissionTurbo?.roles || {};
      const accordion = this.container.querySelector('.permission-turbo-accordion');

      if (!accordion) {
        console.error('Permissions Turbo: Accordion container not found');
        return;
      }

      // Clear existing content
      accordion.innerHTML = '';

      // Build provider list from labels data (grouped by provider)
      const providers = Object.keys(labels).sort(function (a, b) {
        const nameA = labels[a].name || a;
        const nameB = labels[b].name || b;
        return nameA.localeCompare(nameB);
      });

      if (providers.length === 0) {
        accordion.innerHTML = '<div class="permission-turbo-empty">' +
          Drupal.t('No permissions found.') + '</div>';
        return;
      }

      // Create provider items
      providers.forEach(function (providerId) {
        const providerData = labels[providerId];
        const permissionCount = Object.keys(providerData.permissions || {}).length;

        const item = document.createElement('div');
        item.className = 'permission-turbo-provider';
        item.setAttribute('data-provider-id', providerId);

        // Create header
        const header = document.createElement('button');
        header.className = 'permission-turbo-provider-header';
        header.setAttribute('type', 'button');
        header.setAttribute('aria-expanded', 'false');
        header.setAttribute('aria-controls', 'provider-content-' + providerId);
        header.innerHTML = '<span class="provider-icon">▶</span> <span class="provider-title">' +
          Drupal.checkPlain(providerData.name || providerId) +
          '</span> <span class="provider-count">(' + permissionCount + ')</span>';

        // Create content container
        const content = document.createElement('div');
        content.className = 'permission-turbo-provider-content';
        content.id = 'provider-content-' + providerId;
        content.setAttribute('role', 'region');
        content.setAttribute('aria-hidden', 'true');
        content.style.display = 'none';
        content.innerHTML = '<div class="loading">' + Drupal.t('Loading...') + '</div>';

        item.appendChild(header);
        item.appendChild(content);
        accordion.appendChild(item);
      });

      this.updateSearchResultCount();
    },

    /**
     * Toggle provider accordion state.
     *
     * @param {HTMLElement} header - Provider header element.
     * @param {boolean} isManual - Whether this is a manual user action (default: true).
     */
    toggleProvider: function (header, isManual) {
      if (typeof isManual === 'undefined') {
        isManual = true;
      }

      const isExpanded = header.getAttribute('aria-expanded') === 'true';
      const providerId = header.closest('.permission-turbo-provider').getAttribute('data-provider-id');
      const contentEl = document.getElementById('provider-content-' + providerId);
      const icon = header.querySelector('.provider-icon');

      if (isExpanded) {
        // Collapse
        header.setAttribute('aria-expanded', 'false');
        contentEl.setAttribute('aria-hidden', 'true');
        contentEl.style.display = 'none';
        icon.textContent = '▶';

        // Track manual collapse
        if (isManual) {
          this.manuallyExpanded.delete(providerId);
          this.searchExpanded.delete(providerId);
        }
      } else {
        // Expand
        header.setAttribute('aria-expanded', 'true');
        contentEl.setAttribute('aria-hidden', 'false');
        contentEl.style.display = 'block';
        icon.textContent = '▼';

        // Track expansion source
        if (isManual) {
          this.manuallyExpanded.add(providerId);
          this.searchExpanded.delete(providerId);
        } else {
          // Only track as search-expanded if not manually expanded
          if (!this.manuallyExpanded.has(providerId)) {
            this.searchExpanded.add(providerId);
          }
        }

        // Lazy load permissions if not already loaded
        if (!this.loadedProviders.has(providerId)) {
          this.loadProviderPermissions(providerId, contentEl);
        }
      }
    },

    /**
     * Collapse a provider programmatically.
     *
     * @param {string} providerId - Provider ID to collapse.
     */
    collapseProvider: function (providerId) {
      const provider = this.container.querySelector('[data-provider-id="' + providerId + '"]');
      if (!provider) { return;
      }

      const header = provider.querySelector('.permission-turbo-provider-header');
      const contentEl = document.getElementById('provider-content-' + providerId);
      const icon = header.querySelector('.provider-icon');

      header.setAttribute('aria-expanded', 'false');
      contentEl.setAttribute('aria-hidden', 'true');
      contentEl.style.display = 'none';
      icon.textContent = '▶';
    },

    /**
     * Collapse all expanded providers.
     */
    collapseAll: function () {
      const self = this;
      const expandedHeaders = this.container.querySelectorAll('.permission-turbo-provider-header[aria-expanded="true"]');

      expandedHeaders.forEach(function (header) {
        const providerId = header.closest('.permission-turbo-provider').getAttribute('data-provider-id');
        self.collapseProvider(providerId);
      });

      // Clear tracking sets
      this.manuallyExpanded.clear();
      this.searchExpanded.clear();
    },

    /**
     * Load provider permissions via AJAX.
     *
     * @param {string} providerId - Provider machine name.
     * @param {HTMLElement} contentEl - Content container element.
     */
    loadProviderPermissions: async function (providerId, contentEl) {
      const self = this;
      const endpointBase = drupalSettings.permissionTurbo?.endpoints?.permissions || '/api/permission-turbo/permissions/';
      const url = drupalSettings.path.baseUrl + endpointBase.replace(/^\//, '') + encodeURIComponent(providerId);

      try {
        const response = await fetch(url, {
          method: 'GET',
          headers: {
            'Content-Type': 'application/json',
          },
        });

        if (!response.ok) {
          throw new Error('Failed to load provider permissions');
        }

        const result = await response.json();
        if (result.success) {
          self.loadedProviders.set(providerId, result.data);
          self.renderProviderPermissions(providerId, contentEl, result.data);
        } else {
          throw new Error(result.error || 'Unknown error');
        }
      } catch (error) {
        console.error('Permissions Turbo: Error loading provider', error);
        contentEl.innerHTML = '<div class="error">' +
          Drupal.t('Failed to load permissions. Please try again.') +
          '</div>';
      }
    },

    /**
     * Render provider permissions as checkbox grid.
     *
     * @param {string} providerId - Provider machine name.
     * @param {HTMLElement} contentEl - Content container element.
     * @param {Object} data - Permission data from API.
     */
    renderProviderPermissions: function (providerId, contentEl, data) {
      const self = this;
      const roles = data.roles || {};
      const permissions = data.permissions || {};
      const roleIds = Object.keys(roles);
      const permissionIds = Object.keys(permissions);

      if (permissionIds.length === 0) {
        contentEl.innerHTML = '<div class="permission-turbo-empty">' +
          Drupal.t('No permissions found for this provider.') +
          '</div>';
        return;
      }

      // Create table structure
      let html = '<table class="permission-turbo-table"><thead><tr>';
      html += '<th class="permission-label">' + Drupal.t('Permission') + '</th>';

      roleIds.forEach(function (roleId) {
        const role = roles[roleId];
        html += '<th class="role-header">' + Drupal.checkPlain(role.label);
        if (role.is_admin) {
          html += ' <small>(' + Drupal.t('Admin') + ')</small>';
        }
        html += '</th>';
      });

      html += '</tr></thead><tbody>';

      // Create rows for each permission
      const searchQuery = this.searchQuery;
      permissionIds.forEach(function (permissionId) {
        const permission = permissions[permissionId];
        const restrictClass = permission.restrict_access ? ' permission-restricted' : '';

        // Apply highlighting if searching
        let titleHtml = self.sanitizeHtml(permission.title);
        let descHtml = permission.description ? self.sanitizeHtml(permission.description) : '';

        if (searchQuery) {
          titleHtml = self.highlightInHtml(titleHtml, searchQuery);
          if (descHtml) {
            descHtml = self.highlightInHtml(descHtml, searchQuery);
          }
        }

        html += '<tr data-permission="' + Drupal.checkPlain(permissionId) + '" class="' + restrictClass + '">';
        html += '<td class="permission-label"><div class="permission-title">' + titleHtml;

        if (permission.restrict_access) {
          html += ' <span class="permission-warning" title="' + Drupal.t('This permission has security implications.') + '">⚠</span>';
        }

        html += '</div>';

        if (descHtml) {
          html += '<div class="permission-description">' + descHtml + '</div>';
        }

        html += '</td>';

        roleIds.forEach(function (roleId) {
          const role = roles[roleId];
          const changeKey = roleId + ':' + permissionId;
          const isChecked = permission.roles && permission.roles[roleId] === true;
          const hasChange = self.changes.has(changeKey);
          const checkedAttr = isChecked ? ' checked' : '';
          const changedClass = hasChange ? ' changed' : '';
          const disabled = role.is_admin ? ' disabled' : '';

          html += '<td class="permission-checkbox' + changedClass + '">';
          html += '<input type="checkbox" class="permission-turbo-checkbox" data-role="' + Drupal.checkPlain(roleId) + '" data-permission="' + Drupal.checkPlain(permissionId) + '" data-original="' + (isChecked ? '1' : '0') + '" ' +
            'value="1"' +
            checkedAttr +
            disabled +
            ' aria-label="' + Drupal.t('Grant @permission to @role', {
              '@permission': permission.title.replace(/<[^>]*>/g, ''),
              '@role': role.label
            }) + '">';
          html += '</td>';
        });

        html += '</tr>';
      });

      html += '</tbody></table>';
      contentEl.innerHTML = html;
    },

    /**
     * Handle checkbox state change.
     *
     * @param {HTMLInputElement} checkbox - Checkbox element.
     */
    handleCheckboxChange: function (checkbox) {
      const role = checkbox.getAttribute('data-role');
      const permission = checkbox.getAttribute('data-permission');
      const changeKey = role + ':' + permission;
      const isChecked = checkbox.checked;
      const originalState = checkbox.getAttribute('data-original') === '1';

      // Track change if different from original
      if (isChecked !== originalState) {
        this.changes.set(changeKey, isChecked);
        checkbox.closest('td').classList.add('changed');
      } else {
        this.changes.delete(changeKey);
        checkbox.closest('td').classList.remove('changed');
      }

      // Update save button state
      this.updateSaveButton();
    },

    /**
     * Update save button state based on changes.
     */
    updateSaveButton: function () {
      if (!this.saveButton) {
        return;
      }

      const changeCount = this.changes.size;

      if (changeCount > 0) {
        this.saveButton.disabled = false;
        this.saveButton.classList.add('has-changes');
        this.saveButton.textContent = Drupal.t('Save @count changes', {
          '@count': changeCount
        });
      } else {
        this.saveButton.disabled = true;
        this.saveButton.classList.remove('has-changes');
        this.saveButton.textContent = Drupal.t('Save changes');
      }
    },

    /**
     * Save permission changes via AJAX.
     */
    saveChanges: async function () {
      const self = this;

      if (this.changes.size === 0) {
        return;
      }

      // Prepare delta data
      const delta = [];
      this.changes.forEach(function (granted, key) {
        const parts = key.split(':');
        delta.push({
          role: parts[0],
          permission: parts[1],
          granted: granted
        });
      });

      // Show saving state
      if (this.saveButton) {
        this.saveButton.disabled = true;
        this.saveButton.textContent = Drupal.t('Saving...');
      }

      const endpointSave = drupalSettings.permissionTurbo?.endpoints?.save || '/api/permission-turbo/save';
      const url = drupalSettings.path.baseUrl + endpointSave.replace(/^\//, '');
      const csrfToken = drupalSettings.permissionTurbo?.csrfToken || '';

      try {
        const response = await fetch(url, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'X-CSRF-Token': csrfToken,
          },
          body: JSON.stringify({ changes: delta })
        });

        const result = await response.json();

        if (!response.ok || !result.success) {
          throw new Error(result.error || 'Failed to save changes');
        }

        // Update original data attributes to reflect saved state
        self.changes.forEach(function (granted, key) {
          const parts = key.split(':');
          const checkbox = self.container.querySelector(
            '.permission-turbo-checkbox[data-role="' + parts[0] + '"][data-permission="' + parts[1] + '"]'
          );
          if (checkbox) {
            checkbox.setAttribute('data-original', granted ? '1' : '0');
          }
        });

        // Clear changes and update UI
        self.changes.clear();
        self.container.querySelectorAll('.changed').forEach(function (el) {
          el.classList.remove('changed');
        });

        self.showMessage(Drupal.t('Permissions saved successfully. @count changes applied.', {
          '@count': result.saved || delta.length
        }), 'success');
        self.updateSaveButton();
      } catch (error) {
        console.error('Permissions Turbo: Error saving changes', error);
        self.showError(error.message || Drupal.t('Failed to save permissions. Please try again.'));

        // Restore save button
        if (self.saveButton) {
          self.updateSaveButton();
        }
      }
    },

    /**
     * Reload provider data after save to sync state.
     */
    reloadProviderData: function () {
      const self = this;
      this.loadedProviders.forEach(function (data, providerId) {
        const contentEl = document.getElementById('provider-content-' + providerId);
        if (contentEl && contentEl.style.display !== 'none') {
          self.loadProviderPermissions(providerId, contentEl);
        }
      });
    },

    /**
     * Filter providers by search query.
     */
    filterProviders: function () {
      const query = this.searchQuery;
      const providers = this.container.querySelectorAll('.permission-turbo-provider');
      let visibleCount = 0;
      const self = this;

      // Collapse all search-expanded providers when search changes
      this.searchExpanded.forEach(function (providerId) {
        self.collapseProvider(providerId);
      });
      this.searchExpanded.clear();

      if (!query) {
        // Show all providers and remove highlighting from loaded ones
        const self = this;
        providers.forEach(function (provider) {
          provider.style.display = 'block';
          visibleCount++;

          // Re-render loaded providers to remove highlighting
          const providerId = provider.getAttribute('data-provider-id');
          const header = provider.querySelector('.permission-turbo-provider-header');
          if (self.loadedProviders.has(providerId) && header.getAttribute('aria-expanded') === 'true') {
            const contentEl = document.getElementById('provider-content-' + providerId);
            if (contentEl) {
              self.renderProviderPermissions(providerId, contentEl, self.loadedProviders.get(providerId));
            }
          }
        });
        this.updateSearchResultCount(visibleCount);
        return;
      }

      // Filter by provider title and loaded permissions
      providers.forEach(function (provider) {
        const providerId = provider.getAttribute('data-provider-id');
        const header = provider.querySelector('.permission-turbo-provider-header');
        const title = header.querySelector('.provider-title').textContent.toLowerCase();

        let matches = title.includes(query);

        // Check loaded permissions if provider is cached
        if (!matches && self.loadedProviders.has(providerId)) {
          const data = self.loadedProviders.get(providerId);
          const permissions = data.permissions || {};

          // Convert object to array for iteration (permissions are keyed by ID)
          matches = Object.values(permissions).some(function (perm) {
            const permTitle = perm.title || '';
            const permDescription = perm.description || '';
            return permTitle.toLowerCase().includes(query) ||
              permDescription.toLowerCase().includes(query);
          });
        }

        // Also check against labels data if provider not yet loaded
        if (!matches && !self.loadedProviders.has(providerId) && self.labels && self.labels[providerId]) {
          const labelData = self.labels[providerId];
          const permissions = labelData.permissions || {};

          matches = Object.values(permissions).some(function (perm) {
            const permTitle = perm.title || '';
            const permDescription = perm.description || '';
            // Strip HTML tags for search comparison
            const cleanDesc = permDescription.replace(/<[^>]*>/g, '');
            return permTitle.toLowerCase().includes(query) ||
              cleanDesc.toLowerCase().includes(query);
          });
        }

        if (matches) {
          provider.style.display = 'block';
          visibleCount++;

          // Auto-expand if searching and not already expanded (not manually expanded)
          if (query && header.getAttribute('aria-expanded') === 'false' && !self.manuallyExpanded.has(providerId)) {
            self.toggleProvider(header, false); // false = not manual, search-triggered
          }

          // Re-render already loaded providers to update highlighting
          if (self.loadedProviders.has(providerId) && header.getAttribute('aria-expanded') === 'true') {
            const contentEl = document.getElementById('provider-content-' + providerId);
            if (contentEl) {
              self.renderProviderPermissions(providerId, contentEl, self.loadedProviders.get(providerId));
            }
          }
        } else {
          provider.style.display = 'none';
        }
      });

      this.updateSearchResultCount(visibleCount);
    },

    /**
     * Update search result count display.
     *
     * @param {number|null} count - Number of visible providers.
     */
    updateSearchResultCount: function (count) {
      const countEl = this.container.querySelector('.permission-turbo-result-count');
      if (!countEl) {
        return;
      }

      if (typeof count === 'undefined') {
        const providers = this.container.querySelectorAll('.permission-turbo-provider');
        count = providers.length;
      }

      if (this.searchQuery) {
        countEl.textContent = Drupal.t('Showing @count results', { '@count': count });
        countEl.style.display = 'block';
      } else {
        countEl.style.display = 'none';
      }

      // Update reset button visibility
      this.updateResetButtonVisibility();
    },

    /**
     * Reset search and restore all providers.
     */
    resetSearch: function () {
      // Clear search input
      if (this.searchInput) {
        this.searchInput.value = '';
      }

      // Clear search query
      this.searchQuery = '';

      // Re-filter (will show all)
      this.filterProviders();

      // Hide reset button
      this.updateResetButtonVisibility();

      // Focus search input
      if (this.searchInput) {
        this.searchInput.focus();
      }
    },

    /**
     * Update reset button visibility based on search state.
     */
    updateResetButtonVisibility: function () {
      if (!this.searchResetButton) {
        return;
      }

      if (this.searchQuery && this.searchQuery.length > 0) {
        this.searchResetButton.style.display = 'inline-block';
      } else {
        this.searchResetButton.style.display = 'none';
      }
    },

    /**
     * Show success message to user.
     *
     * @param {string} message - Message text.
     * @param {string} type - Message type (status, warning, error).
     */
    showMessage: function (message, type) {
      type = type || 'status';
      const messagesContainer = this.container.querySelector('.permission-turbo-messages');

      if (!messagesContainer) {
        // Fallback to Drupal messages
        if (typeof Drupal.announce === 'function') {
          Drupal.announce(message);
        }
        return;
      }

      const messageEl = document.createElement('div');
      messageEl.className = 'permission-turbo-message permission-turbo-message--' + type;
      messageEl.setAttribute('role', type === 'error' ? 'alert' : 'status');
      messageEl.textContent = message;

      messagesContainer.appendChild(messageEl);

      // Auto-remove after 5 seconds
      setTimeout(function () {
        messageEl.style.opacity = '0';
        setTimeout(function () {
          messageEl.remove();
        }, 300);
      }, 5000);
    },

    /**
     * Show error message to user.
     *
     * @param {string} message - Error message text.
     */
    showError: function (message) {
      this.showMessage(message, 'error');
    },

    /**
     * Highlight search terms in text.
     *
     * @param {string} text - Text to highlight within.
     * @param {string} query - Search query to highlight.
     * @return {string} Text with highlighted matches.
     */
    highlightSearchTerm: function (text, query) {
      if (!query || !text) {
        return text;
      }

      // Escape regex special characters in query
      const escapedQuery = query.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
      const regex = new RegExp('(' + escapedQuery + ')', 'gi');

      return text.replace(regex, '<mark class="permission-turbo-highlight">$1</mark>');
    },

    /**
     * Highlight search terms in HTML content safely.
     * Only highlights text nodes, preserving HTML structure.
     *
     * @param {string} html - HTML string to process.
     * @param {string} query - Search query to highlight.
     * @return {string} HTML with highlighted matches in text content.
     */
    highlightInHtml: function (html, query) {
      if (!query || !html) {
        return html;
      }

      // Create a temporary element to parse HTML
      const temp = document.createElement('div');
      temp.innerHTML = html;

      // Escape regex special characters in query
      const escapedQuery = query.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
      const regex = new RegExp('(' + escapedQuery + ')', 'gi');

      // Recursively process text nodes
      function processNode(node) {
        if (node.nodeType === Node.TEXT_NODE) {
          const text = node.textContent;
          if (regex.test(text)) {
            const span = document.createElement('span');
            span.innerHTML = text.replace(regex, '<mark class="permission-turbo-highlight">$1</mark>');
            node.parentNode.replaceChild(span, node);
          }
        } else if (node.nodeType === Node.ELEMENT_NODE && node.tagName !== 'MARK') {
          // Process child nodes (copy to array first as we may modify)
          Array.from(node.childNodes).forEach(processNode);
        }
      }

      processNode(temp);
      return temp.innerHTML;
    },

    /**
     * Sanitize HTML to allow only safe tags.
     *
     * Allows common formatting tags used in Drupal permission descriptions
     * like <em>, <strong>, <code>, <a>, while preventing XSS attacks.
     *
     * @param {string} html - HTML string to sanitize.
     * @return {string} Sanitized HTML string.
     */
    sanitizeHtml: function (html) {
      if (!html || typeof html !== 'string') {
        return '';
      }

      // Use Drupal's filterXss if available
      if (typeof Drupal.filterXss === 'function') {
        return Drupal.filterXss(html);
      }

      // Fallback: Allow only safe tags, strip everything else
      // Create a temporary element to parse HTML
      const temp = document.createElement('div');
      temp.innerHTML = html;

      // List of allowed tags
      const allowedTags = ['em', 'strong', 'code', 'a', 'span', 'b', 'i', 'br', 'small'];

      // Recursively clean the DOM
      function cleanNode(node) {
        const children = Array.from(node.childNodes);
        children.forEach(function (child) {
          if (child.nodeType === Node.ELEMENT_NODE) {
            const tagName = child.tagName.toLowerCase();
            if (!allowedTags.includes(tagName)) {
              // Replace element with its text content
              const text = document.createTextNode(child.textContent);
              node.replaceChild(text, child);
            } else {
              // Remove all attributes except safe ones for <a> tags
              const attrs = Array.from(child.attributes);
              attrs.forEach(function (attr) {
                if (tagName === 'a' && (attr.name === 'href' || attr.name === 'target')) {
                  // Validate href to prevent javascript: URLs
                  if (attr.name === 'href' && attr.value.toLowerCase().startsWith('javascript:')) {
                    child.removeAttribute('href');
                  }
                } else if (attr.name === 'class') {
                  // Allow class attribute for styling (e.g., placeholder)
                } else {
                  child.removeAttribute(attr.name);
                }
              });
              cleanNode(child);
            }
          }
        });
      }

      cleanNode(temp);
      return temp.innerHTML;
    }
  };

  /**
   * Drupal behavior for Permissions Turbo.
   */
  Drupal.behaviors.permissionTurbo = {
    attach: function (context, settings) {
      const containers = once('permission-turbo', '.permission-turbo-container', context);

      containers.forEach(function (container) {
        PermissionTurbo.init(container);
      });
    }
  };

})(Drupal, drupalSettings, once);
