<?php

declare(strict_types=1);

namespace Drupal\permission_turbo\Controller;

use Drupal\Core\Access\CsrfTokenGenerator;
use Drupal\Core\Controller\ControllerBase;
use Drupal\permission_turbo\Service\PermissionDataService;
use Drupal\permission_turbo\Service\PermissionSaveService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for Permissions Turbo pages and API endpoints.
 *
 * Provides a high-performance permission management interface with:
 * - Lazy loading of permission checkboxes
 * - Delta-based saving (only changed permissions)
 * - RESTful API endpoints for AJAX operations.
 */
class PermissionTurboController extends ControllerBase {

  /**
   * The permission data service.
   *
   * @var \Drupal\permission_turbo\Service\PermissionDataService
   */
  protected PermissionDataService $dataService;

  /**
   * The permission save service.
   *
   * @var \Drupal\permission_turbo\Service\PermissionSaveService
   */
  protected PermissionSaveService $saveService;

  /**
   * The CSRF token generator.
   *
   * @var \Drupal\Core\Access\CsrfTokenGenerator
   */
  protected CsrfTokenGenerator $csrfToken;

  /**
   * Constructs a PermissionTurboController object.
   *
   * @param \Drupal\permission_turbo\Service\PermissionDataService $data_service
   *   The permission data service.
   * @param \Drupal\permission_turbo\Service\PermissionSaveService $save_service
   *   The permission save service.
   * @param \Drupal\Core\Access\CsrfTokenGenerator $csrf_token
   *   The CSRF token generator.
   */
  public function __construct(
    PermissionDataService $data_service,
    PermissionSaveService $save_service,
    CsrfTokenGenerator $csrf_token,
  ) {
    $this->dataService = $data_service;
    $this->saveService = $save_service;
    $this->csrfToken = $csrf_token;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('permission_turbo.data'),
      $container->get('permission_turbo.save'),
      $container->get('csrf_token'),
    );
  }

  /**
   * Renders the main permission page.
   *
   * This page provides a high-performance interface for managing permissions
   * with lazy loading and instant search capabilities.
   *
   * @return array
   *   A render array containing the permission management interface.
   */
  public function permissionsPage(): array {
    $roles = $this->dataService->getRoles();

    // Transform roles to simple array for JavaScript.
    $roles_data = [];
    foreach ($roles as $role_id => $role) {
      $roles_data[$role_id] = [
        'id' => $role_id,
        'label' => $role->label(),
        'is_admin' => $role->isAdmin(),
        'weight' => $role->getWeight(),
      ];
    }

    return [
      '#theme' => 'permission_turbo_page',
      '#roles' => $roles_data,
      '#attached' => [
        'library' => ['permission_turbo/permission-turbo'],
        'drupalSettings' => [
          'permissionTurbo' => [
            'endpoints' => [
              'labels' => '/api/permission-turbo/labels',
              'permissions' => '/api/permission-turbo/permissions/',
              'save' => '/api/permission-turbo/save',
            ],
            'roles' => $roles_data,
            'csrfToken' => $this->csrfToken->get('permission_turbo'),
          ],
        ],
      ],
    ];
  }

  /**
   * API endpoint: Get all permission labels.
   *
   * Returns lightweight permission data for the initial page load.
   * This includes permission names and providers but not role states.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response containing permission labels grouped by provider.
   */
  public function getLabels(): JsonResponse {
    try {
      $labels = $this->dataService->getLabels();

      return new JsonResponse([
        'success' => TRUE,
        'data' => $labels,
      ]);
    }
    catch (\Exception $e) {
      $this->getLogger('permission_turbo')->error('Failed to get labels: @message', [
        '@message' => $e->getMessage(),
      ]);

      return new JsonResponse([
        'success' => FALSE,
        'error' => $this->t('Failed to load permission labels.'),
      ], 500);
    }
  }

  /**
   * API endpoint: Get permissions for a specific provider.
   *
   * Returns full permission data including role states for lazy loading.
   *
   * @param string $provider
   *   The module/provider machine name.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response containing permission data with role states.
   */
  public function getPermissions(string $provider): JsonResponse {
    try {
      $permissions = $this->dataService->getProviderPermissions($provider);

      if (empty($permissions)) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => $this->t('Provider not found or has no permissions.'),
        ], 404);
      }

      // Get roles for the response.
      $roles = $this->dataService->getRoles();
      $roles_data = [];
      foreach ($roles as $role_id => $role) {
        $roles_data[$role_id] = [
          'id' => $role_id,
          'label' => $role->label(),
          'is_admin' => $role->isAdmin(),
        ];
      }

      return new JsonResponse([
        'success' => TRUE,
        'data' => [
          'provider' => $provider,
          'permissions' => $permissions,
          'roles' => $roles_data,
        ],
      ]);
    }
    catch (\Exception $e) {
      $this->getLogger('permission_turbo')->error('Failed to get permissions for @provider: @message', [
        '@provider' => $provider,
        '@message' => $e->getMessage(),
      ]);

      return new JsonResponse([
        'success' => FALSE,
        'error' => $this->t('Failed to load permissions.'),
      ], 500);
    }
  }

  /**
   * API endpoint: Save changed permissions.
   *
   * Accepts only the changed permissions (delta) for efficient saving.
   * Requires CSRF token validation for security.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request containing the changes.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with save results.
   */
  public function savePermissions(Request $request): JsonResponse {
    // Validate CSRF token.
    $token = $request->headers->get('X-CSRF-Token');
    if (!$token || !$this->csrfToken->validate($token, 'permission_turbo')) {
      return new JsonResponse([
        'success' => FALSE,
        'error' => $this->t('Invalid CSRF token.'),
      ], 403);
    }

    // Parse request body.
    $content = $request->getContent();
    $data = json_decode($content, TRUE);

    if (json_last_error() !== JSON_ERROR_NONE) {
      return new JsonResponse([
        'success' => FALSE,
        'error' => $this->t('Invalid JSON in request body.'),
      ], 400);
    }

    if (!isset($data['changes']) || !is_array($data['changes'])) {
      return new JsonResponse([
        'success' => FALSE,
        'error' => $this->t('Invalid request format. Expected "changes" array.'),
      ], 400);
    }

    // Validate each change has required fields.
    foreach ($data['changes'] as $index => $change) {
      if (!isset($change['role'], $change['permission'], $change['granted'])) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => $this->t('Change at index @index is missing required fields (role, permission, granted).', [
            '@index' => $index,
          ]),
        ], 400);
      }
    }

    try {
      // Transform changes from [{role, permission, granted}, ...] to {role_id => {permission => bool}}.
      $grouped_changes = [];
      foreach ($data['changes'] as $change) {
        $role_id = $change['role'];
        $permission = $change['permission'];
        $granted = (bool) $change['granted'];

        if (!isset($grouped_changes[$role_id])) {
          $grouped_changes[$role_id] = [];
        }
        $grouped_changes[$role_id][$permission] = $granted;
      }

      // Validate changes against existing permissions and roles.
      $validation = $this->saveService->validateChanges($grouped_changes);
      if (!$validation['valid']) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => $this->t('Validation failed.'),
          'details' => $validation['errors'],
        ], 400);
      }

      // Save changes.
      $result = $this->saveService->saveChanges($grouped_changes);

      // Invalidate caches on success.
      if ($result['success']) {
        $this->dataService->invalidateCache();
      }

      return new JsonResponse($result);
    }
    catch (\Exception $e) {
      $this->getLogger('permission_turbo')->error('Failed to save permissions: @message', [
        '@message' => $e->getMessage(),
      ]);

      return new JsonResponse([
        'success' => FALSE,
        'error' => $this->t('Failed to save permissions. Please try again.'),
      ], 500);
    }
  }

}
