<?php

declare(strict_types=1);

namespace Drupal\permission_turbo\Service;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\user\PermissionHandlerInterface;

/**
 * Service for retrieving permission data efficiently with caching.
 *
 * This service provides optimized methods for fetching permission data
 * with proper caching to improve performance on the permissions UI.
 */
class PermissionDataService {

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected CacheBackendInterface $cache;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The permission handler.
   *
   * @var \Drupal\user\PermissionHandlerInterface
   */
  protected PermissionHandlerInterface $permissionHandler;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected RendererInterface $renderer;

  /**
   * Constructs a new PermissionDataService.
   *
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\user\PermissionHandlerInterface $permission_handler
   *   The permission handler.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   */
  public function __construct(
    CacheBackendInterface $cache,
    EntityTypeManagerInterface $entity_type_manager,
    PermissionHandlerInterface $permission_handler,
    ModuleHandlerInterface $module_handler,
    RendererInterface $renderer,
  ) {
    $this->cache = $cache;
    $this->entityTypeManager = $entity_type_manager;
    $this->permissionHandler = $permission_handler;
    $this->moduleHandler = $module_handler;
    $this->renderer = $renderer;
  }

  /**
   * Get lightweight label data for all permissions grouped by provider.
   *
   * This method returns minimal data needed for the initial page load,
   * containing only permission names and their providers. The full
   * permission data (including role states) is loaded on demand.
   *
   * @return array
   *   An array of permission labels grouped by provider, keyed by permission
   *   machine name. Each entry contains:
   *   - 'title': The human-readable permission title.
   *   - 'provider': The module/provider that defines this permission.
   *   - 'description': Brief description of what the permission allows.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function getLabels(): array {
    $cid = 'permission_turbo:labels';
    $cached = $this->cache->get($cid);

    if ($cached !== FALSE) {
      return $cached->data;
    }

    $labels = [];
    $permissions = $this->permissionHandler->getPermissions();

    foreach ($permissions as $permission_name => $permission) {
      $provider = $permission['provider'] ?? 'unknown';

      if (!isset($labels[$provider])) {
        $labels[$provider] = [
          'name' => $this->getProviderName($provider),
          'permissions' => [],
        ];
      }

      $labels[$provider]['permissions'][$permission_name] = [
        'title' => (string) $permission['title'],
        'provider' => $provider,
        'description' => $this->renderDescription($permission['description'] ?? ''),
      ];
    }

    // Sort providers alphabetically.
    ksort($labels);

    // Sort permissions within each provider.
    foreach ($labels as $provider => $data) {
      ksort($labels[$provider]['permissions']);
    }

    $this->cache->set($cid, $labels, CacheBackendInterface::CACHE_PERMANENT, [
      'permission_turbo',
      'config:user.role',
      'config:core.extension',
    ]);

    return $labels;
  }

  /**
   * Get full permission data with role states for a specific provider.
   *
   * This method returns complete permission data including which roles
   * have each permission assigned. This is loaded on-demand when a
   * provider's permissions are expanded in the UI.
   *
   * @param string $provider
   *   The module/provider machine name.
   *
   * @return array
   *   An array of permissions for the provider with role assignment states.
   *   Each permission contains:
   *   - 'title': The human-readable permission title.
   *   - 'description': Detailed description of the permission.
   *   - 'provider': The module/provider machine name.
   *   - 'roles': Array of role IDs that have this permission, keyed by role ID
   *              with boolean values.
   *   - 'restrict_access': Whether this is a restricted permission.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function getProviderPermissions(string $provider): array {
    $cid = "permission_turbo:provider:$provider";
    $cached = $this->cache->get($cid);

    if ($cached !== FALSE) {
      return $cached->data;
    }

    $provider_permissions = [];
    $permissions = $this->permissionHandler->getPermissions();
    $roles = $this->getRoles();

    // Build role permissions map for efficient lookup.
    $role_permissions = [];
    foreach ($roles as $role_id => $role) {
      $role_permissions[$role_id] = $role->getPermissions();
    }

    // Filter and build permissions for this provider.
    foreach ($permissions as $permission_name => $permission) {
      if (($permission['provider'] ?? '') !== $provider) {
        continue;
      }

      $permission_data = [
        'title' => (string) $permission['title'],
        'description' => $this->renderDescription($permission['description'] ?? ''),
        'provider' => $provider,
        'roles' => [],
        'restrict_access' => !empty($permission['restrict access']),
      ];

      // Check which roles have this permission.
      foreach ($role_permissions as $role_id => $perms) {
        $permission_data['roles'][$role_id] = in_array($permission_name, $perms, TRUE);
      }

      $provider_permissions[$permission_name] = $permission_data;
    }

    // Sort by permission name.
    ksort($provider_permissions);

    $this->cache->set($cid, $provider_permissions, CacheBackendInterface::CACHE_PERMANENT, [
      'permission_turbo',
      'config:user.role',
      'config:core.extension',
    ]);

    return $provider_permissions;
  }

  /**
   * Get all roles for the header.
   *
   * Returns all user roles in the system, ordered by weight.
   * This is used to build the role column headers in the UI.
   *
   * @return array
   *   An array of role entities, keyed by role ID.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function getRoles(): array {
    $cid = 'permission_turbo:roles';
    $cached = $this->cache->get($cid);

    if ($cached !== FALSE) {
      return $cached->data;
    }

    /** @var \Drupal\user\RoleStorageInterface $role_storage */
    $role_storage = $this->entityTypeManager->getStorage('user_role');
    $roles = $role_storage->loadMultiple();

    // Sort roles by weight.
    uasort($roles, function ($a, $b) {
      return $a->getWeight() <=> $b->getWeight();
    });

    $this->cache->set($cid, $roles, CacheBackendInterface::CACHE_PERMANENT, [
      'permission_turbo',
      'config:user.role',
    ]);

    return $roles;
  }

  /**
   * Clear permission turbo caches.
   *
   * This method invalidates all caches used by the permission turbo module,
   * forcing fresh data to be loaded on the next request.
   */
  public function invalidateCache(): void {
    // Use Drupal's Cache API to invalidate tags properly.
    Cache::invalidateTags(['permission_turbo']);
  }

  /**
   * Render a permission description to a string.
   *
   * Handles various input types including strings, TranslatableMarkup objects,
   * and render arrays.
   *
   * @param mixed $description
   *   The description value which may be a string, TranslatableMarkup, or array.
   *
   * @return string
   *   The rendered description as a plain string.
   */
  protected function renderDescription(mixed $description): string {
    if (empty($description)) {
      return '';
    }

    // If it's already a string, return as-is.
    if (is_string($description)) {
      return $description;
    }

    // If it's a TranslatableMarkup or Stringable object.
    if (is_object($description) && method_exists($description, '__toString')) {
      return (string) $description;
    }

    // If it's a render array, try to render it.
    if (is_array($description)) {
      try {
        return (string) $this->renderer->renderPlain($description);
      }
      catch (\Exception $e) {
        // If rendering fails, return empty string.
        return '';
      }
    }

    return '';
  }

  /**
   * Get human-readable provider name.
   *
   * Converts a module machine name to a human-readable label.
   *
   * @param string $provider
   *   The provider machine name.
   *
   * @return string
   *   The human-readable provider name.
   */
  protected function getProviderName(string $provider): string {
    try {
      // Use ModuleHandlerInterface::getName() which is safer and handles
      // edge cases properly.
      if ($this->moduleHandler->moduleExists($provider)) {
        return $this->moduleHandler->getName($provider);
      }
    }
    catch (\Exception $e) {
      // Fall back to provider machine name if any error occurs.
    }

    return $provider;
  }

}
