<?php

declare(strict_types=1);

namespace Drupal\permission_turbo\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\user\PermissionHandlerInterface;

/**
 * Service for saving permission changes.
 *
 * This service handles validation and saving of permission changes
 * with detailed logging and error handling.
 */
class PermissionSaveService {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $currentUser;

  /**
   * The permission handler.
   *
   * @var \Drupal\user\PermissionHandlerInterface
   */
  protected PermissionHandlerInterface $permissionHandler;

  /**
   * The permission data service.
   *
   * @var \Drupal\permission_turbo\Service\PermissionDataService
   */
  protected PermissionDataService $permissionDataService;

  /**
   * Constructs a new PermissionSaveService.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Drupal\user\PermissionHandlerInterface $permission_handler
   *   The permission handler.
   * @param \Drupal\permission_turbo\Service\PermissionDataService $permission_data_service
   *   The permission data service.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    LoggerChannelFactoryInterface $logger_factory,
    AccountProxyInterface $current_user,
    PermissionHandlerInterface $permission_handler,
    PermissionDataService $permission_data_service,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->logger = $logger_factory->get('permission_turbo');
    $this->currentUser = $current_user;
    $this->permissionHandler = $permission_handler;
    $this->permissionDataService = $permission_data_service;
  }

  /**
   * Save only changed permissions with detailed results.
   *
   * This method processes permission changes grouped by role for efficiency.
   * It validates all changes before saving and provides detailed results
   * about what was changed.
   *
   * @param array $changes
   *   An array of permission changes in the format:
   *   [
   *     'role_id' => [
   *       'permission_name' => TRUE/FALSE,
   *     ],
   *   ]
   *   Where TRUE means grant permission, FALSE means revoke permission.
   *
   * @return array
   *   An array containing:
   *   - 'success': Boolean indicating overall success.
   *   - 'saved': Number of successfully saved changes.
   *   - 'errors': Array of error messages.
   *   - 'changes': Detailed array of changes made, grouped by role.
   *   - 'skipped': Number of changes skipped (no actual change).
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function saveChanges(array $changes): array {
    $result = [
      'success' => TRUE,
      'saved' => 0,
      'errors' => [],
      'changes' => [],
      'skipped' => 0,
    ];

    // Validate changes first.
    $validation = $this->validateChanges($changes);
    if (!$validation['valid']) {
      $result['success'] = FALSE;
      $result['errors'] = $validation['errors'];
      return $result;
    }

    /** @var \Drupal\user\RoleStorageInterface $role_storage */
    $role_storage = $this->entityTypeManager->getStorage('user_role');

    // Group changes by role for efficient processing.
    foreach ($changes as $role_id => $permissions) {
      if (empty($permissions)) {
        continue;
      }

      // Load the role.
      /** @var \Drupal\user\RoleInterface|null $role */
      $role = $role_storage->load($role_id);

      if (!$role) {
        $result['errors'][] = "Role '$role_id' not found.";
        $result['success'] = FALSE;
        continue;
      }

      $role_changes = [
        'granted' => [],
        'revoked' => [],
      ];

      foreach ($permissions as $permission_name => $grant) {
        $has_permission = $role->hasPermission($permission_name);

        // Skip if there's no actual change.
        if (($grant && $has_permission) || (!$grant && !$has_permission)) {
          $result['skipped']++;
          continue;
        }

        try {
          if ($grant) {
            $role->grantPermission($permission_name);
            $role_changes['granted'][] = $permission_name;
            $this->logger->info(
              'Granted permission "@permission" to role "@role" by user @user',
              [
                '@permission' => $permission_name,
                '@role' => $role->label(),
                '@user' => $this->currentUser->getAccountName(),
              ]
            );
          }
          else {
            $role->revokePermission($permission_name);
            $role_changes['revoked'][] = $permission_name;
            $this->logger->info(
              'Revoked permission "@permission" from role "@role" by user @user',
              [
                '@permission' => $permission_name,
                '@role' => $role->label(),
                '@user' => $this->currentUser->getAccountName(),
              ]
            );
          }

          $result['saved']++;
        }
        catch (\Exception $e) {
          $result['errors'][] = sprintf(
            'Error %s permission "%s" for role "%s": %s',
            $grant ? 'granting' : 'revoking',
            $permission_name,
            $role->label(),
            $e->getMessage()
          );
          $result['success'] = FALSE;
          $this->logger->error(
            'Failed to @action permission "@permission" for role "@role": @error',
            [
              '@action' => $grant ? 'grant' : 'revoke',
              '@permission' => $permission_name,
              '@role' => $role->label(),
              '@error' => $e->getMessage(),
            ]
          );
        }
      }

      // Save the role if changes were made.
      if (!empty($role_changes['granted']) || !empty($role_changes['revoked'])) {
        try {
          $role->save();
          $result['changes'][$role_id] = $role_changes;
        }
        catch (\Exception $e) {
          $result['errors'][] = sprintf(
            'Error saving role "%s": %s',
            $role->label(),
            $e->getMessage()
          );
          $result['success'] = FALSE;
          $this->logger->error(
            'Failed to save role "@role": @error',
            [
              '@role' => $role->label(),
              '@error' => $e->getMessage(),
            ]
          );
        }
      }
    }

    // Invalidate cache after successful changes.
    if ($result['saved'] > 0) {
      $this->permissionDataService->invalidateCache();
      $this->logger->info(
        'Saved @count permission changes by user @user',
        [
          '@count' => $result['saved'],
          '@user' => $this->currentUser->getAccountName(),
        ]
      );
    }

    return $result;
  }

  /**
   * Validate changes before saving.
   *
   * This method performs validation checks on the provided changes
   * to ensure they are valid before attempting to save them.
   *
   * @param array $changes
   *   An array of permission changes in the format:
   *   [
   *     'role_id' => [
   *       'permission_name' => TRUE/FALSE,
   *     ],
   *   ].
   *
   * @return array
   *   An array containing:
   *   - 'valid': Boolean indicating if validation passed.
   *   - 'errors': Array of validation error messages.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function validateChanges(array $changes): array {
    $result = [
      'valid' => TRUE,
      'errors' => [],
    ];

    if (empty($changes)) {
      $result['valid'] = FALSE;
      $result['errors'][] = 'No changes provided.';
      return $result;
    }

    // Get all valid permissions and roles.
    $valid_permissions = array_keys($this->permissionHandler->getPermissions());
    /** @var \Drupal\user\RoleStorageInterface $role_storage */
    $role_storage = $this->entityTypeManager->getStorage('user_role');
    $valid_roles = array_keys($role_storage->loadMultiple());

    foreach ($changes as $role_id => $permissions) {
      // Validate role ID.
      if (!in_array($role_id, $valid_roles, TRUE)) {
        $result['valid'] = FALSE;
        $result['errors'][] = "Invalid role ID: '$role_id'.";
        continue;
      }

      // Validate permissions for this role.
      foreach ($permissions as $permission_name => $grant) {
        if (!in_array($permission_name, $valid_permissions, TRUE)) {
          $result['valid'] = FALSE;
          $result['errors'][] = "Invalid permission: '$permission_name' for role '$role_id'.";
        }

        if (!is_bool($grant)) {
          $result['valid'] = FALSE;
          $result['errors'][] = "Invalid grant value for permission '$permission_name' in role '$role_id'. Must be boolean.";
        }
      }
    }

    return $result;
  }

}
