<?php

declare(strict_types=1);

namespace Drupal\Tests\permission_turbo\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\user\Entity\Role;

/**
 * Functional tests for Permissions Turbo page and API endpoints.
 *
 * @group permission_turbo
 */
class PermissionTurboPageTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'permission_turbo',
    'user',
    'system',
    'node',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Admin user with permission to administer permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * Regular user without permission to administer permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $regularUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create admin user with permission to administer permissions.
    $this->adminUser = $this->drupalCreateUser([
      'administer permissions',
    ]);

    // Create regular user without permission.
    $this->regularUser = $this->drupalCreateUser([]);
  }

  /**
   * Tests that anonymous users cannot access the permission turbo page.
   */
  public function testPageAccessDeniedForAnonymous(): void {
    $this->drupalGet('/admin/people/permissions-turbo');
    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Tests that users without permission cannot access the page.
   */
  public function testPageAccessDeniedForUnprivilegedUser(): void {
    $this->drupalLogin($this->regularUser);
    $this->drupalGet('/admin/people/permissions-turbo');
    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Tests that users with 'administer permissions' can access the page.
   */
  public function testPageAccessGrantedWithPermission(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/people/permissions-turbo');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Permissions (Turbo)');
  }

  /**
   * Tests that the page renders with proper structure.
   */
  public function testPageRendersWithProperStructure(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/people/permissions-turbo');

    // Check for the main container.
    $this->assertSession()->elementExists('css', '.permission-turbo-container');

    // Check for JavaScript attachment.
    $this->assertSession()->responseContains('permission-turbo/permission-turbo');

    // Check for drupalSettings with endpoints.
    $this->assertSession()->responseContains('permissionTurbo');
    $this->assertSession()->responseContains('endpoints');
  }

  /**
   * Tests labels API endpoint returns JSON.
   */
  public function testLabelsApiReturnsJson(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/api/permission-turbo/labels');

    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->responseHeaderEquals('Content-Type', 'application/json');

    // Parse JSON response.
    $response = json_decode($this->getSession()->getPage()->getContent(), TRUE);
    $this->assertIsArray($response);
    $this->assertTrue($response['success']);
    $this->assertArrayHasKey('data', $response);
  }

  /**
   * Tests labels API endpoint requires permission.
   */
  public function testLabelsApiRequiresPermission(): void {
    $this->drupalLogin($this->regularUser);
    $this->drupalGet('/api/permission-turbo/labels');
    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Tests labels API returns proper structure.
   */
  public function testLabelsApiReturnsProperStructure(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/api/permission-turbo/labels');

    $response = json_decode($this->getSession()->getPage()->getContent(), TRUE);

    $this->assertTrue($response['success']);
    $this->assertIsArray($response['data']);

    // Check that providers are present (e.g., 'node', 'user', 'system').
    $this->assertNotEmpty($response['data']);

    // Pick first provider and verify structure.
    $firstProvider = reset($response['data']);
    $this->assertArrayHasKey('id', $firstProvider);
    $this->assertArrayHasKey('label', $firstProvider);
    $this->assertArrayHasKey('permissions', $firstProvider);
    $this->assertIsArray($firstProvider['permissions']);

    // Verify permission structure if any permissions exist.
    if (!empty($firstProvider['permissions'])) {
      $firstPermission = reset($firstProvider['permissions']);
      $this->assertArrayHasKey('id', $firstPermission);
      $this->assertArrayHasKey('title', $firstPermission);
      $this->assertArrayHasKey('description', $firstPermission);
      $this->assertArrayHasKey('restrict_access', $firstPermission);
      $this->assertArrayHasKey('security_warning', $firstPermission);
    }
  }

  /**
   * Tests permissions API endpoint returns JSON for valid provider.
   */
  public function testPermissionsApiReturnsJsonForValidProvider(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/api/permission-turbo/permissions/node');

    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->responseHeaderEquals('Content-Type', 'application/json');

    $response = json_decode($this->getSession()->getPage()->getContent(), TRUE);
    $this->assertIsArray($response);
    $this->assertTrue($response['success']);
    $this->assertArrayHasKey('data', $response);
  }

  /**
   * Tests permissions API endpoint returns 404 for invalid provider.
   */
  public function testPermissionsApiReturns404ForInvalidProvider(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/api/permission-turbo/permissions/nonexistent_module');

    $this->assertSession()->statusCodeEquals(404);

    $response = json_decode($this->getSession()->getPage()->getContent(), TRUE);
    $this->assertFalse($response['success']);
    $this->assertArrayHasKey('error', $response);
  }

  /**
   * Tests permissions API returns proper structure.
   */
  public function testPermissionsApiReturnsProperStructure(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/api/permission-turbo/permissions/node');

    $response = json_decode($this->getSession()->getPage()->getContent(), TRUE);

    $this->assertTrue($response['success']);
    $this->assertArrayHasKey('data', $response);

    $data = $response['data'];
    $this->assertArrayHasKey('provider', $data);
    $this->assertArrayHasKey('provider_label', $data);
    $this->assertArrayHasKey('roles', $data);
    $this->assertArrayHasKey('permissions', $data);

    $this->assertEquals('node', $data['provider']);
    $this->assertIsArray($data['roles']);
    $this->assertIsArray($data['permissions']);

    // Verify role structure if roles exist.
    if (!empty($data['roles'])) {
      $firstRole = reset($data['roles']);
      $this->assertArrayHasKey('id', $firstRole);
      $this->assertArrayHasKey('label', $firstRole);
      $this->assertArrayHasKey('is_admin', $firstRole);
      $this->assertArrayHasKey('weight', $firstRole);
    }

    // Verify permission structure if permissions exist.
    if (!empty($data['permissions'])) {
      $firstPermission = reset($data['permissions']);
      $this->assertArrayHasKey('id', $firstPermission);
      $this->assertArrayHasKey('title', $firstPermission);
      $this->assertArrayHasKey('roles', $firstPermission);
      $this->assertIsArray($firstPermission['roles']);
    }
  }

  /**
   * Tests save API endpoint requires POST method.
   */
  public function testSaveApiRequiresPostMethod(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/api/permission-turbo/save');

    // GET request should fail.
    $this->assertSession()->statusCodeNotEquals(200);
  }

  /**
   * Tests save API endpoint requires CSRF token.
   */
  public function testSaveApiRequiresCsrfToken(): void {
    $this->drupalLogin($this->adminUser);

    // Get CSRF token.
    $csrfToken = \Drupal::csrfToken()->get('permission_turbo_api');

    // Attempt save without token.
    $changes = [
      'changes' => [
        [
          'role' => 'authenticated',
          'permission' => 'access content',
          'granted' => TRUE,
        ],
      ],
    ];

    $this->drupalPost(
      '/api/permission-turbo/save',
      json_encode($changes),
      [
        'Content-Type' => 'application/json',
      ]
    );

    $response = json_decode($this->getSession()->getPage()->getContent(), TRUE);
    $this->assertFalse($response['success']);
    $this->assertStringContainsString('CSRF', $response['error']);
  }

  /**
   * Tests save API endpoint with valid CSRF token and data.
   */
  public function testSaveApiWithValidCsrfTokenAndData(): void {
    $this->drupalLogin($this->adminUser);

    // Get CSRF token.
    $csrfToken = \Drupal::csrfToken()->get('permission_turbo_api');

    // Create a test role.
    $role = Role::create([
      'id' => 'test_role',
      'label' => 'Test Role',
    ]);
    $role->save();

    // Prepare changes.
    $changes = [
      'changes' => [
        [
          'role' => 'test_role',
          'permission' => 'access content',
          'granted' => TRUE,
        ],
      ],
    ];

    // Make POST request with CSRF token.
    $this->drupalPost(
      '/api/permission-turbo/save',
      json_encode($changes),
      [
        'Content-Type' => 'application/json',
        'X-CSRF-Token' => $csrfToken,
      ]
    );

    $this->assertSession()->statusCodeEquals(200);

    $response = json_decode($this->getSession()->getPage()->getContent(), TRUE);
    $this->assertTrue($response['success']);
    $this->assertEquals(1, $response['processed']);

    // Verify the permission was actually saved.
    $role = Role::load('test_role');
    $this->assertTrue($role->hasPermission('access content'));
  }

  /**
   * Tests save API validates JSON structure.
   */
  public function testSaveApiValidatesJsonStructure(): void {
    $this->drupalLogin($this->adminUser);
    $csrfToken = \Drupal::csrfToken()->get('permission_turbo_api');

    // Invalid JSON - missing 'changes' key.
    $invalidData = ['invalid' => 'data'];

    $this->drupalPost(
      '/api/permission-turbo/save',
      json_encode($invalidData),
      [
        'Content-Type' => 'application/json',
        'X-CSRF-Token' => $csrfToken,
      ]
    );

    $response = json_decode($this->getSession()->getPage()->getContent(), TRUE);
    $this->assertFalse($response['success']);
    $this->assertStringContainsString('changes', strtolower($response['error']));
  }

  /**
   * Tests save API validates change structure.
   */
  public function testSaveApiValidatesChangeStructure(): void {
    $this->drupalLogin($this->adminUser);
    $csrfToken = \Drupal::csrfToken()->get('permission_turbo_api');

    // Change missing 'permission' field.
    $invalidChanges = [
      'changes' => [
        [
          'role' => 'authenticated',
          'granted' => TRUE,
        ],
      ],
    ];

    $this->drupalPost(
      '/api/permission-turbo/save',
      json_encode($invalidChanges),
      [
        'Content-Type' => 'application/json',
        'X-CSRF-Token' => $csrfToken,
      ]
    );

    $response = json_decode($this->getSession()->getPage()->getContent(), TRUE);
    $this->assertFalse($response['success']);
    $this->assertStringContainsString('missing required fields', strtolower($response['error']));
  }

  /**
   * Tests save API validates granted field is boolean.
   */
  public function testSaveApiValidatesGrantedIsBoolean(): void {
    $this->drupalLogin($this->adminUser);
    $csrfToken = \Drupal::csrfToken()->get('permission_turbo_api');

    // 'granted' is a string instead of boolean.
    $invalidChanges = [
      'changes' => [
        [
          'role' => 'authenticated',
          'permission' => 'access content',
          'granted' => 'true',
        ],
      ],
    ];

    $this->drupalPost(
      '/api/permission-turbo/save',
      json_encode($invalidChanges),
      [
        'Content-Type' => 'application/json',
        'X-CSRF-Token' => $csrfToken,
      ]
    );

    $response = json_decode($this->getSession()->getPage()->getContent(), TRUE);
    $this->assertFalse($response['success']);
    $this->assertStringContainsString('boolean', strtolower($response['error']));
  }

  /**
   * Tests actual permission saving via API.
   */
  public function testActualPermissionSavingViaApi(): void {
    $this->drupalLogin($this->adminUser);
    $csrfToken = \Drupal::csrfToken()->get('permission_turbo_api');

    // Create test roles.
    $editor = Role::create([
      'id' => 'editor',
      'label' => 'Editor',
    ]);
    $editor->save();

    $manager = Role::create([
      'id' => 'manager',
      'label' => 'Manager',
    ]);
    $manager->save();

    // Prepare multiple changes.
    $changes = [
      'changes' => [
        [
          'role' => 'editor',
          'permission' => 'access content',
          'granted' => TRUE,
        ],
        [
          'role' => 'editor',
          'permission' => 'access user profiles',
          'granted' => TRUE,
        ],
        [
          'role' => 'manager',
          'permission' => 'access content',
          'granted' => FALSE,
        ],
      ],
    ];

    // Save permissions.
    $this->drupalPost(
      '/api/permission-turbo/save',
      json_encode($changes),
      [
        'Content-Type' => 'application/json',
        'X-CSRF-Token' => $csrfToken,
      ]
    );

    $response = json_decode($this->getSession()->getPage()->getContent(), TRUE);
    $this->assertTrue($response['success']);
    $this->assertEquals(2, $response['processed']);
    $this->assertCount(3, $response['changes']);

    // Reload roles and verify permissions.
    $editor = Role::load('editor');
    $this->assertTrue($editor->hasPermission('access content'));
    $this->assertTrue($editor->hasPermission('access user profiles'));

    $manager = Role::load('manager');
    $this->assertFalse($manager->hasPermission('access content'));
  }

  /**
   * Tests that save API invalidates cache.
   */
  public function testSaveApiInvalidatesCache(): void {
    $this->drupalLogin($this->adminUser);
    $csrfToken = \Drupal::csrfToken()->get('permission_turbo_api');

    // Create test role.
    $role = Role::create([
      'id' => 'cache_test',
      'label' => 'Cache Test',
    ]);
    $role->save();

    // Get labels before save (this should cache the data).
    $this->drupalGet('/api/permission-turbo/labels');
    $beforeResponse = json_decode($this->getSession()->getPage()->getContent(), TRUE);

    // Make a permission change.
    $changes = [
      'changes' => [
        [
          'role' => 'cache_test',
          'permission' => 'access content',
          'granted' => TRUE,
        ],
      ],
    ];

    $this->drupalPost(
      '/api/permission-turbo/save',
      json_encode($changes),
      [
        'Content-Type' => 'application/json',
        'X-CSRF-Token' => $csrfToken,
      ]
    );

    // Get labels after save - cache should be invalidated.
    $this->drupalGet('/api/permission-turbo/labels');
    $afterResponse = json_decode($this->getSession()->getPage()->getContent(), TRUE);

    // Both should succeed.
    $this->assertTrue($beforeResponse['success']);
    $this->assertTrue($afterResponse['success']);
  }

  /**
   * Tests save API returns proper error for non-existent permission.
   */
  public function testSaveApiReturnsErrorForNonExistentPermission(): void {
    $this->drupalLogin($this->adminUser);
    $csrfToken = \Drupal::csrfToken()->get('permission_turbo_api');

    // Create test role.
    $role = Role::create([
      'id' => 'test_error',
      'label' => 'Test Error',
    ]);
    $role->save();

    // Try to save a non-existent permission.
    $changes = [
      'changes' => [
        [
          'role' => 'test_error',
          'permission' => 'this_permission_does_not_exist',
          'granted' => TRUE,
        ],
      ],
    ];

    $this->drupalPost(
      '/api/permission-turbo/save',
      json_encode($changes),
      [
        'Content-Type' => 'application/json',
        'X-CSRF-Token' => $csrfToken,
      ]
    );

    $response = json_decode($this->getSession()->getPage()->getContent(), TRUE);

    // The service validates and should return error.
    $this->assertFalse($response['success']);
  }

  /**
   * Helper method to make POST request with headers.
   *
   * @param string $url
   *   The URL to POST to.
   * @param string $body
   *   The request body.
   * @param array $headers
   *   The request headers.
   */
  protected function drupalPost(string $url, string $body, array $headers = []): void {
    $client = $this->getSession()->getDriver()->getClient();
    $client->request('POST', $url, [], [], $headers, $body);
  }

}
