<?php

declare(strict_types=1);

namespace Drupal\Tests\permission_turbo\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\user\Entity\Role;

/**
 * Kernel tests for PermissionSaveService.
 *
 * @coversDefaultClass \Drupal\permission_turbo\Service\PermissionSaveService
 * @group permission_turbo
 */
class PermissionSaveServiceTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'permission_turbo',
  ];

  /**
   * The permission save service under test.
   *
   * @var \Drupal\permission_turbo\Service\PermissionSaveService
   */
  protected $permissionSaveService;

  /**
   * Test roles created for testing.
   *
   * @var \Drupal\user\RoleInterface[]
   */
  protected $testRoles = [];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installConfig(['user', 'system']);

    // Get the permission save service.
    $this->permissionSaveService = $this->container->get('permission_turbo.save');

    // Create test roles.
    $this->createTestRoles();
  }

  /**
   * Creates test roles for testing.
   */
  protected function createTestRoles(): void {
    // Create editor role.
    $editor = Role::create([
      'id' => 'editor',
      'label' => 'Editor',
      'weight' => 3,
    ]);
    $editor->save();
    $this->testRoles['editor'] = $editor;

    // Create content_manager role.
    $content_manager = Role::create([
      'id' => 'content_manager',
      'label' => 'Content Manager',
      'weight' => 4,
    ]);
    $content_manager->save();
    $this->testRoles['content_manager'] = $content_manager;

    // Create administrator role (admin role).
    $admin = Role::create([
      'id' => 'test_admin',
      'label' => 'Test Administrator',
      'weight' => 10,
      'is_admin' => TRUE,
    ]);
    $admin->save();
    $this->testRoles['test_admin'] = $admin;
  }

  /**
   * Tests saveChanges with valid permission changes.
   *
   * @covers ::saveChanges
   */
  public function testSaveChangesWithValidChanges(): void {
    $changes = [
      [
        'role' => 'editor',
        'permission' => 'access content',
        'granted' => TRUE,
      ],
      [
        'role' => 'editor',
        'permission' => 'access user profiles',
        'granted' => TRUE,
      ],
      [
        'role' => 'content_manager',
        'permission' => 'access content',
        'granted' => FALSE,
      ],
    ];

    $result = $this->permissionSaveService->saveChanges($changes);

    $this->assertTrue($result['success']);
    $this->assertEquals(2, $result['processed']);
    $this->assertCount(3, $result['changes']);
    $this->assertEmpty($result['errors']);

    // Verify the permissions were actually saved.
    $editor = Role::load('editor');
    $this->assertTrue($editor->hasPermission('access content'));
    $this->assertTrue($editor->hasPermission('access user profiles'));

    $content_manager = Role::load('content_manager');
    $this->assertFalse($content_manager->hasPermission('access content'));
  }

  /**
   * Tests saveChanges with invalid role ID.
   *
   * @covers ::saveChanges
   */
  public function testSaveChangesWithInvalidRole(): void {
    $changes = [
      [
        'role' => 'nonexistent_role',
        'permission' => 'access content',
        'granted' => TRUE,
      ],
    ];

    $result = $this->permissionSaveService->saveChanges($changes);

    $this->assertTrue($result['success']);
    $this->assertEquals(0, $result['processed']);
    $this->assertNotEmpty($result['errors']);
    $this->assertStringContainsString('nonexistent_role', $result['errors'][0]);
    $this->assertStringContainsString('not found', $result['errors'][0]);
  }

  /**
   * Tests saveChanges skips admin roles.
   *
   * @covers ::saveChanges
   */
  public function testSaveChangesSkipsAdminRole(): void {
    $changes = [
      [
        'role' => 'test_admin',
        'permission' => 'access content',
        'granted' => TRUE,
      ],
    ];

    $result = $this->permissionSaveService->saveChanges($changes);

    $this->assertTrue($result['success']);
    $this->assertEquals(0, $result['processed']);
    $this->assertNotEmpty($result['errors']);
    $this->assertStringContainsString('test_admin', $result['errors'][0]);
    $this->assertStringContainsString('admin', $result['errors'][0]);
  }

  /**
   * Tests saveChanges with no actual changes (granted state same as current).
   *
   * @covers ::saveChanges
   */
  public function testSaveChangesWithNoActualChanges(): void {
    // First grant a permission.
    $editor = $this->testRoles['editor'];
    $editor->grantPermission('access content');
    $editor->save();

    // Try to "change" it to the same state.
    $changes = [
      [
        'role' => 'editor',
        'permission' => 'access content',
        'granted' => TRUE,
      ],
    ];

    $result = $this->permissionSaveService->saveChanges($changes);

    $this->assertTrue($result['success']);
    $this->assertEquals(0, $result['processed']);
    $this->assertEmpty($result['changes']);
    $this->assertEmpty($result['errors']);
  }

  /**
   * Tests saveChanges revoking permissions.
   *
   * @covers ::saveChanges
   */
  public function testSaveChangesRevokingPermissions(): void {
    // First grant permissions.
    $editor = $this->testRoles['editor'];
    $editor->grantPermission('access content');
    $editor->grantPermission('access user profiles');
    $editor->save();

    // Now revoke them.
    $changes = [
      [
        'role' => 'editor',
        'permission' => 'access content',
        'granted' => FALSE,
      ],
      [
        'role' => 'editor',
        'permission' => 'access user profiles',
        'granted' => FALSE,
      ],
    ];

    $result = $this->permissionSaveService->saveChanges($changes);

    $this->assertTrue($result['success']);
    $this->assertEquals(1, $result['processed']);
    $this->assertCount(2, $result['changes']);

    // Verify permissions were revoked.
    $editor = Role::load('editor');
    $this->assertFalse($editor->hasPermission('access content'));
    $this->assertFalse($editor->hasPermission('access user profiles'));

    // Verify action is 'revoked'.
    $this->assertEquals('revoked', $result['changes'][0]['action']);
    $this->assertEquals('revoked', $result['changes'][1]['action']);
  }

  /**
   * Tests saveChanges with mixed grant and revoke operations.
   *
   * @covers ::saveChanges
   */
  public function testSaveChangesWithMixedOperations(): void {
    // Grant one permission first.
    $editor = $this->testRoles['editor'];
    $editor->grantPermission('access user profiles');
    $editor->save();

    $changes = [
      [
        'role' => 'editor',
        'permission' => 'access content',
        'granted' => TRUE,
      ],
      [
        'role' => 'editor',
        'permission' => 'access user profiles',
        'granted' => FALSE,
      ],
    ];

    $result = $this->permissionSaveService->saveChanges($changes);

    $this->assertTrue($result['success']);
    $this->assertEquals(1, $result['processed']);
    $this->assertCount(2, $result['changes']);

    // Verify results.
    $editor = Role::load('editor');
    $this->assertTrue($editor->hasPermission('access content'));
    $this->assertFalse($editor->hasPermission('access user profiles'));

    // Check actions.
    $grantedChange = array_filter($result['changes'], fn($c) => $c['action'] === 'granted');
    $revokedChange = array_filter($result['changes'], fn($c) => $c['action'] === 'revoked');
    $this->assertCount(1, $grantedChange);
    $this->assertCount(1, $revokedChange);
  }

  /**
   * Tests saveChanges with multiple roles.
   *
   * @covers ::saveChanges
   */
  public function testSaveChangesWithMultipleRoles(): void {
    $changes = [
      [
        'role' => 'editor',
        'permission' => 'access content',
        'granted' => TRUE,
      ],
      [
        'role' => 'content_manager',
        'permission' => 'access content',
        'granted' => TRUE,
      ],
      [
        'role' => 'content_manager',
        'permission' => 'access user profiles',
        'granted' => TRUE,
      ],
    ];

    $result = $this->permissionSaveService->saveChanges($changes);

    $this->assertTrue($result['success']);
    $this->assertEquals(2, $result['processed']);
    $this->assertCount(3, $result['changes']);

    // Verify both roles were updated.
    $editor = Role::load('editor');
    $this->assertTrue($editor->hasPermission('access content'));

    $content_manager = Role::load('content_manager');
    $this->assertTrue($content_manager->hasPermission('access content'));
    $this->assertTrue($content_manager->hasPermission('access user profiles'));
  }

  /**
   * Tests validateChanges with valid changes.
   *
   * @covers ::validateChanges
   */
  public function testValidateChangesWithValidData(): void {
    $changes = [
      [
        'role' => 'editor',
        'permission' => 'access content',
        'granted' => TRUE,
      ],
    ];

    $result = $this->permissionSaveService->validateChanges($changes);

    $this->assertTrue($result['valid']);
    $this->assertEmpty($result['errors']);
  }

  /**
   * Tests validateChanges with missing role field.
   *
   * @covers ::validateChanges
   */
  public function testValidateChangesWithMissingRole(): void {
    $changes = [
      [
        'permission' => 'access content',
        'granted' => TRUE,
      ],
    ];

    $result = $this->permissionSaveService->validateChanges($changes);

    $this->assertFalse($result['valid']);
    $this->assertNotEmpty($result['errors']);
    $this->assertStringContainsString('missing required fields', $result['errors'][0]);
  }

  /**
   * Tests validateChanges with missing permission field.
   *
   * @covers ::validateChanges
   */
  public function testValidateChangesWithMissingPermission(): void {
    $changes = [
      [
        'role' => 'editor',
        'granted' => TRUE,
      ],
    ];

    $result = $this->permissionSaveService->validateChanges($changes);

    $this->assertFalse($result['valid']);
    $this->assertNotEmpty($result['errors']);
    $this->assertStringContainsString('missing required fields', $result['errors'][0]);
  }

  /**
   * Tests validateChanges with missing granted field.
   *
   * @covers ::validateChanges
   */
  public function testValidateChangesWithMissingGranted(): void {
    $changes = [
      [
        'role' => 'editor',
        'permission' => 'access content',
      ],
    ];

    $result = $this->permissionSaveService->validateChanges($changes);

    $this->assertFalse($result['valid']);
    $this->assertNotEmpty($result['errors']);
    $this->assertStringContainsString('missing required fields', $result['errors'][0]);
  }

  /**
   * Tests validateChanges with non-existent role.
   *
   * @covers ::validateChanges
   */
  public function testValidateChangesWithNonExistentRole(): void {
    $changes = [
      [
        'role' => 'fake_role',
        'permission' => 'access content',
        'granted' => TRUE,
      ],
    ];

    $result = $this->permissionSaveService->validateChanges($changes);

    $this->assertFalse($result['valid']);
    $this->assertNotEmpty($result['errors']);
    $this->assertStringContainsString('fake_role', $result['errors'][0]);
    $this->assertStringContainsString('does not exist', $result['errors'][0]);
  }

  /**
   * Tests validateChanges with non-existent permission.
   *
   * @covers ::validateChanges
   */
  public function testValidateChangesWithNonExistentPermission(): void {
    $changes = [
      [
        'role' => 'editor',
        'permission' => 'fake_permission_that_does_not_exist',
        'granted' => TRUE,
      ],
    ];

    $result = $this->permissionSaveService->validateChanges($changes);

    $this->assertFalse($result['valid']);
    $this->assertNotEmpty($result['errors']);
    $this->assertStringContainsString('fake_permission_that_does_not_exist', $result['errors'][0]);
    $this->assertStringContainsString('does not exist', $result['errors'][0]);
  }

  /**
   * Tests validateChanges with multiple invalid changes.
   *
   * @covers ::validateChanges
   */
  public function testValidateChangesWithMultipleErrors(): void {
    $changes = [
      [
        'role' => 'fake_role',
        'permission' => 'access content',
        'granted' => TRUE,
      ],
      [
        'role' => 'editor',
        'permission' => 'fake_permission',
        'granted' => TRUE,
      ],
      [
        'permission' => 'access content',
        'granted' => TRUE,
      ],
    ];

    $result = $this->permissionSaveService->validateChanges($changes);

    $this->assertFalse($result['valid']);
    $this->assertCount(3, $result['errors']);
  }

  /**
   * Tests validateChanges with empty changes array.
   *
   * @covers ::validateChanges
   */
  public function testValidateChangesWithEmptyArray(): void {
    $changes = [];

    $result = $this->permissionSaveService->validateChanges($changes);

    $this->assertTrue($result['valid']);
    $this->assertEmpty($result['errors']);
  }

  /**
   * Tests that saveChanges groups changes by role efficiently.
   *
   * @covers ::saveChanges
   */
  public function testSaveChangesGroupsByRole(): void {
    // Create multiple changes for the same role.
    $changes = [
      [
        'role' => 'editor',
        'permission' => 'access content',
        'granted' => TRUE,
      ],
      [
        'role' => 'editor',
        'permission' => 'access user profiles',
        'granted' => TRUE,
      ],
      [
        'role' => 'editor',
        'permission' => 'view the administration theme',
        'granted' => TRUE,
      ],
    ];

    $result = $this->permissionSaveService->saveChanges($changes);

    // Should only save the role once, not three times.
    $this->assertEquals(1, $result['processed']);
    $this->assertCount(3, $result['changes']);

    // Verify all permissions were granted.
    $editor = Role::load('editor');
    $this->assertTrue($editor->hasPermission('access content'));
    $this->assertTrue($editor->hasPermission('access user profiles'));
    $this->assertTrue($editor->hasPermission('view the administration theme'));
  }

}
