<?php

declare(strict_types=1);

namespace Drupal\Tests\permission_turbo\Unit;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\permission_turbo\Service\PermissionDataService;
use Drupal\Tests\UnitTestCase;
use Drupal\user\PermissionHandlerInterface;
use Drupal\user\RoleInterface;

/**
 * Unit tests for PermissionDataService.
 *
 * @coversDefaultClass \Drupal\permission_turbo\Service\PermissionDataService
 * @group permission_turbo
 */
class PermissionDataServiceTest extends UnitTestCase {

  /**
   * The cache backend mock.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $cache;

  /**
   * The entity type manager mock.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * The permission handler mock.
   *
   * @var \Drupal\user\PermissionHandlerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $permissionHandler;

  /**
   * The module handler mock.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $moduleHandler;

  /**
   * The permission data service under test.
   *
   * @var \Drupal\permission_turbo\Service\PermissionDataService
   */
  protected $permissionDataService;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create mocks in the correct order matching the service constructor.
    $this->cache = $this->createMock(CacheBackendInterface::class);
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->permissionHandler = $this->createMock(PermissionHandlerInterface::class);
    $this->moduleHandler = $this->createMock(ModuleHandlerInterface::class);

    // Create service with mocked dependencies.
    $this->permissionDataService = new PermissionDataService(
      $this->cache,
      $this->entityTypeManager,
      $this->permissionHandler,
      $this->moduleHandler
    );
  }

  /**
   * Tests getLabels returns cached data when available.
   *
   * @covers ::getLabels
   */
  public function testGetLabelsReturnsCache(): void {
    $cachedData = [
      'node' => [
        'name' => 'Node',
        'permissions' => [
          'access content' => [
            'title' => 'View published content',
            'provider' => 'node',
            'description' => 'Allow viewing of published content.',
          ],
        ],
      ],
    ];

    $cacheObject = (object) ['data' => $cachedData];
    $this->cache->expects($this->once())
      ->method('get')
      ->with('permission_turbo:labels')
      ->willReturn($cacheObject);

    // Permission handler should not be called when cache is available.
    $this->permissionHandler->expects($this->never())
      ->method('getPermissions');

    $result = $this->permissionDataService->getLabels();

    $this->assertEquals($cachedData, $result);
  }

  /**
   * Tests getLabels generates and caches data when cache is empty.
   *
   * @covers ::getLabels
   */
  public function testGetLabelsGeneratesWhenCacheMiss(): void {
    $permissions = [
      'access content' => [
        'title' => 'View published content',
        'description' => 'Allow viewing of published content.',
        'provider' => 'node',
      ],
      'create article content' => [
        'title' => 'Article: Create new content',
        'description' => 'Allow creating new article content.',
        'provider' => 'node',
        'restrict access' => TRUE,
      ],
    ];

    $this->cache->expects($this->once())
      ->method('get')
      ->with('permission_turbo:labels')
      ->willReturn(FALSE);

    $this->permissionHandler->expects($this->once())
      ->method('getPermissions')
      ->willReturn($permissions);

    $this->moduleHandler->expects($this->once())
      ->method('moduleExists')
      ->with('node')
      ->willReturn(TRUE);

    $this->moduleHandler->expects($this->once())
      ->method('getName')
      ->with('node')
      ->willReturn('Node');

    $this->cache->expects($this->once())
      ->method('set')
      ->with(
        'permission_turbo:labels',
        $this->callback(function ($data) {
          return isset($data['node'])
            && $data['node']['name'] === 'Node'
            && count($data['node']['permissions']) === 2
            && isset($data['node']['permissions']['access content'])
            && isset($data['node']['permissions']['create article content']);
        }),
        CacheBackendInterface::CACHE_PERMANENT,
        ['permission_turbo', 'config:user.role', 'config:core.extension']
      );

    $result = $this->permissionDataService->getLabels();

    $this->assertArrayHasKey('node', $result);
    $this->assertEquals('Node', $result['node']['name']);
    $this->assertCount(2, $result['node']['permissions']);
  }

  /**
   * Tests getProviderPermissions returns cached data when available.
   *
   * @covers ::getProviderPermissions
   */
  public function testGetProviderPermissionsReturnsCache(): void {
    $cachedData = [
      'access content' => [
        'title' => 'View published content',
        'description' => 'Test description',
        'provider' => 'node',
        'roles' => ['authenticated' => TRUE, 'anonymous' => FALSE],
        'restrict_access' => FALSE,
      ],
    ];

    $cacheObject = (object) ['data' => $cachedData];
    $this->cache->expects($this->once())
      ->method('get')
      ->with('permission_turbo:provider:node')
      ->willReturn($cacheObject);

    $result = $this->permissionDataService->getProviderPermissions('node');

    $this->assertEquals($cachedData, $result);
  }

  /**
   * Tests getProviderPermissions generates data when cache is empty.
   *
   * @covers ::getProviderPermissions
   */
  public function testGetProviderPermissionsGeneratesWhenCacheMiss(): void {
    $permissions = [
      'access content' => [
        'title' => 'View published content',
        'description' => 'Test description',
        'provider' => 'node',
        'restrict access' => FALSE,
      ],
      'other permission' => [
        'title' => 'Other permission',
        'description' => 'Other module permission',
        'provider' => 'other_module',
      ],
    ];

    // Mock role.
    $role = $this->createMock(RoleInterface::class);
    $role->method('getPermissions')
      ->willReturn(['access content']);
    $role->method('getWeight')
      ->willReturn(0);

    $roleStorage = $this->createMock(EntityStorageInterface::class);
    $roleStorage->method('loadMultiple')
      ->willReturn(['authenticated' => $role]);

    $this->entityTypeManager->method('getStorage')
      ->with('user_role')
      ->willReturn($roleStorage);

    // First call for getRoles cache, second for provider permissions.
    $this->cache->method('get')
      ->willReturn(FALSE);

    $this->permissionHandler->method('getPermissions')
      ->willReturn($permissions);

    // Expect cache set to be called (for roles and for provider permissions).
    $this->cache->expects($this->exactly(2))
      ->method('set');

    $result = $this->permissionDataService->getProviderPermissions('node');

    $this->assertArrayHasKey('access content', $result);
    $this->assertArrayNotHasKey('other permission', $result);
    $this->assertEquals('View published content', $result['access content']['title']);
    $this->assertTrue($result['access content']['roles']['authenticated']);
  }

  /**
   * Tests getRoles returns role entities sorted by weight.
   *
   * @covers ::getRoles
   */
  public function testGetRolesReturnsSortedRoles(): void {
    $role1 = $this->createMock(RoleInterface::class);
    $role1->method('getWeight')->willReturn(5);

    $role2 = $this->createMock(RoleInterface::class);
    $role2->method('getWeight')->willReturn(0);

    $role3 = $this->createMock(RoleInterface::class);
    $role3->method('getWeight')->willReturn(10);

    $roleStorage = $this->createMock(EntityStorageInterface::class);
    $roleStorage->expects($this->once())
      ->method('loadMultiple')
      ->willReturn([
        'role_a' => $role1,
        'role_b' => $role2,
        'role_c' => $role3,
      ]);

    $this->entityTypeManager->expects($this->once())
      ->method('getStorage')
      ->with('user_role')
      ->willReturn($roleStorage);

    $this->cache->method('get')->willReturn(FALSE);

    $result = $this->permissionDataService->getRoles();

    $this->assertCount(3, $result);
    // Roles should be sorted by weight.
    $keys = array_keys($result);
    // Weight 0.
    $this->assertEquals('role_b', $keys[0]);
    // Weight 5.
    $this->assertEquals('role_a', $keys[1]);
    // Weight 10.
    $this->assertEquals('role_c', $keys[2]);
  }

  /**
   * Tests getRoles returns cached data when available.
   *
   * @covers ::getRoles
   */
  public function testGetRolesReturnsCache(): void {
    $role = $this->createMock(RoleInterface::class);
    $cachedRoles = ['authenticated' => $role];

    $cacheObject = (object) ['data' => $cachedRoles];
    $this->cache->expects($this->once())
      ->method('get')
      ->with('permission_turbo:roles')
      ->willReturn($cacheObject);

    // Entity type manager should not be called when cache is available.
    $this->entityTypeManager->expects($this->never())
      ->method('getStorage');

    $result = $this->permissionDataService->getRoles();

    $this->assertEquals($cachedRoles, $result);
  }

  /**
   * Tests renderDescription with various input types.
   *
   * @covers ::renderDescription
   */
  public function testRenderDescriptionWithString(): void {
    // Test via getLabels which uses renderDescription internally.
    $permissions = [
      'test_perm' => [
        'title' => 'Test Permission',
        'description' => 'Simple string description',
        'provider' => 'test',
      ],
    ];

    $this->cache->method('get')->willReturn(FALSE);
    $this->permissionHandler->method('getPermissions')->willReturn($permissions);
    $this->moduleHandler->method('moduleExists')->willReturn(FALSE);

    $result = $this->permissionDataService->getLabels();

    $this->assertEquals(
      'Simple string description',
      $result['test']['permissions']['test_perm']['description']
    );
  }

  /**
   * Tests renderDescription with empty input.
   *
   * @covers ::renderDescription
   */
  public function testRenderDescriptionWithEmpty(): void {
    $permissions = [
      'test_perm' => [
        'title' => 'Test Permission',
        'description' => '',
        'provider' => 'test',
      ],
    ];

    $this->cache->method('get')->willReturn(FALSE);
    $this->permissionHandler->method('getPermissions')->willReturn($permissions);
    $this->moduleHandler->method('moduleExists')->willReturn(FALSE);

    $result = $this->permissionDataService->getLabels();

    $this->assertEquals('', $result['test']['permissions']['test_perm']['description']);
  }

  /**
   * Tests getProviderName returns module name when module exists.
   *
   * @covers ::getProviderName
   */
  public function testGetProviderNameReturnsModuleName(): void {
    $permissions = [
      'test_perm' => [
        'title' => 'Test',
        'provider' => 'system',
      ],
    ];

    $this->cache->method('get')->willReturn(FALSE);
    $this->permissionHandler->method('getPermissions')->willReturn($permissions);
    $this->moduleHandler->method('moduleExists')
      ->with('system')
      ->willReturn(TRUE);
    $this->moduleHandler->method('getName')
      ->with('system')
      ->willReturn('System');

    $result = $this->permissionDataService->getLabels();

    $this->assertEquals('System', $result['system']['name']);
  }

  /**
   * Tests getProviderName falls back to machine name when module not found.
   *
   * @covers ::getProviderName
   */
  public function testGetProviderNameFallsBackToMachineName(): void {
    $permissions = [
      'test_perm' => [
        'title' => 'Test',
        'provider' => 'unknown_module',
      ],
    ];

    $this->cache->method('get')->willReturn(FALSE);
    $this->permissionHandler->method('getPermissions')->willReturn($permissions);
    $this->moduleHandler->method('moduleExists')
      ->with('unknown_module')
      ->willReturn(FALSE);

    $result = $this->permissionDataService->getLabels();

    $this->assertEquals('unknown_module', $result['unknown_module']['name']);
  }

  /**
   * Tests that restrict_access flag is properly passed through.
   *
   * @covers ::getProviderPermissions
   */
  public function testRestrictAccessFlagIsPreserved(): void {
    $permissions = [
      'dangerous_perm' => [
        'title' => 'Dangerous Permission',
        'description' => 'This is restricted',
        'provider' => 'test',
        'restrict access' => TRUE,
      ],
      'safe_perm' => [
        'title' => 'Safe Permission',
        'description' => 'This is not restricted',
        'provider' => 'test',
      ],
    ];

    $role = $this->createMock(RoleInterface::class);
    $role->method('getPermissions')->willReturn([]);
    $role->method('getWeight')->willReturn(0);

    $roleStorage = $this->createMock(EntityStorageInterface::class);
    $roleStorage->method('loadMultiple')->willReturn(['test_role' => $role]);

    $this->entityTypeManager->method('getStorage')
      ->with('user_role')
      ->willReturn($roleStorage);

    $this->cache->method('get')->willReturn(FALSE);
    $this->permissionHandler->method('getPermissions')->willReturn($permissions);

    $result = $this->permissionDataService->getProviderPermissions('test');

    $this->assertTrue($result['dangerous_perm']['restrict_access']);
    $this->assertFalse($result['safe_perm']['restrict_access']);
  }

}
