<?php

declare(strict_types=1);

namespace Drupal\permission_watchdog\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\user\RoleInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Permission watchdog settings for this site.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The entity type manager service.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = parent::create($container);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'permission_watchdog_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['permission_watchdog.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['all_roles'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Track all roles for changes to permissions'),
      '#default_value' => $this->config('permission_watchdog.settings')->get('all_roles'),
    ];

    /** @var \Drupal\user\RoleInterface[] $roles */
    $roles = $this->entityTypeManager->getStorage('user_role')->loadMultiple();
    // Admin roles don't store permissions, so we would have nothing to compare
    // against.
    $filtered_roles = array_filter($roles, static fn(RoleInterface $role) => !$role->isAdmin());
    $options = array_map(static fn(RoleInterface $role) => $role->label(), $filtered_roles);

    $form['roles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Track changes to permissions on the following roles'),
      '#description' => $this->t('Administrative roles are not present in this list because they have all permissions implicitly, e.g. without actually storing the whole list. So it is impossible to compare the changes. Also, no changes to their permissions could be done in the UI.'),
      '#default_value' => $this->config('permission_watchdog.settings')->get('roles'),
      '#options' => $options,
      '#states' => [
        'visible' => [
          ':input[name="all_roles"]' => ['checked' => FALSE],
        ],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $allRoles = filter_var($form_state->getValue('all_roles'), FILTER_VALIDATE_BOOLEAN);
    $enabledRoles = array_filter($form_state->getValue('roles'), static fn($v) => !empty($v));

    if ($allRoles) {
      $form_state->setValue('roles', []);
    }
    elseif (empty($enabledRoles)) {
      $form_state->setErrorByName('roles', $this->t('At least one role must be selected'));
    }
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $enabledRoles = array_filter($form_state->getValue('roles'), static fn($v) => !empty($v));

    $this->config('permission_watchdog.settings')
      ->set('all_roles', $form_state->getValue('all_roles'))
      ->set('roles', array_values($enabledRoles))
      ->save();
    parent::submitForm($form, $form_state);
  }

}
