<?php

declare(strict_types=1);

namespace Drupal\permission_watchdog\Hook;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\permission_watchdog\Entity\RoleChangeLog;
use Drupal\user\RoleInterface;

/**
 * Hook implementations for permission_watchdog module.
 */
class PermissionWatchdogHooks {

  use StringTranslationTrait;

  /**
   * Constructs a new PermissionWatchdogHooks object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected AccountProxyInterface $currentUser,
  ) {}

  /**
   * Implements hook_form_user_admin_permissions_alter().
   *
   * Stores current permissions per role in the form state and sets additional
   * submit callback.
   */
  #[Hook('form_user_admin_permissions_alter')]
  public function formUserAdminPermissionsAlter(array &$form, FormStateInterface $form_state): void {
    $config = $this->configFactory->get('permission_watchdog.settings');
    $roleStorage = $this->entityTypeManager->getStorage('user_role');
    $allRoles = filter_var($config->get('all_roles'), FILTER_VALIDATE_BOOLEAN);

    if ($allRoles) {
      $roles = $roleStorage->loadMultiple();
    }
    else {
      $roles = $roleStorage->loadMultiple($config->get('roles'));
    }

    // Admin roles don't store permissions, so we would have nothing to compare
    // against.
    /** @var \Drupal\user\RoleInterface[] $roles */
    $roles = array_filter($roles, fn(RoleInterface $role) => !$role->isAdmin());

    if (empty($roles)) {
      return;
    }

    // Store roles for submit callback.
    $form['permissions_watchdog_roles'] = [
      '#type' => 'value',
      '#value' => array_keys($roles),
    ];

    /*
     * There are multiple forms extending this form:
     *   - EntityPermissionsForm
     *   - UserPermissionsModuleSpecificForm
     *   - UserPermissionsRoleSpecificForm
     *
     * thankfully all of them use the same method to fetch permissions.
     * For this reason we don't want to duplicate the code from
     * all of them, but rather just want to use the result.
     */
    $formObject = $form_state->getBuildInfo()['callback_object'];
    $reflection = new \ReflectionMethod($formObject, 'permissionsByProvider');
    $permissionsByProvider = $reflection->invoke($formObject);

    // We want to store role permission with the same structure as
    // 'user_admin_permissions' form will submit them, to simplify comparison.
    $rolePermissions = [];
    foreach ($roles as $roleName => $role) {
      $rolePermissionsAll = $role->getPermissions();
      foreach ($permissionsByProvider as $permissions) {
        foreach (array_keys($permissions) as $permission) {
          // Roles store only permissions they have access to, this is
          // represented by integer `1` in submitted from the table checkbox.
          $rolePermissions[$roleName][$permission] = in_array($permission, $rolePermissionsAll, TRUE) ? 1 : 0;
        }
      }
    }

    // Store current role permission to compare in submit callback.
    $form['permissions_watchdog_permissions'] = [
      '#type' => 'value',
      '#value' => $rolePermissions,
    ];

    $form['#submit'][] = [self::class, 'formUserAdminPermissionsSubmit'];
  }

  /**
   * Submit callback for 'user_admin_permissions' form.
   *
   * Compares previous role permissions to the current and stores the change.
   */
  public static function formUserAdminPermissionsSubmit(array $form, FormStateInterface $form_state): void {
    \Drupal::service(self::class)->doFormUserAdminPermissionsSubmit($form, $form_state);
  }

  /**
   * Processes the 'user_admin_permissions' form submission.
   *
   * Compares previous role permissions to the current and stores the change.
   */
  public function doFormUserAdminPermissionsSubmit(array $form, FormStateInterface $form_state): void {
    $storage = $this->entityTypeManager->getStorage('role_change_log');
    $roles = $form_state->getValue('permissions_watchdog_roles');
    if (empty($roles)) {
      return;
    }

    // In testUserPermissionsRoleSpecificForm() we couldn't tell which roles are
    // actually rendered in permissions table up until now, without unnecessary
    // processing of the form structure, but now it's easy to tell.
    $roles = array_intersect($roles, array_keys($form_state->getValue('role_names')));

    foreach ($roles as $role) {
      $oldPermissions = $form_state->getValue(['permissions_watchdog_permissions', $role]);
      $newPermissions = $form_state->getValue($role);
      $changed = array_diff_assoc($newPermissions, $oldPermissions);
      if (empty($changed)) {
        continue;
      }

      /** @var \Drupal\permission_watchdog\Entity\RoleChangeLog $change */
      $change = $storage->create([
        'uid' => $this->currentUser->id(),
        'role' => $role,
      ]);
      foreach ($changed as $permission => $status) {
        $change->get('actions')->appendItem([
          'action' => (bool) $status ? RoleChangeLog::PERMISSION_ADDED : RoleChangeLog::PERMISSION_REMOVED,
          'permission' => $permission,
        ]);
      }

      $change->save();
    }
  }

  /**
   * Implements hook_form_views_exposed_form_alter().
   *
   * Makes role filter to be a select rather than plain text input.
   */
  #[Hook('form_views_exposed_form_alter')]
  public function formViewsExposedFormAlter(array &$form, FormStateInterface $form_state): void {
    if ($form_state->get('view')->id() !== 'roles_change_log') {
      return;
    }

    try {
      $rolesStorage = $this->entityTypeManager->getStorage('user_role');
    }
    catch (\Exception $e) {
      return;
    }
    /** @var \Drupal\user\RoleInterface[] $ids */
    $ids = $rolesStorage->loadMultiple();
    // Admin roles don't store permissions,
    // so we would have nothing to compare
    // against.
    $filteredRoles = array_filter($ids, fn(RoleInterface $role) => !$role->isAdmin());
    $options = array_map(fn(RoleInterface $role) => $role->label(), $filteredRoles);

    $form['role'] = [
      '#type' => 'select',
      '#options' => $options,
      '#empty_value' => '',
      '#empty_option' => $this->t('- Any -'),
    ];
  }

}
