<?php

declare(strict_types=1);

namespace Drupal\permission_watchdog\Plugin\DevelGenerate;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\devel_generate\DevelGenerateBase;
use Drupal\permission_watchdog\Entity\RoleChangeLog;
use Drupal\user\PermissionHandlerInterface;
use Drupal\user\RoleStorageInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a ContentDevelGenerate plugin.
 *
 * @DevelGenerate(
 *   id = "role_change",
 *   label = @Translation("Role change"),
 *   description = @Translation("Generate a given number of content. Optionally delete current content."),
 *   url = "role-change",
 *   permission = "administer devel_generate",
 *   settings = {
 *     "num" = 50,
 *     "kill" = FALSE,
 *     "max_actions" = 5,
 *   }
 * )
 */
class RoleChangeDevelGenerate extends DevelGenerateBase implements ContainerFactoryPluginInterface {

  /**
   * The date formatter service.
   */
  protected DateFormatterInterface $dateFormatter;

  /**
   * Provides system time.
   */
  protected TimeInterface $time;

  /**
   * Database connection.
   */
  protected Connection $database;

  /**
   * Permission handler.
   */
  protected PermissionHandlerInterface $permissionHandler;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->dateFormatter = $container->get('date.formatter');
    $instance->time = $container->get('datetime.time');
    $instance->database = $container->get('database');
    $instance->permissionHandler = $container->get('user.permissions');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {

    $form['kill'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('<strong>Delete all changes</strong> before generating new ?'),
      '#default_value' => $this->getSetting('kill'),
    ];
    $form['num'] = [
      '#type' => 'number',
      '#title' => $this->t('How many changes would you like to generate?'),
      '#default_value' => $this->getSetting('num'),
      '#required' => TRUE,
      '#min' => 0,
    ];
    $form['max_actions'] = [
      '#type' => 'number',
      '#title' => $this->t('How many actions per change would you like to generate?'),
      '#default_value' => $this->getSetting('max_actions'),
      '#required' => TRUE,
      '#min' => 0,
    ];

    $options = [1 => $this->t('Now')];
    foreach ([3600, 86400, 604800, 2592000, 31536000] as $interval) {
      $options[$interval] = $this->dateFormatter->formatInterval($interval, 1) . ' ' . $this->t('ago');
    }
    $form['time_range'] = [
      '#type' => 'select',
      '#title' => $this->t('How far back in time should the changes be dated?'),
      '#description' => $this->t('Change creation dates will be distributed randomly from the current time, back to the selected time.'),
      '#options' => $options,
      '#default_value' => 604800,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  protected function generateElements(array $values): void {
    /** @var non-empty-array<string> $actions */
    $actions = [
      RoleChangeLog::PERMISSION_ADDED,
      RoleChangeLog::PERMISSION_REMOVED,
    ];
    $roles = $this->getRoleStorage()->loadMultiple();
    $options = [];
    foreach ($roles as $role) {
      $options[] = $role->id();
    }
    $permissions = $this->permissionHandler->getPermissions();
    if (!empty($values['kill'])) {
      $this->contentKill();
    }

    $query = $this->database->select('users', 'u')
      ->fields('u', ['uid'])
      ->range(0, 50)
      ->orderRandom();
    $result = $query->execute();
    $authors = is_null($result) ? [] : $result->fetchCol();

    if ($authors === [] || $options === [] || $permissions === []) {
      throw new \Exception('Cannot generate content: no users, roles, or permissions available.');
    }

    for ($i = 1; $i <= $values['num']; $i++) {
      $uid = $authors[array_rand($authors)];
      $role = $options[array_rand($options)];

      $change_values = [
        'uid' => $uid,
        'role' => $role,
        'timestamp' => $this->time->getRequestTime() - mt_rand(0, intval($values['time_range'])),
      ];
      for ($j = 0; $j < rand(1, intval($values['max_actions'])); $j++) {
        $change_values['actions'][] = [
          'action' => $actions[array_rand($actions)],
          'permission' => array_rand($permissions),
        ];
      }

      $change = $this->getRoleChangeLogStorage()->create($change_values);
      $change->save();
    }
  }

  /**
   * Deletes all changes.
   */
  protected function contentKill(): void {
    $storage = $this->getRoleChangeLogStorage();
    $ids = $storage->getQuery()
      ->accessCheck(FALSE)
      ->execute();

    if (!empty($ids)) {
      $changes = $storage->loadMultiple($ids);
      $storage->delete($changes);
      $this->setMessage($this->t('Deleted %count changes.', ['%count' => count($ids)]));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateDrushParams(array $args, array $options = []): array {
    return [];
  }

  /**
   * Retrieves the role storage.
   *
   * @return \Drupal\user\RoleStorageInterface
   *   The role storage.
   */
  protected function getRoleStorage(): RoleStorageInterface {
    return $this->entityTypeManager->getStorage('user_role');
  }

  /**
   * Retrieves the role change log storage.
   *
   * @return \Drupal\Core\Entity\EntityStorageInterface
   *   The role change log storage.
   */
  protected function getRoleChangeLogStorage(): EntityStorageInterface {
    return $this->entityTypeManager->getStorage('role_change_log');
  }

}
