<?php

declare(strict_types=1);

namespace Drupal\Tests\permission_watchdog\Functional;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\permission_watchdog\Entity\RoleChangeLog;
use Drupal\permission_watchdog\Plugin\Field\FieldType\PermissionActionItem;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\media\Traits\MediaTypeCreationTrait;
use Drupal\Tests\taxonomy\Traits\TaxonomyTestTrait;
use Drupal\user\UserInterface;

/**
 * Test capturing changes to the role permissions in UI.
 *
 * @group permission_watchdog
 */
class PermissionWatchdogTest extends BrowserTestBase {

  use MediaTypeCreationTrait;
  use TaxonomyTestTrait;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['permission_watchdog', 'node', 'media', 'taxonomy'];

  /**
   * Administrative user.
   */
  protected UserInterface $adminUser;

  /**
   * Change log storage.
   */
  protected EntityStorageInterface $storage;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->storage = $this->container->get('entity_type.manager')->getStorage('role_change_log');
    $changes = $this->storage->loadMultiple();
    self::assertEquals(0, \count($changes), 'No changes logged');
    $user = $this->drupalCreateUser(['administer permissions']);
    self::assertNotFalse($user, 'User creation failed');
    $this->adminUser = $user;
    $this->drupalLogin($this->adminUser);
    $this->drupalCreateRole([], 'content_editor');
  }

  /**
   * Test main form.
   */
  public function testUserPermissionsForm(): void {
    $this->drupalGet('admin/people/permissions');
    $change = [
      'authenticated[change own username]' => 1,
    ];
    $this->submitForm($change, 'Save permissions');
    $changes = $this->storage->loadByProperties(
      [
        'uid' => $this->adminUser->id(),
        'role' => 'authenticated',
      ]);
    self::assertEquals(1, \count($changes), 'One change logged');
    $change = array_shift($changes);
    self::assertInstanceOf(RoleChangeLog::class, $change);
    $actionField = $change->actions[0];
    self::assertInstanceOf(PermissionActionItem::class, $actionField);
    self::assertEquals(RoleChangeLog::PERMISSION_ADDED, $actionField->action, 'Correct action logged');
    self::assertEquals('change own username', $actionField->permission, 'Correct permission logged');
  }

  /**
   * Test module specific form.
   */
  public function testUserPermissionsModuleSpecificForm(): void {
    $this->drupalGet('admin/people/permissions/module/node');
    $change = [
      'content_editor[bypass node access]' => 1,
      'anonymous[access content]' => 0,
    ];
    $this->submitForm($change, 'Save permissions');
    $changes = $this->storage->loadMultiple();
    self::assertEquals(2, \count($changes), 'Two changes logged');
    foreach ($changes as $changeEntity) {
      /** @var \Drupal\permission_watchdog\Entity\RoleChangeLog $changeEntity */
      self::assertEquals($this->adminUser->id(), $changeEntity->uid->entity->id(), 'User ID match');
      $roleEntity = $changeEntity->role->entity;
      $expectedRole = $roleEntity->id() === 'content_editor' ? 'content_editor' : 'anonymous';
      self::assertEquals($expectedRole, $roleEntity->id(), 'Role match');
      $actionField = $changeEntity->actions[0];
      self::assertInstanceOf(PermissionActionItem::class, $actionField);
      $expectedAction = $actionField->action === RoleChangeLog::PERMISSION_ADDED ? RoleChangeLog::PERMISSION_ADDED : RoleChangeLog::PERMISSION_REMOVED;
      self::assertEquals($expectedAction, $actionField->action, 'Action match');
      $expectedPermission = $actionField->permission === 'bypass node access' ? 'bypass node access' : 'access content';
      self::assertEquals($expectedPermission, $actionField->permission, 'Permission match');
    }
  }

  /**
   * Test role specific form.
   */
  public function testUserPermissionsRoleSpecificForm(): void {
    $this->drupalGet('admin/people/permissions/content_editor');
    $change = [
      'content_editor[administer filters]' => 1,
    ];
    $this->submitForm($change, 'Save permissions');
    $changes = $this->storage->loadByProperties(
      [
        'uid' => $this->adminUser->id(),
        'role' => 'content_editor',
      ]);
    self::assertEquals(1, \count($changes), 'One change logged');
    $change = array_shift($changes);
    self::assertInstanceOf(RoleChangeLog::class, $change);
    $actionField = $change->actions[0];
    self::assertInstanceOf(PermissionActionItem::class, $actionField);
    self::assertEquals(RoleChangeLog::PERMISSION_ADDED, $actionField->action, 'Correct action logged');
    self::assertEquals('administer filters', $actionField->permission, 'Correct permission logged');
  }

  /**
   * Test node entity bundle form.
   *
   * Available on entity.node_type.entity_permissions_form route.
   */
  public function testEntityPermissionsFormNode(): void {
    $this->drupalCreateContentType(['type' => 'page']);
    $this->drupalGet('admin/structure/types/manage/page/permissions');
    $change = [
      'authenticated[delete any page content]' => 1,
    ];
    $this->submitForm($change, 'Save permissions');
    $changes = $this->storage->loadByProperties(
      [
        'uid' => $this->adminUser->id(),
        'role' => 'authenticated',
      ]);
    self::assertEquals(1, \count($changes), 'One change logged');
    $change = array_shift($changes);
    self::assertInstanceOf(RoleChangeLog::class, $change);
    $actionField = $change->actions[0];
    self::assertInstanceOf(PermissionActionItem::class, $actionField);
    self::assertEquals(RoleChangeLog::PERMISSION_ADDED, $actionField->action, 'Correct action logged');
    self::assertEquals('delete any page content', $actionField->permission, 'Correct permission logged');
  }

  /**
   * Test taxonomy term entity bundle form.
   *
   * Available on entity.taxonomy_vocabulary.entity_permissions_form route.
   */
  public function testEntityPermissionsFormTerm(): void {
    $this->createVocabulary(['vid' => 'tags']);
    $this->drupalGet('admin/structure/taxonomy/manage/tags/overview/permissions');
    $change = [
      'content_editor[delete terms in tags]' => 1,
    ];
    $this->submitForm($change, 'Save permissions');
    $changes = $this->storage->loadByProperties(
      [
        'uid' => $this->adminUser->id(),
        'role' => 'content_editor',
      ]);
    self::assertEquals(1, \count($changes), 'One change logged');
    $change = array_shift($changes);
    self::assertInstanceOf(RoleChangeLog::class, $change);
    $actionField = $change->actions[0];
    self::assertInstanceOf(PermissionActionItem::class, $actionField);
    self::assertEquals(RoleChangeLog::PERMISSION_ADDED, $actionField->action, 'Correct action logged');
    self::assertEquals('delete terms in tags', $actionField->permission, 'Correct permission logged');
  }

  /**
   * Test media entity bundle form.
   *
   * Available on entity.media_type.entity_permissions_form route.
   */
  public function testEntityPermissionsFormMedia(): void {
    $this->createMediaType('file', ['id' => 'document']);
    // Test media "document" form.
    $this->drupalGet('admin/structure/media/manage/document/permissions');
    $change = [
      'content_editor[delete any document media]' => 1,
    ];
    $this->submitForm($change, 'Save permissions');
    $changes = $this->storage->loadByProperties(
      [
        'uid' => $this->adminUser->id(),
        'role' => 'content_editor',
      ]);
    self::assertEquals(1, \count($changes), 'One change logged');
    $change = array_shift($changes);
    self::assertInstanceOf(RoleChangeLog::class, $change);
    $actionField = $change->actions[0];
    self::assertInstanceOf(PermissionActionItem::class, $actionField);
    self::assertEquals(RoleChangeLog::PERMISSION_ADDED, $actionField->action, 'Correct action logged');
    self::assertEquals('delete any document media', $actionField->permission, 'Correct permission logged');
  }

}
