<?php

namespace Drupal\Tests\permissionspolicy\Unit;

use Drupal\permissionspolicy\PermissionsPolicy;
use Drupal\Tests\UnitTestCase;

/**
 * Test PermissionsPolicy behavior.
 *
 * @coversDefaultClass \Drupal\permissionspolicy\PermissionsPolicy
 * @group permissionspolicy
 */
class PermissionsPolicyTest extends UnitTestCase {

  /**
   * Test that invalid feature names cause an exception.
   *
   * @covers ::setFeature
   * @covers ::isValidFeatureName
   * @covers ::validateFeatureName
   */
  public function testSetInvalidPolicy() {
    $policy = new PermissionsPolicy();

    $this->expectException(\InvalidArgumentException::class);
    $policy->setFeature('foo', PermissionsPolicy::ORIGIN_SELF);
  }

  /**
   * Test that invalid feature names cause an exception.
   *
   * @covers ::appendFeature
   * @covers ::isValidFeatureName
   * @covers ::validateFeatureName
   */
  public function testAppendInvalidPolicy() {
    $policy = new PermissionsPolicy();

    $this->expectException(\InvalidArgumentException::class);
    $policy->appendFeature('foo', PermissionsPolicy::ORIGIN_SELF);
  }

  /**
   * Test setting a single value to a feature.
   *
   * @covers ::setFeature
   * @covers ::hasFeature
   * @covers ::getFeature
   * @covers ::isValidFeatureName
   * @covers ::validateFeatureName
   * @covers ::getHeaderValue
   */
  public function testSetSingle() {
    $policy = new PermissionsPolicy();

    $policy->setFeature('geolocation', PermissionsPolicy::ORIGIN_SELF);

    $this->assertTrue($policy->hasFeature('geolocation'));
    $this->assertEquals(
      $policy->getFeature('geolocation'),
      ['self']
    );
    $this->assertEquals(
      "geolocation=self",
      $policy->getHeaderValue()
    );
  }

  /**
   * Test appending a single value to an uninitialized feature.
   *
   * @covers ::appendFeature
   * @covers ::hasFeature
   * @covers ::getFeature
   * @covers ::isValidFeatureName
   * @covers ::validateFeatureName
   * @covers ::getHeaderValue
   */
  public function testAppendSingle() {
    $policy = new PermissionsPolicy();

    $policy->appendFeature('geolocation', PermissionsPolicy::ORIGIN_SELF);

    $this->assertTrue($policy->hasFeature('geolocation'));
    $this->assertEquals(
      ['self'],
      $policy->getFeature('geolocation')
    );
    $this->assertEquals(
      "geolocation=self",
      $policy->getHeaderValue()
    );
  }

  /**
   * Test that a feature is overridden when set with a new value.
   *
   * @covers ::setFeature
   * @covers ::isValidFeatureName
   * @covers ::getHeaderValue
   */
  public function testSetMultiple() {
    $policy = new PermissionsPolicy();

    $policy->setFeature('geolocation', PermissionsPolicy::ORIGIN_ANY);
    $policy->setFeature('geolocation', [
      PermissionsPolicy::ORIGIN_SELF,
      'one.example.com',
    ]);

    $this->assertEquals(
      'geolocation=(self "one.example.com")',
      $policy->getHeaderValue()
    );
  }

  /**
   * Test that appending to a feature extends the existing value.
   *
   * @covers ::appendFeature
   * @covers ::isValidFeatureName
   * @covers ::getHeaderValue
   */
  public function testAppendMultiple() {
    $policy = new PermissionsPolicy();

    $policy->appendFeature('geolocation', PermissionsPolicy::ORIGIN_SELF);
    $policy->appendFeature('camera', [
      PermissionsPolicy::ORIGIN_SELF,
      'two.example.com',
    ]);
    $policy->appendFeature('geolocation', 'one.example.com');

    $this->assertEquals(
      'camera=(self "two.example.com"), geolocation=(self "one.example.com")',
      $policy->getHeaderValue()
    );
  }

  /**
   * Setting an empty value for a feature.
   *
   * @covers ::setFeature
   * @covers ::isValidFeatureName
   * @covers ::getHeaderValue
   */
  public function testSetEmpty() {
    $policy = new PermissionsPolicy();
    $policy->setFeature('geolocation', PermissionsPolicy::ORIGIN_SELF);
    $policy->setFeature('camera', [PermissionsPolicy::ORIGIN_SELF]);
    $policy->setFeature('camera', []);

    $this->assertEquals(
      "camera=(), geolocation=self",
      $policy->getHeaderValue()
    );

    $policy = new PermissionsPolicy();
    $policy->setFeature('geolocation', PermissionsPolicy::ORIGIN_SELF);
    $policy->setFeature('camera', [PermissionsPolicy::ORIGIN_SELF]);
    $policy->setFeature('camera', '');

    $this->assertEquals(
      "camera=(), geolocation=self",
      $policy->getHeaderValue()
    );
  }

  /**
   * Appending an empty value shouldn't change the feature.
   *
   * @covers ::appendFeature
   * @covers ::isValidFeatureName
   * @covers ::getHeaderValue
   */
  public function testAppendEmpty() {
    $policy = new PermissionsPolicy();

    $policy->appendFeature('geolocation', PermissionsPolicy::ORIGIN_SELF);
    $this->assertEquals(
      'geolocation=self',
      $policy->getHeaderValue()
    );

    $policy->appendFeature('geolocation', '');
    $policy->appendFeature('camera', []);
    $this->assertEquals(
      'camera=(), geolocation=self',
      $policy->getHeaderValue()
    );
  }

  /**
   * Test that source values are not repeated in the header.
   *
   * @covers ::setFeature
   * @covers ::appendFeature
   * @covers ::isValidFeatureName
   * @covers ::getHeaderValue
   */
  public function testDuplicate() {
    $policy = new PermissionsPolicy();

    // Provide identical sources in an array.
    $policy->setFeature('geolocation', [
      PermissionsPolicy::ORIGIN_SELF,
      PermissionsPolicy::ORIGIN_SELF,
    ]);
    // Provide identical sources in a string.
    $policy->setFeature('camera', 'one.example.com one.example.com');

    // Provide identical sources through both set and append.
    $policy->setFeature('microphone', ['two.example.com', 'two.example.com']);
    $policy->appendFeature('microphone', ['two.example.com', 'two.example.com']);

    $this->assertEquals(
      'camera="one.example.com", geolocation=self, microphone="two.example.com"',
      $policy->getHeaderValue()
    );
  }

  /**
   * Test that removed features are not output in the header.
   *
   * @covers ::removeFeature
   * @covers ::isValidFeatureName
   * @covers ::getHeaderValue
   */
  public function testRemove() {
    $policy = new PermissionsPolicy();

    $policy->setFeature('geolocation', [PermissionsPolicy::ORIGIN_SELF]);
    $policy->setFeature('camera', 'example.com');

    $policy->removeFeature('camera');

    $this->assertEquals(
      'geolocation=self',
      $policy->getHeaderValue()
    );
  }

  /**
   * Test that removing an invalid feature name causes an exception.
   *
   * @covers ::removeFeature
   * @covers ::isValidFeatureName
   * @covers ::validateFeatureName
   */
  public function testRemoveInvalid() {
    $policy = new PermissionsPolicy();

    $this->expectException(\InvalidArgumentException::class);
    $policy->removeFeature('foo');
  }

  /**
   * Test that invalid feature values cause an exception.
   *
   * @covers ::appendFeature
   */
  public function testInvalidValue() {
    $policy = new PermissionsPolicy();

    $this->expectException(\InvalidArgumentException::class);
    $policy->appendFeature('geolocation', 12);
  }

  /**
   * Test reducing the allow list when 'none' is included.
   *
   * @covers ::reduceAllowList
   */
  public function testReduceAllowListWithNone() {
    $policy = new PermissionsPolicy();

    $policy->setFeature('geolocation', [
      PermissionsPolicy::ORIGIN_NONE,
      'example.com',
    ]);
    $this->assertEquals(
      'geolocation=()',
      $policy->getHeaderValue()
    );
  }

  /**
   * Test reducing allow list when any host allowed.
   *
   * @covers ::reduceAllowList
   */
  public function testReduceAllowListAny() {
    $policy = new PermissionsPolicy();

    $policy->setFeature('geolocation', [
      PermissionsPolicy::ORIGIN_ANY,
      'example.com',
      'https://example.com',
      'http:',
      'https:',
    ]);
    $this->assertEquals(
      "geolocation=*",
      $policy->getHeaderValue()
    );
  }

  /**
   * Test reducing the source list when 'http:' is included.
   *
   * @covers ::reduceAllowList
   */
  public function testReduceAllowListWithHttp() {
    $policy = new PermissionsPolicy();

    $policy->setFeature('geolocation', [
      'http:',
      // Hosts without protocol should be kept.
      // (e.g. this would allow ftp://example.com)
      'example.com',
      // HTTP hosts should be removed.
      'http://example.org',
      'https://example.net',
    ]);

    $this->assertEquals(
      'geolocation=("http:" "example.com")',
      $policy->getHeaderValue()
    );
  }

  /**
   * Test reducing the source list when 'https:' is included.
   *
   * @covers ::reduceAllowList
   */
  public function testReduceAllowListWithHttps() {
    $policy = new PermissionsPolicy();

    $policy->setFeature('geolocation', [
      'https:',
      // Non-secure hosts should be kept.
      'example.com',
      'http://example.org',
      // Secure Hosts should be removed.
      'https://example.net',
    ]);

    $this->assertEquals(
      'geolocation=("https:" "example.com" "http://example.org")',
      $policy->getHeaderValue()
    );
  }

  /**
   * @covers ::__toString
   */
  public function testToString() {
    $policy = new PermissionsPolicy();

    $policy->setFeature('geolocation', PermissionsPolicy::ORIGIN_SELF);
    $policy->setFeature('camera', [
      PermissionsPolicy::ORIGIN_SELF,
      'example.com',
    ]);

    $this->assertEquals(
      'Permissions-Policy: camera=(self "example.com"), geolocation=self',
      $policy->__toString()
    );
  }

}
