<?php

namespace Drupal\phonepay_payment\Methods;

use Drupal\phonepay_payment\PhonePeBaseClass;

/**
 * Standard checkout of the phonepe payment process.
 */
class StandardCheckout extends PhonePeBaseClass {

  /**
   * Transaction.
   *
   * @var \Drupal\phonepay_payment\PhonePeBaseClass
   */
  private $transaction;

  /**
   * Transaction response.
   *
   * @var \Drupal\phonepay_payment\PhonePeBaseClass
   */
  private $transactionResponse;

  const TRANSACTION_STATUS_SUCCESS = "PAYMENT_SUCCESS";

  /**
   * Create new transaction http://developer.phonepe.com/v1/reference/pay-api-1.
   *
   * @param int $amountInPaisa
   *   Payment amount in paisa.
   * @param string $mobile
   *   Payment mobile no.
   * @param string $merchentTransactionID
   *   Payment merchent.
   * @param string $redirectUrl
   *   Payment URL.
   * @param string $callbackUrl
   *   Payment call by URL.
   * @param string $redirectMode
   *   REDIRECT or POST.
   *
   * @return StandardCheckout
   *   Transaction instance.
   */
  public function createTransaction($amountInPaisa, $mobile, $merchentTransactionID, $redirectUrl = NULL, $callbackUrl = NULL, $redirectMode = "REDIRECT") {
    // Check if the amount is valid.
    if (!is_numeric($amountInPaisa)) {
      throw new \Exception("Amount should be numeric");
    }

    // Check if the mobile number is valid.
    if (!is_numeric($mobile)) {
      throw new \Exception("Mobile number should be numeric");
    }

    // According to the api, the redirect mode should be REDIRECT or POST.
    if ($redirectMode != "REDIRECT" && $redirectMode != "POST") {
      throw new \Exception("Invalid redirect mode, should be REDIRECT or POST");
    }

    if (!$redirectUrl) {
      $redirectUrl = $this->config['REDIRECT_URL'];
    }
    if (!$callbackUrl) {
      $callbackUrl = $this->config['CALLBACK_URL'];
    }

    $payload = [
      "merchantId" => $this->config['MERCHANT_ID'],
      "merchantTransactionId" => $merchentTransactionID,
      "merchantUserId" => $this->config['MERCHANT_USER_ID'],
      "amount" => $amountInPaisa,
      "redirectUrl" => $redirectUrl,
      "redirectMode" => $redirectMode,
      "callbackUrl" => $callbackUrl,
      "mobileNumber" => $mobile,
      "paymentInstrument" => [
        "type" => "PAY_PAGE",
      ],
    ];

    $response = $this->executeRequest("POST", "/pg/v1/pay", $payload);
    $this->transaction = $response;
    return $this;
  }

  /**
   * Returns full transaction response from server when transaction is created.
   *
   * @return array
   *   Transaction instance.
   */
  public function getCreatedTransaction() {
    if (!$this->transaction) {
      throw new \Exception("Transaction not created");
    }

    return $this->transaction;
  }

  /**
   * Returns transaction URL from server if the transaction is created.
   *
   * @return string
   *   Transaction URL.
   */
  public function getTransactionUrl() {
    if (!$this->transaction) {
      throw new \Exception("Transaction not created");
    }

    return $this->transaction['data']['instrumentResponse']['redirectInfo']['url'];
  }

  /**
   * Verifies the response from the server using phonepe's algorithm.
   *
   * @param string $payload
   *   Payment payload.
   * @param string $xVerify
   *   Payment verification.
   *
   * @return bool
   *   Returns if Payment is verified.
   */
  public function verifyResponse($payload, $xVerify) {
    return parent::verifyResponse($payload, $xVerify);
  }

  /**
   * Returns decoded response if verified the response from the server.
   *
   * @param string $payload
   *   Payment payload.
   * @param string $xVerify
   *   Payment verification.
   *
   * @return array|bool
   *   Returns verified decoded response.
   */
  public function verifyAndGetResponse($payload, $xVerify) {
    if ($this->verifyResponse($payload, $xVerify)) {
      return json_decode(base64_decode($payload), TRUE);
    }
    return FALSE;
  }

  /**
   * Verifies the response from the server directly using the global variables.
   *
   * @return array
   *   Returns the verified payment response.
   */
  public function verifyPaymentFromCallback() {
    // Check if the request method is not POST.
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
      throw new \Exception("Invalid request method");
    }

    // Check if X-VERIFY is set.
    if (!isset($_SERVER['HTTP_X_VERIFY'])) {
      throw new \Exception("X-VERIFY not set");
    }
    $xVerify = $_SERVER['HTTP_X_VERIFY'];

    // Get Content.
    $content = trim(file_get_contents("php://input"));
    $payload = json_decode($content, TRUE);
    $base64Payload = $payload['response'];

    // Validate X-VERIFY.
    if (!$this->verifyResponse($base64Payload, $xVerify)) {
      throw new \Exception("Invalid X-VERIFY");
    }

    // Decode Payload.
    $payload = json_decode(base64_decode($base64Payload), TRUE);

    // Store transaction response.
    $this->transactionResponse = $payload;

    return $payload;
  }

  /**
   * Return decoded response, verifed from the server directly by global var.
   *
   * @return array
   *   Return verified decoded response
   */
  public function getTransactionResponse() {
    if (!$this->transactionResponse) {
      $this->verifyPaymentFromCallback();
    }

    return $this->transactionResponse;
  }

  /**
   * Return transaction Id, if verified from the server directly by global var.
   *
   * @return string
   *   Return transaction id.
   */
  public function getTransactionId() {
    if (!$this->transactionResponse) {
      $this->verifyPaymentFromCallback();
    }

    return $this->transactionResponse['data']['transactionId'];
  }

  /**
   * Return transaciton status, verifies response by the server by global var.
   *
   * @return string
   *   Return transaction status.
   */
  public function getTransactionStatus() {
    if (!$this->transactionResponse) {
      $this->verifyPaymentFromCallback();
    }

    return $this->transactionResponse['code'];
  }

  /**
   * Return True if transaction is done and verified by server by global var.
   *
   * @return bool
   *   Return True if Transaction is success else false.
   */
  public function isTransactionSuccess() {
    if ($this->getTransactionStatus() == self::TRANSACTION_STATUS_SUCCESS) {
      return TRUE;
    }
    return FALSE;
  }

  /**
   * Return transaction amount, verified by server directly using global var.
   *
   * @return int
   *   Return transaction amount.
   */
  public function getTransactionAmount() {
    if (!$this->transactionResponse) {
      $this->verifyPaymentFromCallback();
    }

    return $this->transactionResponse['data']['amount'];
  }

  /**
   * Return transaction instrument, verified respone by server by global var.
   *
   * @return array
   *   Return transaction instrument.
   */
  public function getTransactionPaymentInstrument() {
    if (!$this->transactionResponse) {
      $this->verifyPaymentFromCallback();
    }

    return $this->transactionResponse['data']['paymentInstrument'];
  }

  /**
   * Return transaction type, verified response by server by global variables.
   *
   * @return string
   *   Return transaction instrument type.
   */
  public function getTransactionPaymentInstrumentType() {
    if (!$this->transactionResponse) {
      $this->verifyPaymentFromCallback();
    }

    return $this->transactionResponse['data']['paymentInstrument']['type'];
  }

  /**
   * Get the transaction response from the server using the transaction ID.
   *
   * @param string $transactionId
   *   Transaction Id.
   *
   * @return array
   *   Return Transaction response by transaction id.
   */
  public function getTransactionResponseByTransactionId($transactionId) {
    $response = $this->executeRequest(
          "GET", "/pg/v1/status/" . $this->config['MERCHANT_ID'] . "/" . $transactionId, NULL, [
            'X-MERCHANT-ID' => $this->config['MERCHANT_ID'],
          ]
      );
    return $response;
  }

  /**
   * Get the transaction status from the server using the transaction ID.
   *
   * @param string $transactionId
   *   Transaction Id.
   *
   * @return string
   *   Return Transaction status from server.
   */
  public function getTransactionStatusByTransactionId($transactionId) {
    $response = $this->getTransactionResponseByTransactionId($transactionId);
    return $response['code'];
  }

  /**
   * Get True if transactino is success from the server using transaction Id.
   *
   * @param string $transactionId
   *   Transaction Id.
   *
   * @return bool
   *   Returns True if transaction is succcess, else false.
   */
  public function isTransactionSuccessByTransactionId($transactionId) {
    if ($this->getTransactionStatusByTransactionId($transactionId) == self::TRANSACTION_STATUS_SUCCESS) {
      return TRUE;
    }
    return FALSE;
  }

  /**
   * Get the transaction amount from the server using the transaction ID.
   *
   * @param string $transactionId
   *   Transaction Id.
   *
   * @return int
   *   Return transactino amount.
   */
  public function getTransactionAmountByTransactionId($transactionId) {
    $response = $this->getTransactionResponseByTransactionId($transactionId);
    return $response['data']['amount'];
  }

  /**
   * Get transaction payment instrument from server using transaction id.
   *
   * @param string $transactionId
   *   Transaction Id.
   *
   * @return array
   *   Return transaction payment instrument.
   */
  public function getTransactionPaymentInstrumentByTransactionId($transactionId) {
    $response = $this->getTransactionResponseByTransactionId($transactionId);
    return $response['data']['paymentInstrument'];
  }

  /**
   * Get transaction payment instrument type from server using transaction id.
   *
   * @param string $transactionId
   *   Transaction Id.
   *
   * @return string
   *   Return transaction payment instrucment type.
   */
  public function getTransactionPaymentInstrumentTypeByTransactionId($transactionId) {
    $response = $this->getTransactionResponseByTransactionId($transactionId);
    return $response['data']['paymentInstrument']['type'];
  }

}
